/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.hadoop.repackaged.gcs.com.google.cloud.hadoop.gcsio;

import com.google.cloud.hadoop.repackaged.gcs.com.google.api.client.googleapis.batch.BatchRequest;
import com.google.cloud.hadoop.repackaged.gcs.com.google.api.client.googleapis.batch.json.JsonBatchCallback;
import com.google.cloud.hadoop.repackaged.gcs.com.google.api.client.googleapis.json.GoogleJsonResponseException;
import com.google.cloud.hadoop.repackaged.gcs.com.google.api.client.http.HttpRequestInitializer;
import com.google.cloud.hadoop.repackaged.gcs.com.google.api.services.storage.Storage;
import com.google.cloud.hadoop.repackaged.gcs.com.google.api.services.storage.StorageRequest;
import com.google.cloud.hadoop.repackaged.gcs.com.google.cloud.hadoop.util.ApiErrorExtractor;
import com.google.cloud.hadoop.repackaged.gcs.com.google.common.base.Preconditions;
import com.google.cloud.hadoop.repackaged.gcs.com.google.common.flogger.GoogleLogger;
import com.google.cloud.hadoop.repackaged.gcs.com.google.common.util.concurrent.MoreExecutors;
import com.google.cloud.hadoop.repackaged.gcs.com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

public class BatchHelper {
    private static final GoogleLogger logger = GoogleLogger.forEnclosingClass();
    private static final ThreadFactory THREAD_FACTORY = new ThreadFactoryBuilder().setNameFormat("gcsfs-batch-helper-%d").setDaemon(true).build();
    private final Queue<QueueRequestCallback> pendingRequests = new ConcurrentLinkedQueue<QueueRequestCallback>();
    private final ExecutorService requestsExecutor;
    private final Queue<Future<Void>> responseFutures = new ConcurrentLinkedQueue<Future<Void>>();
    private final HttpRequestInitializer requestInitializer;
    private final Storage gcs;
    private final long maxRequestsPerBatch;
    private final Lock flushLock = new ReentrantLock();

    private BatchHelper(HttpRequestInitializer requestInitializer, Storage gcs, long maxRequestsPerBatch, int numThreads) {
        this.requestInitializer = requestInitializer;
        this.gcs = gcs;
        this.requestsExecutor = numThreads == 0 ? MoreExecutors.newDirectExecutorService() : BatchHelper.newRequestsExecutor(numThreads);
        this.maxRequestsPerBatch = maxRequestsPerBatch;
    }

    private static ExecutorService newRequestsExecutor(int numThreads) {
        ThreadPoolExecutor requestsExecutor = new ThreadPoolExecutor(numThreads, numThreads, 5L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(numThreads * 20), THREAD_FACTORY);
        requestsExecutor.allowCoreThreadTimeOut(true);
        requestsExecutor.setRejectedExecutionHandler(new ThreadPoolExecutor.CallerRunsPolicy());
        return requestsExecutor;
    }

    public <T> void queue(StorageRequest<T> req, JsonBatchCallback<T> callback) throws IOException {
        Preconditions.checkState(!this.requestsExecutor.isShutdown() && !this.requestsExecutor.isTerminated(), "requestsExecutor should not be terminated to queue batch requests");
        if (this.maxRequestsPerBatch == 1L) {
            this.responseFutures.add(this.requestsExecutor.submit(() -> {
                this.execute(req, callback);
                return null;
            }));
        } else {
            this.pendingRequests.add(batch -> req.queue(batch, callback));
            this.flushIfPossibleAndRequired();
        }
    }

    public <T> void execute(StorageRequest<T> req, JsonBatchCallback<T> callback) throws IOException {
        try {
            Object result = req.execute();
            callback.onSuccess(result, req.getLastResponseHeaders());
        }
        catch (IOException e) {
            GoogleJsonResponseException je = ApiErrorExtractor.getJsonResponseExceptionOrNull(e);
            if (je == null) {
                throw e;
            }
            callback.onFailure(je.getDetails(), je.getHeaders());
        }
    }

    private void flushIfPossibleAndRequired() throws IOException {
        if ((long)this.pendingRequests.size() >= this.maxRequestsPerBatch) {
            this.flushIfPossible(false);
        }
    }

    private void flushIfPossible(boolean flushAll) throws IOException {
        if (flushAll) {
            this.flushLock.lock();
        } else if (this.pendingRequests.isEmpty() || !this.flushLock.tryLock()) {
            return;
        }
        try {
            do {
                this.flushPendingRequests();
                if (!flushAll) continue;
                this.awaitRequestsCompletion();
            } while (flushAll && (!this.pendingRequests.isEmpty() || !this.responseFutures.isEmpty()));
        }
        finally {
            this.flushLock.unlock();
        }
    }

    private void flushPendingRequests() throws IOException {
        if (this.pendingRequests.isEmpty()) {
            return;
        }
        BatchRequest batch = this.gcs.batch(this.requestInitializer);
        while ((long)batch.size() < this.maxRequestsPerBatch && !this.pendingRequests.isEmpty()) {
            this.pendingRequests.remove().enqueue(batch);
        }
        this.responseFutures.add(this.requestsExecutor.submit(() -> {
            batch.execute();
            return null;
        }));
    }

    public void flush() throws IOException {
        try {
            this.flushIfPossible(true);
            Preconditions.checkState(this.pendingRequests.isEmpty(), "pendingRequests should be empty after flush");
            Preconditions.checkState(this.responseFutures.isEmpty(), "responseFutures should be empty after flush");
        }
        finally {
            this.requestsExecutor.shutdown();
            try {
                if (!this.requestsExecutor.awaitTermination(1L, TimeUnit.SECONDS)) {
                    ((GoogleLogger.Api)logger.atWarning()).log("Forcibly shutting down batch helper thread pool.");
                    this.requestsExecutor.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                ((GoogleLogger.Api)((GoogleLogger.Api)logger.atFine()).withCause(e)).log("Failed to await termination: forcibly shutting down batch helper thread pool.");
                this.requestsExecutor.shutdownNow();
            }
        }
    }

    public boolean isEmpty() {
        return this.pendingRequests.isEmpty();
    }

    private void awaitRequestsCompletion() throws IOException {
        while (!this.responseFutures.isEmpty() && (long)this.pendingRequests.size() < this.maxRequestsPerBatch) {
            try {
                this.responseFutures.remove().get();
            }
            catch (InterruptedException | ExecutionException e) {
                if (e.getCause() instanceof IOException) {
                    throw (IOException)e.getCause();
                }
                throw new RuntimeException("Failed to execute batch", e);
            }
        }
    }

    protected static interface QueueRequestCallback {
        public void enqueue(BatchRequest var1) throws IOException;
    }

    public static class Factory {
        public BatchHelper newBatchHelper(HttpRequestInitializer requestInitializer, Storage gcs, long maxRequestsPerBatch) {
            return new BatchHelper(requestInitializer, gcs, maxRequestsPerBatch, 0);
        }

        BatchHelper newBatchHelper(HttpRequestInitializer requestInitializer, Storage gcs, long maxRequestsPerBatch, long totalRequests, int maxThreads) {
            Preconditions.checkArgument(maxRequestsPerBatch > 0L, "maxRequestsPerBatch should be greater than 0");
            Preconditions.checkArgument(totalRequests > 0L, "totalRequests should be greater than 0");
            Preconditions.checkArgument(maxThreads >= 0, "maxThreads should be greater or equal to 0");
            if (totalRequests == 1L) {
                return new BatchHelper(requestInitializer, gcs, 1L, 0);
            }
            if (maxThreads == 0) {
                return new BatchHelper(requestInitializer, gcs, maxRequestsPerBatch, maxThreads);
            }
            long requestsPerBatch = (long)Math.ceil((double)totalRequests / (double)maxThreads);
            requestsPerBatch = Math.min(requestsPerBatch, maxRequestsPerBatch);
            int numThreads = Math.toIntExact((long)Math.ceil((double)totalRequests / (double)requestsPerBatch));
            numThreads = Math.min(numThreads, maxThreads);
            return new BatchHelper(requestInitializer, gcs, requestsPerBatch, numThreads);
        }
    }
}

