/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.server.storage;

import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.Files;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.server.RaftServerConfigKeys;
import org.apache.ratis.server.impl.RaftConfiguration;
import org.apache.ratis.server.impl.RaftServerConstants;
import org.apache.ratis.server.impl.ServerProtoUtils;
import org.apache.ratis.server.storage.MetaFile;
import org.apache.ratis.server.storage.RaftStorageDirectory;
import org.apache.ratis.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RaftStorage
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(RaftStorage.class);
    private final RaftStorageDirectory storageDir;
    private final RaftStorageDirectory.StorageState state;
    private final RaftServerConfigKeys.Log.CorruptionPolicy logCorruptionPolicy;
    private volatile MetaFile metaFile;

    public RaftStorage(File dir, RaftServerConstants.StartupOption option) throws IOException {
        this(dir, option, RaftServerConfigKeys.Log.CorruptionPolicy.getDefault());
    }

    public RaftStorage(File dir, RaftServerConstants.StartupOption option, RaftServerConfigKeys.Log.CorruptionPolicy logCorruptionPolicy) throws IOException {
        this.storageDir = new RaftStorageDirectory(dir);
        if (option == RaftServerConstants.StartupOption.FORMAT) {
            if (this.storageDir.analyzeStorage(false) == RaftStorageDirectory.StorageState.NON_EXISTENT) {
                throw new IOException("Cannot format " + this.storageDir);
            }
            this.storageDir.lock();
            this.format();
            this.state = this.storageDir.analyzeStorage(false);
            Preconditions.assertTrue(this.state == RaftStorageDirectory.StorageState.NORMAL);
        } else {
            this.state = this.analyzeAndRecoverStorage(true);
            if (this.state != RaftStorageDirectory.StorageState.NORMAL) {
                this.storageDir.unlock();
                throw new IOException("Cannot load " + this.storageDir + ". Its state: " + (Object)((Object)this.state));
            }
        }
        this.logCorruptionPolicy = logCorruptionPolicy;
    }

    RaftStorageDirectory.StorageState getState() {
        return this.state;
    }

    public RaftServerConfigKeys.Log.CorruptionPolicy getLogCorruptionPolicy() {
        return this.logCorruptionPolicy;
    }

    private void format() throws IOException {
        this.storageDir.clearDirectory();
        this.metaFile = this.writeMetaFile(0L, "");
        LOG.info("Storage directory " + this.storageDir.getRoot() + " has been successfully formatted.");
    }

    private MetaFile writeMetaFile(long term, String votedFor) throws IOException {
        MetaFile metaFile = new MetaFile(this.storageDir.getMetaFile());
        metaFile.set(term, votedFor);
        return metaFile;
    }

    private void cleanMetaTmpFile() throws IOException {
        Files.deleteIfExists(this.storageDir.getMetaTmpFile().toPath());
    }

    private RaftStorageDirectory.StorageState analyzeAndRecoverStorage(boolean toLock) throws IOException {
        RaftStorageDirectory.StorageState storageState = this.storageDir.analyzeStorage(toLock);
        if (storageState == RaftStorageDirectory.StorageState.NORMAL) {
            this.metaFile = new MetaFile(this.storageDir.getMetaFile());
            Preconditions.assertTrue(this.metaFile.exists(), () -> "Meta file " + this.metaFile + " does not exists.");
            this.metaFile.readFile();
            this.cleanMetaTmpFile();
            return RaftStorageDirectory.StorageState.NORMAL;
        }
        if (storageState == RaftStorageDirectory.StorageState.NOT_FORMATTED && this.storageDir.isCurrentEmpty()) {
            this.format();
            return RaftStorageDirectory.StorageState.NORMAL;
        }
        return storageState;
    }

    public RaftStorageDirectory getStorageDir() {
        return this.storageDir;
    }

    @Override
    public void close() throws IOException {
        this.storageDir.unlock();
    }

    public MetaFile getMetaFile() {
        return this.metaFile;
    }

    public void writeRaftConfiguration(RaftProtos.LogEntryProto conf) {
        File confFile = this.storageDir.getMetaConfFile();
        try (FileOutputStream fio = new FileOutputStream(confFile);){
            conf.writeTo(fio);
        }
        catch (Exception e) {
            LOG.error("Failed writing configuration to file:" + confFile, (Throwable)e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public RaftConfiguration readRaftConfiguration() {
        File confFile = this.storageDir.getMetaConfFile();
        try (FileInputStream fio = new FileInputStream(confFile);){
            RaftProtos.LogEntryProto confProto = ((RaftProtos.LogEntryProto.Builder)RaftProtos.LogEntryProto.newBuilder().mergeFrom(fio)).build();
            RaftConfiguration raftConfiguration = ServerProtoUtils.toRaftConfiguration(confProto);
            return raftConfiguration;
        }
        catch (Exception e) {
            LOG.error("Failed reading configuration from file:" + confFile, (Throwable)e);
            return null;
        }
    }

    public String toString() {
        return this.getClass().getSimpleName() + ":" + this.getStorageDir();
    }
}

