/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.server.raftlog.segmented;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.server.RaftServerConfigKeys;
import org.apache.ratis.server.impl.ServerProtoUtils;
import org.apache.ratis.server.metrics.RaftLogMetrics;
import org.apache.ratis.server.protocol.TermIndex;
import org.apache.ratis.server.raftlog.RaftLogIOException;
import org.apache.ratis.server.raftlog.segmented.SegmentedRaftLogFormat;
import org.apache.ratis.server.raftlog.segmented.SegmentedRaftLogInputStream;
import org.apache.ratis.server.storage.RaftStorage;
import org.apache.ratis.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.apache.ratis.thirdparty.com.google.common.cache.CacheLoader;
import org.apache.ratis.thirdparty.com.google.protobuf.CodedOutputStream;
import org.apache.ratis.util.FileUtils;
import org.apache.ratis.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LogSegment
implements Comparable<Long> {
    static final Logger LOG = LoggerFactory.getLogger(LogSegment.class);
    private volatile boolean isOpen;
    private long totalSize = SegmentedRaftLogFormat.getHeaderLength();
    private final long startIndex;
    private volatile long endIndex;
    private final RaftStorage storage;
    private RaftLogMetrics raftLogMetrics;
    private final LogEntryLoader cacheLoader = new LogEntryLoader(this.raftLogMetrics);
    private final AtomicInteger loadingTimes = new AtomicInteger();
    private final List<LogRecord> records = new ArrayList<LogRecord>();
    private final Map<TermIndex, RaftProtos.LogEntryProto> entryCache = new ConcurrentHashMap<TermIndex, RaftProtos.LogEntryProto>();
    private final Set<TermIndex> configEntries = new HashSet<TermIndex>();

    static long getEntrySize(RaftProtos.LogEntryProto entry) {
        int serialized = ServerProtoUtils.removeStateMachineData(entry).getSerializedSize();
        return serialized + CodedOutputStream.computeUInt32SizeNoTag(serialized) + 4;
    }

    static LogSegment newOpenSegment(RaftStorage storage, long start, RaftLogMetrics raftLogMetrics) {
        Preconditions.assertTrue(start >= 0L);
        return new LogSegment(storage, true, start, start - 1L, raftLogMetrics);
    }

    @VisibleForTesting
    static LogSegment newCloseSegment(RaftStorage storage, long start, long end, RaftLogMetrics raftLogMetrics) {
        Preconditions.assertTrue(start >= 0L && end >= start);
        return new LogSegment(storage, false, start, end, raftLogMetrics);
    }

    public static int readSegmentFile(File file, long start, long end, boolean isOpen, RaftServerConfigKeys.Log.CorruptionPolicy corruptionPolicy, RaftLogMetrics raftLogMetrics, Consumer<RaftProtos.LogEntryProto> entryConsumer) throws IOException {
        int count = 0;
        try (SegmentedRaftLogInputStream in = new SegmentedRaftLogInputStream(file, start, end, isOpen, raftLogMetrics);){
            RaftProtos.LogEntryProto next;
            RaftProtos.LogEntryProto prev = null;
            while ((next = in.nextEntry()) != null) {
                if (prev != null) {
                    Preconditions.assertTrue(next.getIndex() == prev.getIndex() + 1L, "gap between entry %s and entry %s", prev, next);
                }
                if (entryConsumer != null) {
                    entryConsumer.accept(next);
                }
                ++count;
                prev = next;
            }
        }
        catch (IOException ioe) {
            switch (corruptionPolicy) {
                case EXCEPTION: {
                    throw ioe;
                }
                case WARN_AND_RETURN: {
                    LOG.warn("Failed to read segment file {} (start={}, end={}, isOpen? {}): only {} entries read successfully", new Object[]{file, start, end, isOpen, count, ioe});
                    break;
                }
                default: {
                    throw new IllegalStateException("Unexpected enum value: " + (Object)((Object)corruptionPolicy) + ", class=" + RaftServerConfigKeys.Log.CorruptionPolicy.class);
                }
            }
        }
        return count;
    }

    static LogSegment loadSegment(RaftStorage storage, File file, long start, long end, boolean isOpen, boolean keepEntryInCache, Consumer<RaftProtos.LogEntryProto> logConsumer, RaftLogMetrics raftLogMetrics) throws IOException {
        boolean corrupted;
        LogSegment segment = isOpen ? LogSegment.newOpenSegment(storage, start, raftLogMetrics) : LogSegment.newCloseSegment(storage, start, end, raftLogMetrics);
        RaftServerConfigKeys.Log.CorruptionPolicy corruptionPolicy = RaftServerConfigKeys.Log.CorruptionPolicy.get(storage, RaftStorage::getLogCorruptionPolicy);
        int entryCount = LogSegment.readSegmentFile(file, start, end, isOpen, corruptionPolicy, raftLogMetrics, entry -> {
            segment.append(keepEntryInCache || isOpen, (RaftProtos.LogEntryProto)entry);
            if (logConsumer != null) {
                logConsumer.accept((RaftProtos.LogEntryProto)entry);
            }
        });
        LOG.info("Successfully read {} entries from segment file {}", (Object)entryCount, (Object)file);
        int expectedEntryCount = Math.toIntExact(end - start + 1L);
        boolean bl = corrupted = entryCount != expectedEntryCount;
        if (corrupted) {
            LOG.warn("Segment file is corrupted: expected to have {} entries but only {} entries read successfully", (Object)expectedEntryCount, (Object)entryCount);
        }
        if (entryCount == 0) {
            FileUtils.deleteFile(file);
            return null;
        }
        if (file.length() > segment.getTotalSize()) {
            FileUtils.truncateFile(file, segment.getTotalSize());
        }
        try {
            segment.assertSegment(start, entryCount, corrupted, end);
        }
        catch (Exception e) {
            throw new IllegalStateException("Failed to read segment file " + file, e);
        }
        return segment;
    }

    private void assertSegment(long expectedStart, int expectedEntryCount, boolean corrupted, long expectedEnd) {
        Preconditions.assertSame(expectedStart, this.getStartIndex(), "Segment start index");
        Preconditions.assertSame(expectedEntryCount, this.records.size(), "Number of records");
        long expectedLastIndex = expectedStart + (long)expectedEntryCount - 1L;
        Preconditions.assertSame(expectedLastIndex, this.getEndIndex(), "Segment end index");
        LogRecord last = this.getLastRecord();
        if (last != null) {
            Preconditions.assertSame(expectedLastIndex, last.getTermIndex().getIndex(), "Index at the last record");
            Preconditions.assertSame(expectedStart, this.records.get(0).getTermIndex().getIndex(), "Index at the first record");
        }
        if (!this.isOpen && !corrupted) {
            Preconditions.assertSame(expectedEnd, expectedLastIndex, "End/last Index");
        }
    }

    private File getSegmentFile() {
        return this.isOpen ? this.storage.getStorageDir().getOpenLogFile(this.startIndex) : this.storage.getStorageDir().getClosedLogFile(this.startIndex, this.endIndex);
    }

    private LogSegment(RaftStorage storage, boolean isOpen, long start, long end, RaftLogMetrics raftLogMetrics) {
        this.storage = storage;
        this.isOpen = isOpen;
        this.startIndex = start;
        this.endIndex = end;
        this.raftLogMetrics = raftLogMetrics;
    }

    long getStartIndex() {
        return this.startIndex;
    }

    long getEndIndex() {
        return this.endIndex;
    }

    boolean isOpen() {
        return this.isOpen;
    }

    int numOfEntries() {
        return Math.toIntExact(this.endIndex - this.startIndex + 1L);
    }

    RaftServerConfigKeys.Log.CorruptionPolicy getLogCorruptionPolicy() {
        return RaftServerConfigKeys.Log.CorruptionPolicy.get(this.storage, RaftStorage::getLogCorruptionPolicy);
    }

    void appendToOpenSegment(RaftProtos.LogEntryProto entry) {
        Preconditions.assertTrue(this.isOpen(), "The log segment %s is not open for append", this);
        this.append(true, entry);
    }

    private void append(boolean keepEntryInCache, RaftProtos.LogEntryProto entry) {
        LogRecord currentLast;
        Objects.requireNonNull(entry, "entry == null");
        if (this.records.isEmpty()) {
            Preconditions.assertTrue(entry.getIndex() == this.startIndex, "gap between start index %s and first entry to append %s", this.startIndex, entry.getIndex());
        }
        if ((currentLast = this.getLastRecord()) != null) {
            Preconditions.assertTrue(entry.getIndex() == currentLast.getTermIndex().getIndex() + 1L, "gap between entries %s and %s", entry.getIndex(), currentLast.getTermIndex().getIndex());
        }
        LogRecord record = new LogRecord(this.totalSize, entry);
        this.records.add(record);
        if (keepEntryInCache) {
            this.entryCache.put(record.getTermIndex(), entry);
        }
        if (entry.hasConfigurationEntry()) {
            this.configEntries.add(record.getTermIndex());
        }
        this.totalSize += LogSegment.getEntrySize(entry);
        this.endIndex = entry.getIndex();
    }

    RaftProtos.LogEntryProto getEntryFromCache(TermIndex ti) {
        return this.entryCache.get(ti);
    }

    synchronized RaftProtos.LogEntryProto loadCache(LogRecord record) throws RaftLogIOException {
        RaftProtos.LogEntryProto entry = this.entryCache.get(record.getTermIndex());
        if (entry != null) {
            return entry;
        }
        try {
            return this.cacheLoader.load(record);
        }
        catch (Exception e) {
            throw new RaftLogIOException(e);
        }
    }

    LogRecord getLogRecord(long index) {
        if (index >= this.startIndex && index <= this.endIndex) {
            return this.records.get(Math.toIntExact(index - this.startIndex));
        }
        return null;
    }

    private LogRecord getLastRecord() {
        return this.records.isEmpty() ? null : this.records.get(this.records.size() - 1);
    }

    TermIndex getLastTermIndex() {
        LogRecord last = this.getLastRecord();
        return last == null ? null : last.getTermIndex();
    }

    boolean isConfigEntry(TermIndex ti) {
        return this.configEntries.contains(ti);
    }

    long getTotalSize() {
        return this.totalSize;
    }

    synchronized void truncate(long fromIndex) {
        Preconditions.assertTrue(fromIndex >= this.startIndex && fromIndex <= this.endIndex);
        for (long index = this.endIndex; index >= fromIndex; --index) {
            LogRecord removed = this.records.remove(Math.toIntExact(index - this.startIndex));
            this.entryCache.remove(removed.getTermIndex());
            this.configEntries.remove(removed.getTermIndex());
            this.totalSize = removed.offset;
        }
        this.isOpen = false;
        this.endIndex = fromIndex - 1L;
    }

    void close() {
        Preconditions.assertTrue(this.isOpen());
        this.isOpen = false;
    }

    public String toString() {
        return this.isOpen() ? "log_inprogress_" + this.startIndex : "log-" + this.startIndex + "_" + this.endIndex;
    }

    @Override
    public int compareTo(Long l) {
        return l >= this.getStartIndex() && l <= this.getEndIndex() ? 0 : (this.getEndIndex() < l ? -1 : 1);
    }

    synchronized void clear() {
        this.records.clear();
        this.entryCache.clear();
        this.configEntries.clear();
        this.endIndex = this.startIndex - 1L;
    }

    int getLoadingTimes() {
        return this.loadingTimes.get();
    }

    synchronized void evictCache() {
        this.entryCache.clear();
    }

    boolean hasCache() {
        return this.isOpen || !this.entryCache.isEmpty();
    }

    boolean containsIndex(long index) {
        return this.startIndex <= index && this.endIndex >= index;
    }

    class LogEntryLoader
    extends CacheLoader<LogRecord, RaftProtos.LogEntryProto> {
        private RaftLogMetrics raftLogMetrics;

        LogEntryLoader(RaftLogMetrics raftLogMetrics) {
            this.raftLogMetrics = raftLogMetrics;
        }

        @Override
        public RaftProtos.LogEntryProto load(LogRecord key) throws IOException {
            File file = LogSegment.this.getSegmentFile();
            LogSegment.readSegmentFile(file, LogSegment.this.startIndex, LogSegment.this.endIndex, LogSegment.this.isOpen, LogSegment.this.getLogCorruptionPolicy(), this.raftLogMetrics, entry -> LogSegment.this.entryCache.put(ServerProtoUtils.toTermIndex(entry), entry));
            LogSegment.this.loadingTimes.incrementAndGet();
            return (RaftProtos.LogEntryProto)Objects.requireNonNull(LogSegment.this.entryCache.get(key.getTermIndex()));
        }
    }

    static class LogRecord {
        private final long offset;
        private final TermIndex termIndex;

        LogRecord(long offset, RaftProtos.LogEntryProto entry) {
            this.offset = offset;
            this.termIndex = ServerProtoUtils.toTermIndex(entry);
        }

        TermIndex getTermIndex() {
            return this.termIndex;
        }

        long getOffset() {
            return this.offset;
        }
    }
}

