/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.server.impl;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.stream.LongStream;
import org.apache.hadoop.ozone.shaded.com.codahale.metrics.Timer;
import org.apache.ratis.conf.RaftProperties;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.protocol.Message;
import org.apache.ratis.protocol.StateMachineException;
import org.apache.ratis.server.RaftServerConfigKeys;
import org.apache.ratis.server.impl.RaftServerImpl;
import org.apache.ratis.server.impl.ServerProtoUtils;
import org.apache.ratis.server.impl.ServerState;
import org.apache.ratis.server.impl.StateMachineMetrics;
import org.apache.ratis.server.raftlog.RaftLog;
import org.apache.ratis.server.raftlog.RaftLogIOException;
import org.apache.ratis.server.raftlog.RaftLogIndex;
import org.apache.ratis.statemachine.SnapshotInfo;
import org.apache.ratis.statemachine.StateMachine;
import org.apache.ratis.statemachine.impl.SnapshotRetentionPolicy;
import org.apache.ratis.util.Daemon;
import org.apache.ratis.util.JavaUtils;
import org.apache.ratis.util.LifeCycle;
import org.apache.ratis.util.MemoizedSupplier;
import org.apache.ratis.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class StateMachineUpdater
implements Runnable {
    static final Logger LOG = LoggerFactory.getLogger(StateMachineUpdater.class);
    private final Consumer<Object> infoIndexChange;
    private final Consumer<Object> debugIndexChange;
    private final String name;
    private final StateMachine stateMachine;
    private final RaftServerImpl server;
    private final RaftLog raftLog;
    private final Long autoSnapshotThreshold;
    private final Thread updater;
    private final RaftLogIndex appliedIndex;
    private final RaftLogIndex snapshotIndex;
    private final AtomicReference<Long> stopIndex = new AtomicReference();
    private volatile State state = State.RUNNING;
    private SnapshotRetentionPolicy snapshotRetentionPolicy;
    private StateMachineMetrics stateMachineMetrics = null;

    StateMachineUpdater(StateMachine stateMachine, RaftServerImpl server, ServerState serverState, long lastAppliedIndex, RaftProperties properties) {
        this.name = serverState.getMemberId() + "-" + this.getClass().getSimpleName();
        this.infoIndexChange = s2 -> LOG.info("{}: {}", (Object)this.name, s2);
        this.debugIndexChange = s2 -> LOG.debug("{}: {}", (Object)this.name, s2);
        this.stateMachine = stateMachine;
        this.server = server;
        this.raftLog = serverState.getLog();
        this.appliedIndex = new RaftLogIndex("appliedIndex", lastAppliedIndex);
        this.snapshotIndex = new RaftLogIndex("snapshotIndex", lastAppliedIndex);
        boolean autoSnapshot = RaftServerConfigKeys.Snapshot.autoTriggerEnabled(properties);
        this.autoSnapshotThreshold = autoSnapshot ? Long.valueOf(RaftServerConfigKeys.Snapshot.autoTriggerThreshold(properties)) : null;
        final int numSnapshotFilesRetained = RaftServerConfigKeys.Snapshot.retentionFileNum(properties);
        this.snapshotRetentionPolicy = new SnapshotRetentionPolicy(){

            @Override
            public int getNumSnapshotsRetained() {
                return numSnapshotFilesRetained;
            }
        };
        this.updater = new Daemon(this);
    }

    void start() {
        this.initializeMetrics();
        this.updater.start();
    }

    private void initializeMetrics() {
        if (this.stateMachineMetrics == null) {
            this.stateMachineMetrics = StateMachineMetrics.getStateMachineMetrics(this.server, this.appliedIndex, this.stateMachine);
        }
    }

    private void stop() {
        this.state = State.STOP;
        try {
            this.stateMachine.close();
            this.stateMachineMetrics.unregister();
        }
        catch (Throwable t) {
            LOG.warn(this.name + ": Failed to close " + this.stateMachine.getClass().getSimpleName() + " " + this.stateMachine, t);
        }
    }

    void stopAndJoin() throws InterruptedException {
        if (this.stopIndex.compareAndSet(null, this.raftLog.getLastCommittedIndex())) {
            this.notifyUpdater();
            LOG.info("{}: set stopIndex = {}", (Object)this, this.stopIndex);
        }
        this.updater.join();
    }

    void reloadStateMachine() {
        this.state = State.RELOAD;
        this.notifyUpdater();
    }

    synchronized void notifyUpdater() {
        this.notifyAll();
    }

    public String toString() {
        return this.name;
    }

    @Override
    public void run() {
        while (this.state != State.STOP) {
            String s2;
            try {
                this.waitForCommit();
                if (this.state == State.RELOAD) {
                    this.reload();
                }
                MemoizedSupplier<List<CompletableFuture<Message>>> futures = this.applyLog();
                this.checkAndTakeSnapshot(futures);
                if (!this.shouldStop()) continue;
                this.checkAndTakeSnapshot(futures);
                this.stop();
            }
            catch (InterruptedException e) {
                if (this.state == State.STOP) {
                    LOG.info("{}: the StateMachineUpdater is interrupted and will exit.", (Object)this);
                    continue;
                }
                s2 = this + ": the StateMachineUpdater is wrongly interrupted";
                LOG.error(s2, (Throwable)e);
                this.server.shutdown(false);
            }
            catch (Throwable t) {
                s2 = this + ": the StateMachineUpdater hits Throwable";
                LOG.error(s2, t);
                this.server.shutdown(false);
            }
        }
    }

    private synchronized void waitForCommit() throws InterruptedException {
        long applied = this.getLastAppliedIndex();
        while (applied >= this.raftLog.getLastCommittedIndex() && this.state == State.RUNNING && !this.shouldStop()) {
            this.wait();
        }
    }

    private void reload() throws IOException {
        Preconditions.assertTrue(this.stateMachine.getLifeCycleState() == LifeCycle.State.PAUSED);
        this.stateMachine.reinitialize();
        SnapshotInfo snapshot = this.stateMachine.getLatestSnapshot();
        Objects.requireNonNull(snapshot, "snapshot == null");
        long i = snapshot.getIndex();
        this.snapshotIndex.setUnconditionally(i, this.infoIndexChange);
        this.appliedIndex.setUnconditionally(i, this.infoIndexChange);
        this.state = State.RUNNING;
    }

    private MemoizedSupplier<List<CompletableFuture<Message>>> applyLog() throws RaftLogIOException {
        long applied;
        MemoizedSupplier<List<CompletableFuture<Message>>> futures = MemoizedSupplier.valueOf(ArrayList::new);
        long committed = this.raftLog.getLastCommittedIndex();
        while ((applied = this.getLastAppliedIndex()) < committed && this.state == State.RUNNING && !this.shouldStop()) {
            long nextIndex = applied + 1L;
            RaftProtos.LogEntryProto next = this.raftLog.get(nextIndex);
            if (next != null) {
                long incremented;
                if (LOG.isTraceEnabled()) {
                    LOG.trace("{}: applying nextIndex={}, nextLog={}", new Object[]{this, nextIndex, ServerProtoUtils.toString(next)});
                } else {
                    LOG.debug("{}: applying nextIndex={}", (Object)this, (Object)nextIndex);
                }
                CompletableFuture<Message> f = this.server.applyLogToStateMachine(next);
                if (f != null) {
                    futures.get().add(f);
                }
                Preconditions.assertTrue((incremented = this.appliedIndex.incrementAndGet(this.debugIndexChange)) == nextIndex);
                continue;
            }
            LOG.debug("{}: logEntry {} is null. There may be snapshot to load. state:{}", new Object[]{this, nextIndex, this.state});
            break;
        }
        return futures;
    }

    private void checkAndTakeSnapshot(MemoizedSupplier<List<CompletableFuture<Message>>> futures) throws ExecutionException, InterruptedException {
        if (this.shouldTakeSnapshot()) {
            if (futures.isInitialized()) {
                JavaUtils.allOf((Collection)futures.get()).get();
            }
            this.takeSnapshot();
        }
    }

    private void takeSnapshot() {
        long i;
        try {
            Timer.Context takeSnapshotTimerContext = this.stateMachineMetrics.getTakeSnapshotTimer().time();
            i = this.stateMachine.takeSnapshot();
            takeSnapshotTimerContext.stop();
            long appliedIndex = this.getLastAppliedIndex();
            if (i > appliedIndex) {
                throw new StateMachineException("Bug in StateMachine: snapshot index = " + i + " > appliedIndex = " + appliedIndex + "; StateMachine class=" + this.stateMachine.getClass().getName() + ", stateMachine=" + this.stateMachine);
            }
            this.stateMachine.getStateMachineStorage().cleanupOldSnapshots(this.snapshotRetentionPolicy);
        }
        catch (IOException e) {
            LOG.error(this.name + ": Failed to take snapshot", (Throwable)e);
            return;
        }
        if (i >= 0L) {
            LOG.info("{}: Took a snapshot at index {}", (Object)this.name, (Object)i);
            this.snapshotIndex.updateIncreasingly(i, this.infoIndexChange);
            LongStream commitIndexStream = this.server.getCommitInfos().stream().mapToLong(RaftProtos.CommitInfoProto::getCommitIndex);
            long purgeIndex = LongStream.concat(LongStream.of(i), commitIndexStream).min().orElse(i);
            this.raftLog.purge(purgeIndex);
        }
    }

    private boolean shouldStop() {
        return Optional.ofNullable(this.stopIndex.get()).filter(i -> i <= this.getLastAppliedIndex()).isPresent();
    }

    private boolean shouldTakeSnapshot() {
        if (this.autoSnapshotThreshold == null) {
            return false;
        }
        if (this.shouldStop()) {
            return this.getLastAppliedIndex() - this.snapshotIndex.get() > 0L;
        }
        return this.state == State.RUNNING && this.getLastAppliedIndex() - this.snapshotIndex.get() >= this.autoSnapshotThreshold;
    }

    private long getLastAppliedIndex() {
        return this.appliedIndex.get();
    }

    long getStateMachineLastAppliedIndex() {
        return this.stateMachine.getLastAppliedTermIndex().getIndex();
    }

    static enum State {
        RUNNING,
        STOP,
        RELOAD;

    }
}

