/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.keyvalue.impl;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.hdds.client.BlockID;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.scm.container.common.helpers.StorageContainerException;
import org.apache.hadoop.ozone.common.ChunkBuffer;
import org.apache.hadoop.ozone.container.common.helpers.BlockData;
import org.apache.hadoop.ozone.container.common.helpers.ChunkInfo;
import org.apache.hadoop.ozone.container.common.impl.ChunkLayOutVersion;
import org.apache.hadoop.ozone.container.common.impl.ContainerData;
import org.apache.hadoop.ozone.container.common.interfaces.Container;
import org.apache.hadoop.ozone.container.common.transport.server.ratis.DispatcherContext;
import org.apache.hadoop.ozone.container.common.volume.HddsVolume;
import org.apache.hadoop.ozone.container.common.volume.VolumeIOStats;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueContainerData;
import org.apache.hadoop.ozone.container.keyvalue.helpers.ChunkUtils;
import org.apache.hadoop.ozone.container.keyvalue.interfaces.ChunkManager;
import org.apache.hadoop.ozone.shaded.com.google.common.base.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FilePerChunkStrategy
implements ChunkManager {
    private static final Logger LOG = LoggerFactory.getLogger(FilePerChunkStrategy.class);
    private final boolean doSyncWrite;

    public FilePerChunkStrategy(boolean sync) {
        this.doSyncWrite = sync;
    }

    private static void checkLayoutVersion(Container container) {
        Preconditions.checkArgument(((ContainerData)container.getContainerData()).getLayOutVersion() == ChunkLayOutVersion.FILE_PER_CHUNK);
    }

    @Override
    public void writeChunk(Container container, BlockID blockID, ChunkInfo info, ChunkBuffer data, DispatcherContext dispatcherContext) throws StorageContainerException {
        FilePerChunkStrategy.checkLayoutVersion(container);
        Preconditions.checkNotNull(dispatcherContext);
        DispatcherContext.WriteChunkStage stage = dispatcherContext.getStage();
        try {
            KeyValueContainerData containerData = (KeyValueContainerData)container.getContainerData();
            HddsVolume volume = containerData.getVolume();
            VolumeIOStats volumeIOStats = volume.getVolumeIOStats();
            File chunkFile = ChunkUtils.getChunkFile(containerData, info);
            boolean isOverwrite = ChunkUtils.validateChunkForOverwrite(chunkFile, info);
            File tmpChunkFile = this.getTmpChunkFile(chunkFile, dispatcherContext);
            if (LOG.isDebugEnabled()) {
                LOG.debug("writing chunk:{} chunk stage:{} chunk file:{} tmp chunk file:{}", new Object[]{info.getChunkName(), stage, chunkFile, tmpChunkFile});
            }
            long len = info.getLen();
            long offset = 0L;
            switch (stage) {
                case WRITE_DATA: {
                    if (isOverwrite) {
                        LOG.warn("ChunkFile already exists {}. Deleting it.", (Object)chunkFile);
                        FileUtil.fullyDelete((File)chunkFile);
                    }
                    if (tmpChunkFile.exists()) {
                        LOG.warn("tmpChunkFile already exists {}. Overwriting it.", (Object)tmpChunkFile);
                    }
                    ChunkUtils.writeData(tmpChunkFile, data, offset, len, volumeIOStats, this.doSyncWrite);
                    break;
                }
                case COMMIT_DATA: {
                    if (isOverwrite) {
                        LOG.warn("ChunkFile already exists {}", (Object)chunkFile);
                        return;
                    }
                    this.commitChunk(tmpChunkFile, chunkFile);
                    containerData.updateWriteStats(len, isOverwrite);
                    break;
                }
                case COMBINED: {
                    ChunkUtils.writeData(chunkFile, data, offset, len, volumeIOStats, this.doSyncWrite);
                    containerData.updateWriteStats(len, isOverwrite);
                    break;
                }
                default: {
                    throw new IOException("Can not identify write operation.");
                }
            }
        }
        catch (StorageContainerException ex) {
            throw ex;
        }
        catch (IOException ex) {
            throw new StorageContainerException("Internal error: ", ex, ContainerProtos.Result.IO_EXCEPTION);
        }
    }

    @Override
    public ChunkBuffer readChunk(Container container, BlockID blockID, ChunkInfo info, DispatcherContext dispatcherContext) throws StorageContainerException {
        FilePerChunkStrategy.checkLayoutVersion(container);
        KeyValueContainerData containerData = (KeyValueContainerData)container.getContainerData();
        HddsVolume volume = containerData.getVolume();
        VolumeIOStats volumeIOStats = volume.getVolumeIOStats();
        File finalChunkFile = ChunkUtils.getChunkFile(containerData, info);
        ArrayList<File> possibleFiles = new ArrayList<File>();
        possibleFiles.add(finalChunkFile);
        if (dispatcherContext != null && dispatcherContext.isReadFromTmpFile()) {
            possibleFiles.add(this.getTmpChunkFile(finalChunkFile, dispatcherContext));
            possibleFiles.add(finalChunkFile);
        }
        long len = info.getLen();
        long offset = 0L;
        ByteBuffer data = ByteBuffer.allocate((int)len);
        for (File chunkFile : possibleFiles) {
            try {
                ChunkUtils.readData(chunkFile, data, offset, len, volumeIOStats);
                return ChunkBuffer.wrap(data);
            }
            catch (StorageContainerException ex) {
                if (ex.getResult() != ContainerProtos.Result.UNABLE_TO_FIND_CHUNK) {
                    throw ex;
                }
                data.clear();
            }
        }
        throw new StorageContainerException("Chunk file can't be found " + ((Object)possibleFiles).toString(), ContainerProtos.Result.UNABLE_TO_FIND_CHUNK);
    }

    @Override
    public void deleteChunk(Container container, BlockID blockID, ChunkInfo info) throws StorageContainerException {
        FilePerChunkStrategy.checkLayoutVersion(container);
        Preconditions.checkNotNull(blockID, "Block ID cannot be null.");
        KeyValueContainerData containerData = (KeyValueContainerData)container.getContainerData();
        File chunkFile = ChunkUtils.getChunkFile(containerData, info);
        if (!chunkFile.exists()) {
            LOG.warn("Chunk file doe not exist. chunk info :" + info.toString());
            return;
        }
        if (info.getLen() != chunkFile.length()) {
            LOG.error("Not Supported Operation. Trying to delete a chunk that is in shared file. chunk info : {}", (Object)info.toString());
            throw new StorageContainerException("Not Supported Operation. Trying to delete a chunk that is in shared file. chunk info : " + info.toString(), ContainerProtos.Result.UNSUPPORTED_REQUEST);
        }
        FileUtil.fullyDelete((File)chunkFile);
    }

    @Override
    public void deleteChunks(Container container, BlockData blockData) throws StorageContainerException {
        for (ContainerProtos.ChunkInfo chunk : blockData.getChunks()) {
            try {
                ChunkInfo chunkInfo = ChunkInfo.getFromProtoBuf(chunk);
                this.deleteChunk(container, blockData.getBlockID(), chunkInfo);
            }
            catch (IOException e) {
                throw new StorageContainerException(e, ContainerProtos.Result.INVALID_ARGUMENT);
            }
        }
    }

    private File getTmpChunkFile(File chunkFile, DispatcherContext dispatcherContext) {
        return new File(chunkFile.getParent(), chunkFile.getName() + "." + "tmp" + "." + dispatcherContext.getTerm() + "." + dispatcherContext.getLogIndex());
    }

    private void commitChunk(File tmpChunkFile, File chunkFile) throws IOException {
        Files.move(tmpChunkFile.toPath(), chunkFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
    }
}

