/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.keyvalue.impl;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdds.client.BlockID;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.scm.container.common.helpers.StorageContainerException;
import org.apache.hadoop.hdds.utils.BatchOperation;
import org.apache.hadoop.hdds.utils.MetadataKeyFilters;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.ozone.container.common.helpers.BlockData;
import org.apache.hadoop.ozone.container.common.impl.ContainerData;
import org.apache.hadoop.ozone.container.common.interfaces.Container;
import org.apache.hadoop.ozone.container.common.utils.ContainerCache;
import org.apache.hadoop.ozone.container.common.utils.ReferenceCountedDB;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueContainerData;
import org.apache.hadoop.ozone.container.keyvalue.helpers.BlockUtils;
import org.apache.hadoop.ozone.container.keyvalue.interfaces.BlockManager;
import org.apache.hadoop.ozone.shaded.com.google.common.base.Preconditions;
import org.apache.hadoop.ozone.shaded.com.google.common.primitives.Longs;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BlockManagerImpl
implements BlockManager {
    static final Logger LOG = LoggerFactory.getLogger(BlockManagerImpl.class);
    private static byte[] blockCommitSequenceIdKey = DFSUtil.string2Bytes((String)"#BCSID");
    private Configuration config;
    private static final String DB_NULL_ERR_MSG = "DB cannot be null here";
    private static final String NO_SUCH_BLOCK_ERR_MSG = "Unable to find the block.";

    public BlockManagerImpl(Configuration conf) {
        Preconditions.checkNotNull(conf, "Config cannot be null");
        this.config = conf;
    }

    @Override
    public long putBlock(Container container, BlockData data) throws IOException {
        Preconditions.checkNotNull(data, "BlockData cannot be null for put operation.");
        Preconditions.checkState(data.getContainerID() >= 0L, "Container Id cannot be negative");
        try (ReferenceCountedDB db = BlockUtils.getDB((KeyValueContainerData)container.getContainerData(), this.config);){
            Preconditions.checkNotNull(db, DB_NULL_ERR_MSG);
            long bcsId = data.getBlockCommitSequenceId();
            long containerBCSId = ((KeyValueContainerData)container.getContainerData()).getBlockCommitSequenceId();
            if (bcsId != 0L && bcsId <= containerBCSId) {
                LOG.warn("blockCommitSequenceId {} in the Container Db is greater than the supplied value {}. Ignoring it", (Object)containerBCSId, (Object)bcsId);
                long l = data.getSize();
                return l;
            }
            BatchOperation batch = new BatchOperation();
            batch.put(Longs.toByteArray(data.getLocalID()), data.getProtoBufMessage().toByteArray());
            batch.put(blockCommitSequenceIdKey, Longs.toByteArray(bcsId));
            db.getStore().writeBatch(batch);
            container.updateBlockCommitSequenceId(bcsId);
            ((ContainerData)container.getContainerData()).incrKeyCount();
            if (LOG.isDebugEnabled()) {
                LOG.debug("Block " + data.getBlockID() + " successfully committed with bcsId " + bcsId + " chunk size " + data.getChunks().size());
            }
            long l = data.getSize();
            return l;
        }
    }

    @Override
    public BlockData getBlock(Container container, BlockID blockID) throws IOException {
        long bcsId = blockID.getBlockCommitSequenceId();
        Preconditions.checkNotNull(blockID, "BlockID cannot be null in GetBlock request");
        Preconditions.checkNotNull(container, "Container cannot be null");
        KeyValueContainerData containerData = (KeyValueContainerData)container.getContainerData();
        try (ReferenceCountedDB db = BlockUtils.getDB(containerData, this.config);){
            Preconditions.checkNotNull(db, DB_NULL_ERR_MSG);
            long containerBCSId = containerData.getBlockCommitSequenceId();
            if (containerBCSId < bcsId) {
                throw new StorageContainerException("Unable to find the block with bcsID " + bcsId + " .Container " + ((ContainerData)container.getContainerData()).getContainerID() + " bcsId is " + containerBCSId + ".", ContainerProtos.Result.UNKNOWN_BCSID);
            }
            byte[] kData = this.getBlockByID(db, blockID);
            ContainerProtos.BlockData blockData = ContainerProtos.BlockData.parseFrom(kData);
            long id = blockData.getBlockID().getBlockCommitSequenceId();
            if (id < bcsId) {
                throw new StorageContainerException("bcsId " + bcsId + " mismatches with existing block Id " + id + " for block " + blockID + ".", ContainerProtos.Result.BCSID_MISMATCH);
            }
            BlockData blockData2 = BlockData.getFromProtoBuf(blockData);
            return blockData2;
        }
    }

    @Override
    public long getCommittedBlockLength(Container container, BlockID blockID) throws IOException {
        KeyValueContainerData containerData = (KeyValueContainerData)container.getContainerData();
        try (ReferenceCountedDB db = BlockUtils.getDB(containerData, this.config);){
            Preconditions.checkNotNull(db, DB_NULL_ERR_MSG);
            byte[] kData = this.getBlockByID(db, blockID);
            ContainerProtos.BlockData blockData = ContainerProtos.BlockData.parseFrom(kData);
            long l = blockData.getSize();
            return l;
        }
    }

    @Override
    public void deleteBlock(Container container, BlockID blockID) throws IOException {
        Preconditions.checkNotNull(blockID, "block ID cannot be null.");
        Preconditions.checkState(blockID.getContainerID() >= 0L, "Container ID cannot be negative.");
        Preconditions.checkState(blockID.getLocalID() >= 0L, "Local ID cannot be negative.");
        KeyValueContainerData cData = (KeyValueContainerData)container.getContainerData();
        try (ReferenceCountedDB db = BlockUtils.getDB(cData, this.config);){
            Preconditions.checkNotNull(db, DB_NULL_ERR_MSG);
            byte[] kKey = Longs.toByteArray(blockID.getLocalID());
            this.getBlockByID(db, blockID);
            db.getStore().delete(kKey);
            ((ContainerData)container.getContainerData()).decrKeyCount();
        }
    }

    /*
     * Loose catch block
     */
    @Override
    public List<BlockData> listBlock(Container container, long startLocalID, int count) throws IOException {
        Preconditions.checkNotNull(container, "container cannot be null");
        Preconditions.checkState(startLocalID >= 0L, "startLocal ID cannot be negative");
        Preconditions.checkArgument(count > 0, "Count must be a positive number.");
        container.readLock();
        try {
            ArrayList<BlockData> result = null;
            KeyValueContainerData cData = (KeyValueContainerData)container.getContainerData();
            try (ReferenceCountedDB db = BlockUtils.getDB(cData, this.config);){
                result = new ArrayList<BlockData>();
                byte[] startKeyInBytes = Longs.toByteArray(startLocalID);
                List<Map.Entry<byte[], byte[]>> range = db.getStore().getSequentialRangeKVs(startKeyInBytes, count, MetadataKeyFilters.getNormalKeyFilter());
                for (Map.Entry<byte[], byte[]> entry : range) {
                    BlockData value = BlockUtils.getBlockData(entry.getValue());
                    BlockData data = new BlockData(value.getBlockID());
                    result.add(data);
                }
                ArrayList<BlockData> arrayList = result;
                return arrayList;
            }
            {
                catch (Throwable throwable) {
                    throw throwable;
                }
            }
        }
        finally {
            container.readUnlock();
        }
    }

    @Override
    public void shutdown() {
        BlockUtils.shutdownCache(ContainerCache.getInstance(this.config));
    }

    private byte[] getBlockByID(ReferenceCountedDB db, BlockID blockID) throws IOException {
        byte[] blockKey = Longs.toByteArray(blockID.getLocalID());
        byte[] blockData = db.getStore().get(blockKey);
        if (blockData == null) {
            throw new StorageContainerException(NO_SUCH_BLOCK_ERR_MSG, ContainerProtos.Result.NO_SUCH_BLOCK);
        }
        return blockData;
    }
}

