/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.common.transport.server.ratis;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Semaphore;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.StorageUnit;
import org.apache.hadoop.hdds.HddsUtils;
import org.apache.hadoop.hdds.conf.DatanodeRatisServerConfig;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.ratis.ContainerCommandRequestMessage;
import org.apache.hadoop.hdds.scm.container.common.helpers.ContainerNotOpenException;
import org.apache.hadoop.hdds.scm.container.common.helpers.StorageContainerException;
import org.apache.hadoop.hdds.utils.Cache;
import org.apache.hadoop.hdds.utils.ResourceLimitCache;
import org.apache.hadoop.ozone.container.common.interfaces.ContainerDispatcher;
import org.apache.hadoop.ozone.container.common.transport.server.ratis.CSMMetrics;
import org.apache.hadoop.ozone.container.common.transport.server.ratis.DispatcherContext;
import org.apache.hadoop.ozone.container.common.transport.server.ratis.XceiverServerRatis;
import org.apache.hadoop.ozone.container.ozoneimpl.ContainerController;
import org.apache.hadoop.ozone.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.ozone.shaded.com.google.common.base.Preconditions;
import org.apache.hadoop.util.Time;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.protocol.Message;
import org.apache.ratis.protocol.RaftClientRequest;
import org.apache.ratis.protocol.RaftGroupId;
import org.apache.ratis.protocol.RaftGroupMemberId;
import org.apache.ratis.protocol.RaftPeerId;
import org.apache.ratis.protocol.StateMachineException;
import org.apache.ratis.server.RaftServer;
import org.apache.ratis.server.impl.RaftServerProxy;
import org.apache.ratis.server.protocol.TermIndex;
import org.apache.ratis.server.storage.RaftStorage;
import org.apache.ratis.statemachine.StateMachineStorage;
import org.apache.ratis.statemachine.TransactionContext;
import org.apache.ratis.statemachine.impl.BaseStateMachine;
import org.apache.ratis.statemachine.impl.SimpleStateMachineStorage;
import org.apache.ratis.statemachine.impl.SingleFileSnapshotInfo;
import org.apache.ratis.thirdparty.com.google.protobuf.ByteString;
import org.apache.ratis.thirdparty.com.google.protobuf.InvalidProtocolBufferException;
import org.apache.ratis.thirdparty.com.google.protobuf.TextFormat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ContainerStateMachine
extends BaseStateMachine {
    static final Logger LOG = LoggerFactory.getLogger(ContainerStateMachine.class);
    private final SimpleStateMachineStorage storage = new SimpleStateMachineStorage();
    private final RaftGroupId gid;
    private final ContainerDispatcher dispatcher;
    private final ContainerController containerController;
    private final XceiverServerRatis ratisServer;
    private final ConcurrentHashMap<Long, CompletableFuture<ContainerProtos.ContainerCommandResponseProto>> writeChunkFutureMap;
    private final Map<Long, Long> container2BCSIDMap;
    private final ExecutorService[] executors;
    private final List<ThreadPoolExecutor> chunkExecutors;
    private final Map<Long, Long> applyTransactionCompletionMap;
    private final Cache<Long, ByteString> stateMachineDataCache;
    private final AtomicBoolean stateMachineHealthy;
    private final Semaphore applyTransactionSemaphore;
    private final CSMMetrics metrics;

    public ContainerStateMachine(RaftGroupId gid, ContainerDispatcher dispatcher, ContainerController containerController, List<ThreadPoolExecutor> chunkExecutors, XceiverServerRatis ratisServer, Configuration conf) {
        this.gid = gid;
        this.dispatcher = dispatcher;
        this.containerController = containerController;
        this.ratisServer = ratisServer;
        this.metrics = CSMMetrics.create(gid);
        this.writeChunkFutureMap = new ConcurrentHashMap();
        this.applyTransactionCompletionMap = new ConcurrentHashMap<Long, Long>();
        int numPendingRequests = OzoneConfiguration.of(conf).getObject(DatanodeRatisServerConfig.class).getLeaderNumPendingRequests();
        int pendingRequestsByteLimit = (int)conf.getStorageSize("dfs.container.ratis.leader.pending.bytes.limit", "1GB", StorageUnit.BYTES);
        this.stateMachineDataCache = new ResourceLimitCache<Long, ByteString>(new ConcurrentHashMap(), (index, data) -> new int[]{1, data.size()}, numPendingRequests, pendingRequestsByteLimit);
        this.chunkExecutors = chunkExecutors;
        this.container2BCSIDMap = new ConcurrentHashMap<Long, Long>();
        int numContainerOpExecutors = conf.getInt("dfs.container.ratis.num.container.op.executors", 10);
        int maxPendingApplyTransactions = conf.getInt("dfs.container.ratis.statemachine.max.pending.apply-transactions", 100000);
        this.applyTransactionSemaphore = new Semaphore(maxPendingApplyTransactions);
        this.stateMachineHealthy = new AtomicBoolean(true);
        this.executors = new ExecutorService[numContainerOpExecutors];
        int i = 0;
        while (i < numContainerOpExecutors) {
            int index2 = i++;
            this.executors[index2] = Executors.newSingleThreadExecutor(r -> {
                Thread t = new Thread(r);
                t.setName("RatisApplyTransactionExecutor " + index2);
                return t;
            });
        }
    }

    @Override
    public StateMachineStorage getStateMachineStorage() {
        return this.storage;
    }

    public CSMMetrics getMetrics() {
        return this.metrics;
    }

    @Override
    public void initialize(RaftServer server, RaftGroupId id, RaftStorage raftStorage) throws IOException {
        super.initialize(server, id, raftStorage);
        this.storage.init(raftStorage);
        this.ratisServer.notifyGroupAdd(this.gid);
        this.loadSnapshot(this.storage.getLatestSnapshot());
    }

    private long loadSnapshot(SingleFileSnapshotInfo snapshot) throws IOException {
        if (snapshot == null) {
            TermIndex empty = TermIndex.newTermIndex(0L, -1L);
            LOG.info("{}: The snapshot info is null. Setting the last applied indexto:{}", (Object)this.gid, (Object)empty);
            this.setLastAppliedTermIndex(empty);
            return empty.getIndex();
        }
        File snapshotFile = snapshot.getFile().getPath().toFile();
        TermIndex last = SimpleStateMachineStorage.getTermIndexFromSnapshotFile(snapshotFile);
        LOG.info("{}: Setting the last applied index to {}", (Object)this.gid, (Object)last);
        this.setLastAppliedTermIndex(last);
        try (FileInputStream fin = new FileInputStream(snapshotFile);){
            ContainerProtos.Container2BCSIDMapProto proto = ContainerProtos.Container2BCSIDMapProto.parseFrom(fin);
            this.container2BCSIDMap.putAll(proto.getContainer2BCSIDMap());
            this.dispatcher.buildMissingContainerSetAndValidate(this.container2BCSIDMap);
        }
        return last.getIndex();
    }

    public void persistContainerSet(OutputStream out) throws IOException {
        ContainerProtos.Container2BCSIDMapProto.Builder builder = ContainerProtos.Container2BCSIDMapProto.newBuilder();
        builder.putAllContainer2BCSID(this.container2BCSIDMap);
        builder.build().writeTo(out);
    }

    public boolean isStateMachineHealthy() {
        return this.stateMachineHealthy.get();
    }

    @Override
    public long takeSnapshot() throws IOException {
        TermIndex ti = this.getLastAppliedTermIndex();
        long startTime = Time.monotonicNow();
        if (!this.isStateMachineHealthy()) {
            String msg = "Failed to take snapshot  for " + this.gid + " as the stateMachine" + " is unhealthy. The last applied index is at " + ti;
            StateMachineException sme = new StateMachineException(msg);
            LOG.error(msg);
            throw sme;
        }
        if (ti != null && ti.getIndex() != -1L) {
            File snapshotFile = this.storage.getSnapshotFile(ti.getTerm(), ti.getIndex());
            LOG.info("{}: Taking a snapshot at:{} file {}", new Object[]{this.gid, ti, snapshotFile});
            try (FileOutputStream fos = new FileOutputStream(snapshotFile);){
                this.persistContainerSet(fos);
                fos.flush();
                fos.getFD().sync();
            }
            catch (IOException ioe) {
                LOG.error("{}: Failed to write snapshot at:{} file {}", new Object[]{this.gid, ti, snapshotFile});
                throw ioe;
            }
            LOG.info("{}: Finished taking a snapshot at:{} file:{} time:{}", new Object[]{this.gid, ti, snapshotFile, Time.monotonicNow() - startTime});
            return ti.getIndex();
        }
        return -1L;
    }

    @Override
    public TransactionContext startTransaction(RaftClientRequest request) throws IOException {
        long startTime = Time.monotonicNowNanos();
        ContainerProtos.ContainerCommandRequestProto proto = this.message2ContainerCommandRequestProto(request.getMessage());
        Preconditions.checkArgument(request.getRaftGroupId().equals(this.gid));
        try {
            this.dispatcher.validateContainerCommand(proto);
        }
        catch (IOException ioe) {
            if (ioe instanceof ContainerNotOpenException) {
                this.metrics.incNumContainerNotOpenVerifyFailures();
            } else {
                this.metrics.incNumStartTransactionVerifyFailures();
                LOG.error("startTransaction validation failed on leader", (Throwable)ioe);
            }
            TransactionContext ctxt = TransactionContext.newBuilder().setClientRequest(request).setStateMachine(this).setServerRole(RaftProtos.RaftPeerRole.LEADER).build();
            ctxt.setException(ioe);
            return ctxt;
        }
        if (proto.getCmdType() == ContainerProtos.Type.WriteChunk) {
            ContainerProtos.WriteChunkRequestProto write = proto.getWriteChunk();
            ContainerProtos.WriteChunkRequestProto commitWriteChunkProto = ContainerProtos.WriteChunkRequestProto.newBuilder().setBlockID(write.getBlockID()).setChunkData(write.getChunkData()).build();
            ContainerProtos.ContainerCommandRequestProto commitContainerCommandProto = ContainerProtos.ContainerCommandRequestProto.newBuilder(proto).setWriteChunk(commitWriteChunkProto).setTraceID(proto.getTraceID()).build();
            return TransactionContext.newBuilder().setClientRequest(request).setStateMachine(this).setServerRole(RaftProtos.RaftPeerRole.LEADER).setStateMachineContext(startTime).setStateMachineData(write.getData()).setLogData(commitContainerCommandProto.toByteString()).build();
        }
        return TransactionContext.newBuilder().setClientRequest(request).setStateMachine(this).setServerRole(RaftProtos.RaftPeerRole.LEADER).setStateMachineContext(startTime).setLogData(proto.toByteString()).build();
    }

    private ByteString getStateMachineData(RaftProtos.StateMachineLogEntryProto entryProto) {
        return entryProto.getStateMachineEntry().getStateMachineData();
    }

    private static ContainerProtos.ContainerCommandRequestProto getContainerCommandRequestProto(RaftGroupId id, ByteString request) throws InvalidProtocolBufferException {
        return ContainerProtos.ContainerCommandRequestProto.newBuilder(ContainerProtos.ContainerCommandRequestProto.parseFrom(request)).setPipelineID(id.getUuid().toString()).build();
    }

    private ContainerProtos.ContainerCommandRequestProto message2ContainerCommandRequestProto(Message message) throws InvalidProtocolBufferException {
        return ContainerCommandRequestMessage.toProto(message.getContent(), this.gid);
    }

    private ContainerProtos.ContainerCommandResponseProto dispatchCommand(ContainerProtos.ContainerCommandRequestProto requestProto, DispatcherContext context) {
        if (LOG.isTraceEnabled()) {
            LOG.trace("{}: dispatch {} containerID={} pipelineID={} traceID={}", new Object[]{this.gid, requestProto.getCmdType(), requestProto.getContainerID(), requestProto.getPipelineID(), requestProto.getTraceID()});
        }
        ContainerProtos.ContainerCommandResponseProto response = this.dispatcher.dispatch(requestProto, context);
        if (LOG.isTraceEnabled()) {
            LOG.trace("{}: response {}", (Object)this.gid, (Object)response);
        }
        return response;
    }

    private ContainerProtos.ContainerCommandResponseProto runCommand(ContainerProtos.ContainerCommandRequestProto requestProto, DispatcherContext context) {
        return this.dispatchCommand(requestProto, context);
    }

    private ExecutorService getCommandExecutor(ContainerProtos.ContainerCommandRequestProto requestProto) {
        int executorId = (int)(requestProto.getContainerID() % (long)this.executors.length);
        return this.executors[executorId];
    }

    private CompletableFuture<Message> handleWriteChunk(ContainerProtos.ContainerCommandRequestProto requestProto, long entryIndex, long term, long startTime) {
        ContainerProtos.WriteChunkRequestProto write = requestProto.getWriteChunk();
        RaftServer server = this.ratisServer.getServer();
        Preconditions.checkState(server instanceof RaftServerProxy);
        try {
            if (((RaftServerProxy)server).getImpl(this.gid).isLeader()) {
                this.stateMachineDataCache.put(entryIndex, write.getData());
            }
        }
        catch (IOException | InterruptedException ioe) {
            return ContainerStateMachine.completeExceptionally(ioe);
        }
        DispatcherContext context = new DispatcherContext.Builder().setTerm(term).setLogIndex(entryIndex).setStage(DispatcherContext.WriteChunkStage.WRITE_DATA).setContainer2BCSIDMap(this.container2BCSIDMap).build();
        CompletableFuture<Message> raftFuture = new CompletableFuture<Message>();
        CompletableFuture<ContainerProtos.ContainerCommandResponseProto> writeChunkFuture = CompletableFuture.supplyAsync(() -> {
            try {
                return this.runCommand(requestProto, context);
            }
            catch (Exception e) {
                LOG.error(this.gid + ": writeChunk writeStateMachineData failed: blockId" + write.getBlockID() + " logIndex " + entryIndex + " chunkName " + write.getChunkData().getChunkName() + e);
                this.metrics.incNumWriteDataFails();
                this.stateMachineHealthy.set(false);
                raftFuture.completeExceptionally(e);
                throw e;
            }
        }, this.getChunkExecutor(requestProto.getWriteChunk()));
        this.writeChunkFutureMap.put(entryIndex, writeChunkFuture);
        if (LOG.isDebugEnabled()) {
            LOG.debug(this.gid + ": writeChunk writeStateMachineData : blockId " + write.getBlockID() + " logIndex " + entryIndex + " chunkName " + write.getChunkData().getChunkName());
        }
        writeChunkFuture.thenApply(r -> {
            if (r.getResult() != ContainerProtos.Result.SUCCESS && r.getResult() != ContainerProtos.Result.CONTAINER_NOT_OPEN && r.getResult() != ContainerProtos.Result.CLOSED_CONTAINER_IO) {
                StorageContainerException sce = new StorageContainerException(r.getMessage(), r.getResult());
                LOG.error(this.gid + ": writeChunk writeStateMachineData failed: blockId" + write.getBlockID() + " logIndex " + entryIndex + " chunkName " + write.getChunkData().getChunkName() + " Error message: " + r.getMessage() + " Container Result: " + r.getResult());
                this.metrics.incNumWriteDataFails();
                this.stateMachineHealthy.set(false);
                raftFuture.completeExceptionally(sce);
            } else {
                this.metrics.incNumBytesWrittenCount(requestProto.getWriteChunk().getChunkData().getLen());
                if (LOG.isDebugEnabled()) {
                    LOG.debug(this.gid + ": writeChunk writeStateMachineData  completed: blockId" + write.getBlockID() + " logIndex " + entryIndex + " chunkName " + write.getChunkData().getChunkName());
                }
                raftFuture.complete(r::toByteString);
                this.metrics.recordWriteStateMachineCompletion(Time.monotonicNowNanos() - startTime);
            }
            this.writeChunkFutureMap.remove(entryIndex);
            return r;
        });
        return raftFuture;
    }

    private ExecutorService getChunkExecutor(ContainerProtos.WriteChunkRequestProto req) {
        int hash = Objects.hashCode(req.getBlockID());
        if (hash == Integer.MIN_VALUE) {
            hash = Integer.MAX_VALUE;
        }
        int i = Math.abs(hash) % this.chunkExecutors.size();
        return this.chunkExecutors.get(i);
    }

    public CompletableFuture<Message> writeStateMachineData(RaftProtos.LogEntryProto entry) {
        try {
            this.metrics.incNumWriteStateMachineOps();
            long writeStateMachineStartTime = Time.monotonicNowNanos();
            ContainerProtos.ContainerCommandRequestProto requestProto = ContainerStateMachine.getContainerCommandRequestProto(this.gid, entry.getStateMachineLogEntry().getLogData());
            ContainerProtos.WriteChunkRequestProto writeChunk = ContainerProtos.WriteChunkRequestProto.newBuilder(requestProto.getWriteChunk()).setData(this.getStateMachineData(entry.getStateMachineLogEntry())).build();
            requestProto = ContainerProtos.ContainerCommandRequestProto.newBuilder(requestProto).setWriteChunk(writeChunk).build();
            ContainerProtos.Type cmdType = requestProto.getCmdType();
            switch (cmdType) {
                case WriteChunk: {
                    return this.handleWriteChunk(requestProto, entry.getIndex(), entry.getTerm(), writeStateMachineStartTime);
                }
            }
            throw new IllegalStateException("Cmd Type:" + cmdType + " should not have state machine data");
        }
        catch (IOException e) {
            this.metrics.incNumWriteStateMachineFails();
            return ContainerStateMachine.completeExceptionally(e);
        }
    }

    @Override
    public CompletableFuture<Message> query(Message request) {
        try {
            this.metrics.incNumQueryStateMachineOps();
            ContainerProtos.ContainerCommandRequestProto requestProto = this.message2ContainerCommandRequestProto(request);
            return CompletableFuture.completedFuture(this.runCommand(requestProto, null)::toByteString);
        }
        catch (IOException e) {
            this.metrics.incNumQueryStateMachineFails();
            return ContainerStateMachine.completeExceptionally(e);
        }
    }

    private ByteString readStateMachineData(ContainerProtos.ContainerCommandRequestProto requestProto, long term, long index) throws IOException {
        this.metrics.incNumReadStateMachineMissCount();
        ContainerProtos.WriteChunkRequestProto writeChunkRequestProto = requestProto.getWriteChunk();
        ContainerProtos.ChunkInfo chunkInfo = writeChunkRequestProto.getChunkData();
        ContainerProtos.ReadChunkRequestProto.Builder readChunkRequestProto = ContainerProtos.ReadChunkRequestProto.newBuilder().setBlockID(writeChunkRequestProto.getBlockID()).setChunkData(chunkInfo);
        ContainerProtos.ContainerCommandRequestProto dataContainerCommandProto = ContainerProtos.ContainerCommandRequestProto.newBuilder(requestProto).setCmdType(ContainerProtos.Type.ReadChunk).setReadChunk(readChunkRequestProto).build();
        DispatcherContext context = new DispatcherContext.Builder().setTerm(term).setLogIndex(index).setReadFromTmpFile(true).build();
        ContainerProtos.ContainerCommandResponseProto response = this.dispatchCommand(dataContainerCommandProto, context);
        if (response.getResult() != ContainerProtos.Result.SUCCESS) {
            StorageContainerException sce = new StorageContainerException(response.getMessage(), response.getResult());
            LOG.error("gid {} : ReadStateMachine failed. cmd {} logIndex {} msg : {} Container Result: {}", new Object[]{this.gid, response.getCmdType(), index, response.getMessage(), response.getResult()});
            this.stateMachineHealthy.set(false);
            throw sce;
        }
        ContainerProtos.ReadChunkResponseProto responseProto = response.getReadChunk();
        ByteString data = responseProto.getData();
        Preconditions.checkNotNull(data, "read chunk data is null for chunk: %s", (Object)chunkInfo);
        Preconditions.checkState((long)data.size() == chunkInfo.getLen(), "read chunk len=%s does not match chunk expected len=%s for chunk:%s", (Object)data.size(), (Object)chunkInfo.getLen(), (Object)chunkInfo);
        return data;
    }

    private ByteString getCachedStateMachineData(Long logIndex, long term, ContainerProtos.ContainerCommandRequestProto requestProto) throws IOException {
        ByteString data = this.stateMachineDataCache.get(logIndex);
        if (data == null) {
            data = this.readStateMachineData(requestProto, term, logIndex);
        }
        return data;
    }

    @Override
    public CompletableFuture<Void> flushStateMachineData(long index) {
        List<CompletableFuture> futureList = this.writeChunkFutureMap.entrySet().stream().filter(x -> (Long)x.getKey() <= index).map(Map.Entry::getValue).collect(Collectors.toList());
        return CompletableFuture.allOf(futureList.toArray(new CompletableFuture[futureList.size()]));
    }

    @Override
    public CompletableFuture<ByteString> readStateMachineData(RaftProtos.LogEntryProto entry) {
        RaftProtos.StateMachineLogEntryProto smLogEntryProto = entry.getStateMachineLogEntry();
        this.metrics.incNumReadStateMachineOps();
        if (!this.getStateMachineData(smLogEntryProto).isEmpty()) {
            return CompletableFuture.completedFuture(ByteString.EMPTY);
        }
        try {
            ContainerProtos.ContainerCommandRequestProto requestProto = ContainerStateMachine.getContainerCommandRequestProto(this.gid, entry.getStateMachineLogEntry().getLogData());
            Preconditions.checkArgument(!HddsUtils.isReadOnly(requestProto));
            if (requestProto.getCmdType() == ContainerProtos.Type.WriteChunk) {
                CompletableFuture<ByteString> future = new CompletableFuture<ByteString>();
                CompletableFuture.supplyAsync(() -> {
                    try {
                        future.complete(this.getCachedStateMachineData(entry.getIndex(), entry.getTerm(), requestProto));
                    }
                    catch (IOException e) {
                        this.metrics.incNumReadStateMachineFails();
                        future.completeExceptionally(e);
                    }
                    return future;
                }, this.getChunkExecutor(requestProto.getWriteChunk()));
                return future;
            }
            throw new IllegalStateException("Cmd type:" + requestProto.getCmdType() + " cannot have state machine data");
        }
        catch (Exception e) {
            this.metrics.incNumReadStateMachineFails();
            LOG.error("{} unable to read stateMachineData:", (Object)this.gid, (Object)e);
            return ContainerStateMachine.completeExceptionally(e);
        }
    }

    private synchronized void updateLastApplied() {
        Long removed;
        Long appliedTerm = null;
        long appliedIndex = -1L;
        long i = this.getLastAppliedTermIndex().getIndex() + 1L;
        while ((removed = this.applyTransactionCompletionMap.remove(i)) != null) {
            appliedTerm = removed;
            appliedIndex = i++;
        }
        if (appliedTerm != null) {
            this.updateLastAppliedTermIndex(appliedTerm, appliedIndex);
        }
    }

    @Override
    public void notifyIndexUpdate(long term, long index) {
        this.applyTransactionCompletionMap.put(index, term);
        this.updateLastApplied();
    }

    @Override
    public CompletableFuture<Message> applyTransaction(TransactionContext trx) {
        long index = trx.getLogEntry().getIndex();
        this.stateMachineDataCache.remove(index);
        DispatcherContext.Builder builder = new DispatcherContext.Builder().setTerm(trx.getLogEntry().getTerm()).setLogIndex(index);
        long applyTxnStartTime = Time.monotonicNowNanos();
        try {
            this.applyTransactionSemaphore.acquire();
            this.metrics.incNumApplyTransactionsOps();
            ContainerProtos.ContainerCommandRequestProto requestProto = ContainerStateMachine.getContainerCommandRequestProto(this.gid, trx.getStateMachineLogEntry().getLogData());
            ContainerProtos.Type cmdType = requestProto.getCmdType();
            if (cmdType == ContainerProtos.Type.WriteChunk) {
                Preconditions.checkArgument(requestProto.getWriteChunk().getData().isEmpty());
                builder.setStage(DispatcherContext.WriteChunkStage.COMMIT_DATA);
            }
            if (cmdType == ContainerProtos.Type.WriteChunk || cmdType == ContainerProtos.Type.PutSmallFile || cmdType == ContainerProtos.Type.PutBlock || cmdType == ContainerProtos.Type.CreateContainer) {
                builder.setContainer2BCSIDMap(this.container2BCSIDMap);
            }
            CompletableFuture<Message> applyTransactionFuture = new CompletableFuture<Message>();
            CompletableFuture<ContainerProtos.ContainerCommandResponseProto> future = CompletableFuture.supplyAsync(() -> {
                try {
                    return this.runCommand(requestProto, builder.build());
                }
                catch (Exception e) {
                    LOG.error("gid {} : ApplyTransaction failed. cmd {} logIndex {} exception {}", new Object[]{this.gid, requestProto.getCmdType(), index, e});
                    this.stateMachineHealthy.compareAndSet(true, false);
                    this.metrics.incNumApplyTransactionsFails();
                    applyTransactionFuture.completeExceptionally(e);
                    throw e;
                }
            }, this.getCommandExecutor(requestProto));
            ((CompletableFuture)future.thenApply(r -> {
                if (trx.getServerRole() == RaftProtos.RaftPeerRole.LEADER) {
                    long startTime = (Long)trx.getStateMachineContext();
                    this.metrics.incPipelineLatency(cmdType, Time.monotonicNowNanos() - startTime);
                }
                if (r.getResult() != ContainerProtos.Result.SUCCESS && r.getResult() != ContainerProtos.Result.CONTAINER_NOT_OPEN && r.getResult() != ContainerProtos.Result.CLOSED_CONTAINER_IO) {
                    StorageContainerException sce = new StorageContainerException(r.getMessage(), r.getResult());
                    LOG.error("gid {} : ApplyTransaction failed. cmd {} logIndex {} msg : {} Container Result: {}", new Object[]{this.gid, r.getCmdType(), index, r.getMessage(), r.getResult()});
                    this.metrics.incNumApplyTransactionsFails();
                    applyTransactionFuture.completeExceptionally(sce);
                    this.stateMachineHealthy.compareAndSet(true, false);
                    this.ratisServer.handleApplyTransactionFailure(this.gid, trx.getServerRole());
                } else {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("gid {} : ApplyTransaction completed. cmd {} logIndex {} msg : {} Container Result: {}", new Object[]{this.gid, r.getCmdType(), index, r.getMessage(), r.getResult()});
                    }
                    applyTransactionFuture.complete(r::toByteString);
                    if (cmdType == ContainerProtos.Type.WriteChunk || cmdType == ContainerProtos.Type.PutSmallFile) {
                        this.metrics.incNumBytesCommittedCount(requestProto.getWriteChunk().getChunkData().getLen());
                    }
                    if (this.isStateMachineHealthy()) {
                        Long previous = this.applyTransactionCompletionMap.put(index, trx.getLogEntry().getTerm());
                        Preconditions.checkState(previous == null);
                        this.updateLastApplied();
                    }
                }
                return applyTransactionFuture;
            })).whenComplete((r, t) -> {
                this.applyTransactionSemaphore.release();
                this.metrics.recordApplyTransactionCompletion(Time.monotonicNowNanos() - applyTxnStartTime);
            });
            return applyTransactionFuture;
        }
        catch (IOException | InterruptedException e) {
            this.metrics.incNumApplyTransactionsFails();
            return ContainerStateMachine.completeExceptionally(e);
        }
    }

    private static <T> CompletableFuture<T> completeExceptionally(Exception e) {
        CompletableFuture future = new CompletableFuture();
        future.completeExceptionally(e);
        return future;
    }

    @Override
    public CompletableFuture<Void> truncateStateMachineData(long index) {
        this.stateMachineDataCache.removeIf(k -> k >= index);
        return CompletableFuture.completedFuture(null);
    }

    @VisibleForTesting
    public void evictStateMachineCache() {
        this.stateMachineDataCache.clear();
    }

    @Override
    public void notifySlowness(RaftProtos.RoleInfoProto roleInfoProto) {
        this.ratisServer.handleNodeSlowness(this.gid, roleInfoProto);
    }

    @Override
    public void notifyExtendedNoLeader(RaftProtos.RoleInfoProto roleInfoProto) {
        this.ratisServer.handleNoLeader(this.gid, roleInfoProto);
    }

    @Override
    public void notifyLogFailed(Throwable t, RaftProtos.LogEntryProto failedEntry) {
        this.ratisServer.handleNodeLogFailure(this.gid, t);
    }

    @Override
    public CompletableFuture<TermIndex> notifyInstallSnapshotFromLeader(RaftProtos.RoleInfoProto roleInfoProto, TermIndex firstTermIndexInLog) {
        this.ratisServer.handleInstallSnapshotFromLeader(this.gid, roleInfoProto, firstTermIndexInLog);
        CompletableFuture<TermIndex> future = new CompletableFuture<TermIndex>();
        future.complete(firstTermIndexInLog);
        return future;
    }

    @Override
    public void notifyGroupRemove() {
        this.ratisServer.notifyGroupRemove(this.gid);
        for (Long cid : this.container2BCSIDMap.keySet()) {
            try {
                this.containerController.markContainerForClose(cid);
                this.containerController.quasiCloseContainer(cid);
            }
            catch (IOException iOException) {}
        }
    }

    @Override
    public void close() throws IOException {
        this.evictStateMachineCache();
        for (ExecutorService executor : this.executors) {
            executor.shutdown();
        }
        this.metrics.unRegister();
    }

    @Override
    public void notifyLeaderChanged(RaftGroupMemberId groupMemberId, RaftPeerId raftPeerId) {
        this.ratisServer.handleLeaderChangedNotification(groupMemberId, raftPeerId);
    }

    @Override
    public String toStateMachineLogEntryString(RaftProtos.StateMachineLogEntryProto proto) {
        return ContainerStateMachine.smProtoToString(this.gid, this.containerController, proto);
    }

    public static String smProtoToString(RaftGroupId gid, ContainerController containerController, RaftProtos.StateMachineLogEntryProto proto) {
        StringBuilder builder = new StringBuilder();
        try {
            ContainerProtos.ContainerCommandRequestProto requestProto = ContainerStateMachine.getContainerCommandRequestProto(gid, proto.getLogData());
            long contId = requestProto.getContainerID();
            builder.append(TextFormat.shortDebugString(requestProto));
            if (containerController != null) {
                String location = containerController.getContainerLocation(contId);
                builder.append(", container path=");
                builder.append(location);
            }
        }
        catch (Throwable t) {
            LOG.info("smProtoToString failed", t);
            builder.append("smProtoToString failed with");
            builder.append(t.getMessage());
        }
        return builder.toString();
    }
}

