/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone;

import java.io.File;
import java.io.IOException;
import java.net.InetAddress;
import java.security.KeyPair;
import java.security.cert.CertificateException;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.hadoop.conf.Configurable;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdds.HddsUtils;
import org.apache.hadoop.hdds.cli.GenericCli;
import org.apache.hadoop.hdds.cli.HddsVersionProvider;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.SCMSecurityProtocolProtos;
import org.apache.hadoop.hdds.protocolPB.SCMSecurityProtocolClientSideTranslatorPB;
import org.apache.hadoop.hdds.security.x509.SecurityConfig;
import org.apache.hadoop.hdds.security.x509.certificate.client.CertificateClient;
import org.apache.hadoop.hdds.security.x509.certificate.client.DNCertificateClient;
import org.apache.hadoop.hdds.security.x509.certificate.utils.CertificateCodec;
import org.apache.hadoop.hdds.security.x509.certificates.utils.CertificateSignRequest;
import org.apache.hadoop.hdds.server.http.RatisDropwizardExports;
import org.apache.hadoop.hdds.tracing.TracingUtil;
import org.apache.hadoop.hdds.utils.HddsServerUtil;
import org.apache.hadoop.ozone.HddsDatanodeHttpServer;
import org.apache.hadoop.ozone.OzoneSecurityUtil;
import org.apache.hadoop.ozone.container.common.helpers.ContainerUtils;
import org.apache.hadoop.ozone.container.common.statemachine.DatanodeStateMachine;
import org.apache.hadoop.ozone.container.common.utils.HddsVolumeUtil;
import org.apache.hadoop.ozone.container.common.volume.HddsVolume;
import org.apache.hadoop.ozone.container.common.volume.MutableVolumeSet;
import org.apache.hadoop.ozone.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.ozone.shaded.com.google.common.base.Preconditions;
import org.apache.hadoop.ozone.shaded.io.prometheus.client.CollectorRegistry;
import org.apache.hadoop.ozone.shaded.org.bouncycastle.pkcs.PKCS10CertificationRequest;
import org.apache.hadoop.ozone.shaded.picocli.CommandLine;
import org.apache.hadoop.security.SecurityUtil;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.authentication.client.AuthenticationException;
import org.apache.hadoop.util.ExitUtil;
import org.apache.hadoop.util.ServicePlugin;
import org.apache.hadoop.util.StringUtils;
import org.apache.ratis.metrics.MetricRegistries;
import org.apache.ratis.metrics.MetricsReporting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@CommandLine.Command(name="ozone datanode", hidden=true, description={"Start the datanode for ozone"}, versionProvider=HddsVersionProvider.class, mixinStandardHelpOptions=true)
public class HddsDatanodeService
extends GenericCli
implements ServicePlugin {
    private static final Logger LOG = LoggerFactory.getLogger(HddsDatanodeService.class);
    private OzoneConfiguration conf;
    private DatanodeDetails datanodeDetails;
    private DatanodeStateMachine datanodeStateMachine;
    private List<ServicePlugin> plugins;
    private CertificateClient dnCertClient;
    private String component;
    private HddsDatanodeHttpServer httpServer;
    private boolean printBanner;
    private String[] args;
    private volatile AtomicBoolean isStopped = new AtomicBoolean(false);

    public HddsDatanodeService() {
    }

    public HddsDatanodeService(boolean printBanner, String[] args) {
        this.printBanner = printBanner;
        this.args = args != null ? Arrays.copyOf(args, args.length) : null;
    }

    @VisibleForTesting
    public static HddsDatanodeService createHddsDatanodeService(String[] args) {
        return HddsDatanodeService.createHddsDatanodeService(args, false);
    }

    private static HddsDatanodeService createHddsDatanodeService(String[] args, boolean printBanner) {
        return new HddsDatanodeService(printBanner, args);
    }

    public static void main(String[] args) {
        try {
            HddsDatanodeService hddsDatanodeService = HddsDatanodeService.createHddsDatanodeService(args, true);
            hddsDatanodeService.run(args);
        }
        catch (Throwable e) {
            LOG.error("Exception in HddsDatanodeService.", e);
            ExitUtil.terminate((int)1, (Throwable)e);
        }
    }

    public static Logger getLogger() {
        return LOG;
    }

    @Override
    public Void call() throws Exception {
        if (this.printBanner) {
            StringUtils.startupShutdownMessage(HddsDatanodeService.class, (String[])this.args, (Logger)LOG);
        }
        this.start(this.createOzoneConfiguration());
        this.join();
        return null;
    }

    public void setConfiguration(OzoneConfiguration configuration) {
        this.conf = configuration;
    }

    public void start(Object service) {
        if (service instanceof Configurable) {
            this.start(new OzoneConfiguration(((Configurable)service).getConf()));
        } else {
            this.start(new OzoneConfiguration());
        }
    }

    public void start(OzoneConfiguration configuration) {
        this.setConfiguration(configuration);
        this.start();
    }

    public void start() {
        MetricRegistries.global().addReporterRegistration(MetricsReporting.jmxReporter());
        MetricRegistries.global().addReporterRegistration(registry -> CollectorRegistry.defaultRegistry.register(new RatisDropwizardExports(registry.getDropWizardMetricRegistry())));
        OzoneConfiguration.activate();
        HddsServerUtil.initializeMetrics(this.conf, "HddsDatanode");
        try {
            String hostname = HddsUtils.getHostName(this.conf);
            String ip = InetAddress.getByName(hostname).getHostAddress();
            this.datanodeDetails = this.initializeDatanodeDetails();
            this.datanodeDetails.setHostName(hostname);
            this.datanodeDetails.setIpAddress(ip);
            TracingUtil.initTracing("HddsDatanodeService." + this.datanodeDetails.getUuidString().substring(0, 8), this.conf);
            LOG.info("HddsDatanodeService host:{} ip:{}", (Object)hostname, (Object)ip);
            if (OzoneSecurityUtil.isSecurityEnabled(this.conf)) {
                this.component = "dn-" + this.datanodeDetails.getUuidString();
                this.dnCertClient = new DNCertificateClient(new SecurityConfig(this.conf), this.datanodeDetails.getCertSerialId());
                if (!SecurityUtil.getAuthenticationMethod((Configuration)this.conf).equals((Object)UserGroupInformation.AuthenticationMethod.KERBEROS)) {
                    throw new AuthenticationException(SecurityUtil.getAuthenticationMethod((Configuration)this.conf) + " authentication method not " + "supported. Datanode user" + " login " + "failed.");
                }
                LOG.info("Ozone security is enabled. Attempting login for Hdds Datanode user. Principal: {},keytab: {}", (Object)this.conf.get("dfs.datanode.kerberos.principal"), (Object)this.conf.get("dfs.datanode.keytab.file"));
                UserGroupInformation.setConfiguration((Configuration)this.conf);
                SecurityUtil.login((Configuration)this.conf, (String)"dfs.datanode.keytab.file", (String)"dfs.datanode.kerberos.principal", (String)hostname);
                LOG.info("Hdds Datanode login successful.");
            }
            if (OzoneSecurityUtil.isSecurityEnabled(this.conf)) {
                this.initializeCertificateClient(this.conf);
            }
            this.datanodeStateMachine = new DatanodeStateMachine(this.datanodeDetails, this.conf, this.dnCertClient, this::terminateDatanode);
            try {
                this.httpServer = new HddsDatanodeHttpServer(this.conf);
                this.httpServer.start();
            }
            catch (Exception ex) {
                LOG.error("HttpServer failed to start.", (Throwable)ex);
            }
            this.startPlugins();
            this.datanodeStateMachine.startDaemon();
            if ("follower".equalsIgnoreCase(System.getenv("OZONE_DATANODE_STANDALONE_TEST"))) {
                this.startRatisForTest();
            }
        }
        catch (IOException e) {
            throw new RuntimeException("Can't start the HDDS datanode plugin", e);
        }
        catch (AuthenticationException ex) {
            throw new RuntimeException("Fail to authentication when starting HDDS datanode plugin", ex);
        }
    }

    private void startRatisForTest() throws IOException {
        String scmId = "scm-01";
        String clusterId = "clusterId";
        this.datanodeStateMachine.getContainer().start(scmId);
        MutableVolumeSet volumeSet = this.getDatanodeStateMachine().getContainer().getVolumeSet();
        Map<String, HddsVolume> volumeMap = volumeSet.getVolumeMap();
        for (Map.Entry<String, HddsVolume> entry : volumeMap.entrySet()) {
            HddsVolume hddsVolume = entry.getValue();
            boolean result = HddsVolumeUtil.checkVolume(hddsVolume, scmId, clusterId, LOG);
            if (result) continue;
            volumeSet.failVolume(hddsVolume.getHddsRootDir().getPath());
        }
    }

    @VisibleForTesting
    public void initializeCertificateClient(OzoneConfiguration config) throws IOException {
        LOG.info("Initializing secure Datanode.");
        CertificateClient.InitResponse response = this.dnCertClient.init();
        LOG.info("Init response: {}", (Object)response);
        switch (response) {
            case SUCCESS: {
                LOG.info("Initialization successful, case:{}.", (Object)response);
                break;
            }
            case GETCERT: {
                this.getSCMSignedCert(config);
                LOG.info("Successfully stored SCM signed certificate, case:{}.", (Object)response);
                break;
            }
            case FAILURE: {
                LOG.error("DN security initialization failed, case:{}.", (Object)response);
                throw new RuntimeException("DN security initialization failed.");
            }
            case RECOVER: {
                LOG.error("DN security initialization failed, case:{}. OM certificate is missing.", (Object)response);
                throw new RuntimeException("DN security initialization failed.");
            }
            default: {
                LOG.error("DN security initialization failed. Init response: {}", (Object)response);
                throw new RuntimeException("DN security initialization failed.");
            }
        }
    }

    private void getSCMSignedCert(OzoneConfiguration config) {
        try {
            PKCS10CertificationRequest csr = this.getCSR(config);
            SCMSecurityProtocolClientSideTranslatorPB secureScmClient = HddsServerUtil.getScmSecurityClient(config);
            SCMSecurityProtocolProtos.SCMGetCertResponseProto response = secureScmClient.getDataNodeCertificateChain(this.datanodeDetails.getProtoBufMessage(), CertificateSignRequest.getEncodedString(csr));
            if (!response.hasX509CACertificate()) {
                throw new RuntimeException("Unable to retrieve datanode certificate chain");
            }
            String pemEncodedCert = response.getX509Certificate();
            this.dnCertClient.storeCertificate(pemEncodedCert, true);
            this.dnCertClient.storeCertificate(response.getX509CACertificate(), true, true);
            this.datanodeDetails.setCertSerialId(CertificateCodec.getX509Certificate(pemEncodedCert).getSerialNumber().toString());
            this.persistDatanodeDetails(this.datanodeDetails);
        }
        catch (IOException | CertificateException e) {
            LOG.error("Error while storing SCM signed certificate.", (Throwable)e);
            throw new RuntimeException(e);
        }
    }

    @VisibleForTesting
    public PKCS10CertificationRequest getCSR(Configuration config) throws IOException {
        CertificateSignRequest.Builder builder = this.dnCertClient.getCSRBuilder();
        KeyPair keyPair = new KeyPair(this.dnCertClient.getPublicKey(), this.dnCertClient.getPrivateKey());
        String hostname = InetAddress.getLocalHost().getCanonicalHostName();
        String subject = UserGroupInformation.getCurrentUser().getShortUserName() + "@" + hostname;
        builder.setCA(false).setKey(keyPair).setConfiguration(config).setSubject(subject);
        LOG.info("Creating csr for DN-> subject:{}", (Object)subject);
        return builder.build();
    }

    private DatanodeDetails initializeDatanodeDetails() throws IOException {
        String idFilePath = HddsServerUtil.getDatanodeIdFilePath(this.conf);
        if (idFilePath == null || idFilePath.isEmpty()) {
            LOG.error("A valid path is needed for config setting {}", (Object)"ozone.scm.datanode.id.dir");
            throw new IllegalArgumentException("ozone.scm.datanode.id.dir must be defined. See https://wiki.apache.org/hadoop/Ozone#Configuration for details on configuring Ozone.");
        }
        Preconditions.checkNotNull(idFilePath);
        File idFile = new File(idFilePath);
        if (idFile.exists()) {
            return ContainerUtils.readDatanodeDetailsFrom(idFile);
        }
        String datanodeUuid = UUID.randomUUID().toString();
        return DatanodeDetails.newBuilder().setUuid(datanodeUuid).build();
    }

    private void persistDatanodeDetails(DatanodeDetails dnDetails) throws IOException {
        String idFilePath = HddsServerUtil.getDatanodeIdFilePath(this.conf);
        if (idFilePath == null || idFilePath.isEmpty()) {
            LOG.error("A valid path is needed for config setting {}", (Object)"ozone.scm.datanode.id.dir");
            throw new IllegalArgumentException("ozone.scm.datanode.id.dir must be defined. See https://wiki.apache.org/hadoop/Ozone#Configuration for details on configuring Ozone.");
        }
        Preconditions.checkNotNull(idFilePath);
        File idFile = new File(idFilePath);
        ContainerUtils.writeDatanodeDetailsTo(dnDetails, idFile);
    }

    private void startPlugins() {
        try {
            this.plugins = this.conf.getInstances("hdds.datanode.plugins", ServicePlugin.class);
        }
        catch (RuntimeException e) {
            String pluginsValue = this.conf.get("hdds.datanode.plugins");
            LOG.error("Unable to load HDDS DataNode plugins. Specified list of plugins: {}", (Object)pluginsValue, (Object)e);
            throw e;
        }
        for (ServicePlugin plugin : this.plugins) {
            try {
                plugin.start((Object)this);
                LOG.info("Started plug-in {}", (Object)plugin);
            }
            catch (Throwable t) {
                LOG.warn("ServicePlugin {} could not be started", (Object)plugin, (Object)t);
            }
        }
    }

    public OzoneConfiguration getConf() {
        return this.conf;
    }

    @VisibleForTesting
    public DatanodeDetails getDatanodeDetails() {
        return this.datanodeDetails;
    }

    @VisibleForTesting
    public DatanodeStateMachine getDatanodeStateMachine() {
        return this.datanodeStateMachine;
    }

    public void join() {
        if (this.datanodeStateMachine != null) {
            try {
                this.datanodeStateMachine.join();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                LOG.info("Interrupted during StorageContainerManager join.");
            }
        }
    }

    public void terminateDatanode() {
        this.stop();
        ExitUtil.terminate((int)1);
    }

    public void stop() {
        if (!this.isStopped.get()) {
            this.isStopped.set(true);
            if (this.plugins != null) {
                for (ServicePlugin plugin : this.plugins) {
                    try {
                        plugin.stop();
                        LOG.info("Stopped plug-in {}", (Object)plugin);
                    }
                    catch (Throwable t) {
                        LOG.warn("ServicePlugin {} could not be stopped", (Object)plugin, (Object)t);
                    }
                }
            }
            if (this.datanodeStateMachine != null) {
                this.datanodeStateMachine.stopDaemon();
            }
            if (this.httpServer != null) {
                try {
                    this.httpServer.stop();
                }
                catch (Exception e) {
                    LOG.error("Stopping HttpServer is failed.", (Throwable)e);
                }
            }
        }
    }

    public void close() {
        if (this.plugins != null) {
            for (ServicePlugin plugin : this.plugins) {
                try {
                    plugin.close();
                }
                catch (Throwable t) {
                    LOG.warn("ServicePlugin {} could not be closed", (Object)plugin, (Object)t);
                }
            }
        }
    }

    @VisibleForTesting
    public String getComponent() {
        return this.component;
    }

    public CertificateClient getCertificateClient() {
        return this.dnCertClient;
    }

    @VisibleForTesting
    public void setCertificateClient(CertificateClient client) {
        this.dnCertClient = client;
    }
}

