/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.utils.db;

import java.io.File;
import java.io.IOException;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.management.ObjectName;
import org.apache.hadoop.hdds.HddsUtils;
import org.apache.hadoop.hdds.StringUtils;
import org.apache.hadoop.hdds.utils.RocksDBStoreMBean;
import org.apache.hadoop.hdds.utils.db.BatchOperation;
import org.apache.hadoop.hdds.utils.db.CodecRegistry;
import org.apache.hadoop.hdds.utils.db.DBCheckpoint;
import org.apache.hadoop.hdds.utils.db.DBStore;
import org.apache.hadoop.hdds.utils.db.DBUpdatesWrapper;
import org.apache.hadoop.hdds.utils.db.RDBBatchOperation;
import org.apache.hadoop.hdds.utils.db.RDBCheckpointManager;
import org.apache.hadoop.hdds.utils.db.RDBMetrics;
import org.apache.hadoop.hdds.utils.db.RDBTable;
import org.apache.hadoop.hdds.utils.db.SequenceNumberNotFoundException;
import org.apache.hadoop.hdds.utils.db.StringCodec;
import org.apache.hadoop.hdds.utils.db.Table;
import org.apache.hadoop.hdds.utils.db.TableConfig;
import org.apache.hadoop.hdds.utils.db.TypedTable;
import org.apache.hadoop.hdds.utils.db.cache.TableCacheImpl;
import org.apache.hadoop.metrics2.util.MBeans;
import org.apache.hadoop.ozone.shaded.com.google.common.base.Preconditions;
import org.apache.hadoop.ozone.shaded.org.rocksdb.ColumnFamilyDescriptor;
import org.apache.hadoop.ozone.shaded.org.rocksdb.ColumnFamilyHandle;
import org.apache.hadoop.ozone.shaded.org.rocksdb.DBOptions;
import org.apache.hadoop.ozone.shaded.org.rocksdb.FlushOptions;
import org.apache.hadoop.ozone.shaded.org.rocksdb.RocksDB;
import org.apache.hadoop.ozone.shaded.org.rocksdb.RocksDBException;
import org.apache.hadoop.ozone.shaded.org.rocksdb.TransactionLogIterator;
import org.apache.hadoop.ozone.shaded.org.rocksdb.WriteOptions;
import org.apache.ratis.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RDBStore
implements DBStore {
    private static final Logger LOG = LoggerFactory.getLogger(RDBStore.class);
    private RocksDB db;
    private File dbLocation;
    private final WriteOptions writeOptions;
    private final DBOptions dbOptions;
    private final CodecRegistry codecRegistry;
    private final Map<String, ColumnFamilyHandle> handleTable;
    private ObjectName statMBeanName;
    private RDBCheckpointManager checkPointManager;
    private String checkpointsParentDir;
    private List<ColumnFamilyHandle> columnFamilyHandles;
    private RDBMetrics rdbMetrics;

    @VisibleForTesting
    public RDBStore(File dbFile, DBOptions options, Set<TableConfig> families) throws IOException {
        this(dbFile, options, new WriteOptions(), families, new CodecRegistry());
    }

    public RDBStore(File dbFile, DBOptions options, WriteOptions writeOptions, Set<TableConfig> families, CodecRegistry registry) throws IOException {
        Preconditions.checkNotNull(dbFile, "DB file location cannot be null");
        Preconditions.checkNotNull(families);
        Preconditions.checkArgument(!families.isEmpty());
        this.handleTable = new HashMap<String, ColumnFamilyHandle>();
        this.codecRegistry = registry;
        ArrayList<ColumnFamilyDescriptor> columnFamilyDescriptors = new ArrayList<ColumnFamilyDescriptor>();
        this.columnFamilyHandles = new ArrayList<ColumnFamilyHandle>();
        for (TableConfig family : families) {
            columnFamilyDescriptors.add(family.getDescriptor());
        }
        this.dbOptions = options;
        this.dbLocation = dbFile;
        this.writeOptions = writeOptions;
        try {
            boolean success;
            this.db = RocksDB.open(this.dbOptions, this.dbLocation.getAbsolutePath(), columnFamilyDescriptors, this.columnFamilyHandles);
            for (int x = 0; x < this.columnFamilyHandles.size(); ++x) {
                this.handleTable.put(StringUtils.bytes2String(this.columnFamilyHandles.get(x).getName()), this.columnFamilyHandles.get(x));
            }
            if (this.dbOptions.statistics() != null) {
                HashMap<String, String> jmxProperties = new HashMap<String, String>();
                jmxProperties.put("dbName", dbFile.getName());
                this.statMBeanName = HddsUtils.registerWithJmxProperties("Ozone", "RocksDbStore", jmxProperties, RocksDBStoreMBean.create(this.dbOptions.statistics(), dbFile.getName()));
                if (this.statMBeanName == null) {
                    LOG.warn("jmx registration failed during RocksDB init, db path :{}", (Object)dbFile.getAbsolutePath());
                }
            }
            this.checkpointsParentDir = Paths.get(this.dbLocation.getParent(), "db.checkpoints").toString();
            File checkpointsDir = new File(this.checkpointsParentDir);
            if (!checkpointsDir.exists() && !(success = checkpointsDir.mkdir())) {
                LOG.warn("Unable to create RocksDB checkpoint directory");
            }
            this.checkPointManager = new RDBCheckpointManager(this.db, "rdb");
            this.rdbMetrics = RDBMetrics.create();
        }
        catch (RocksDBException e) {
            String msg = "Failed init RocksDB, db path : " + dbFile.getAbsolutePath() + ", " + "exception :" + (e.getCause() == null ? e.getClass().getCanonicalName() + " " + e.getMessage() : e.getCause().getClass().getCanonicalName() + " " + e.getCause().getMessage());
            throw RDBStore.toIOException(msg, e);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("RocksDB successfully opened.");
            LOG.debug("[Option] dbLocation= {}", (Object)this.dbLocation.getAbsolutePath());
            LOG.debug("[Option] createIfMissing = {}", (Object)options.createIfMissing());
            LOG.debug("[Option] maxOpenFiles= {}", (Object)options.maxOpenFiles());
        }
    }

    public static IOException toIOException(String msg, RocksDBException e) {
        String statusCode = e.getStatus() == null ? "N/A" : e.getStatus().getCodeString();
        String errMessage = e.getMessage() == null ? "Unknown error" : e.getMessage();
        String output = msg + "; status : " + statusCode + "; message : " + errMessage;
        return new IOException(output, e);
    }

    @Override
    public void compactDB() throws IOException {
        if (this.db != null) {
            try {
                this.db.compactRange();
            }
            catch (RocksDBException e) {
                throw RDBStore.toIOException("Failed to compact db", e);
            }
        }
    }

    @Override
    public void close() throws IOException {
        for (ColumnFamilyHandle handle : this.handleTable.values()) {
            handle.close();
        }
        if (this.statMBeanName != null) {
            MBeans.unregister((ObjectName)this.statMBeanName);
            this.statMBeanName = null;
        }
        this.rdbMetrics.unRegister();
        if (this.db != null) {
            this.db.close();
        }
        if (this.dbOptions != null) {
            this.dbOptions.close();
        }
        if (this.writeOptions != null) {
            this.writeOptions.close();
        }
    }

    public <K, V> void move(K key, Table<K, V> source, Table<K, V> dest) throws IOException {
        try (BatchOperation batchOperation = this.initBatchOperation();){
            V value = source.get(key);
            dest.putWithBatch(batchOperation, key, value);
            source.deleteWithBatch(batchOperation, key);
            this.commitBatchOperation(batchOperation);
        }
    }

    public <K, V> void move(K key, V value, Table<K, V> source, Table<K, V> dest) throws IOException {
        this.move(key, key, value, source, dest);
    }

    public <K, V> void move(K sourceKey, K destKey, V value, Table<K, V> source, Table<K, V> dest) throws IOException {
        try (BatchOperation batchOperation = this.initBatchOperation();){
            dest.putWithBatch(batchOperation, destKey, value);
            source.deleteWithBatch(batchOperation, sourceKey);
            this.commitBatchOperation(batchOperation);
        }
    }

    @Override
    public long getEstimatedKeyCount() throws IOException {
        try {
            return this.db.getLongProperty("rocksdb.estimate-num-keys");
        }
        catch (RocksDBException e) {
            throw RDBStore.toIOException("Unable to get the estimated count.", e);
        }
    }

    @Override
    public BatchOperation initBatchOperation() {
        return new RDBBatchOperation();
    }

    @Override
    public void commitBatchOperation(BatchOperation operation) throws IOException {
        ((RDBBatchOperation)operation).commit(this.db, this.writeOptions);
    }

    @VisibleForTesting
    protected ObjectName getStatMBeanName() {
        return this.statMBeanName;
    }

    @Override
    public Table<byte[], byte[]> getTable(String name) throws IOException {
        ColumnFamilyHandle handle = this.handleTable.get(name);
        if (handle == null) {
            throw new IOException("No such table in this DB. TableName : " + name);
        }
        return new RDBTable(this.db, handle, this.writeOptions, this.rdbMetrics);
    }

    public <K, V> Table<K, V> getTable(String name, Class<K> keyType, Class<V> valueType) throws IOException {
        return new TypedTable<K, V>(this.getTable(name), this.codecRegistry, keyType, valueType);
    }

    public <K, V> Table<K, V> getTable(String name, Class<K> keyType, Class<V> valueType, TableCacheImpl.CacheCleanupPolicy cleanupPolicy) throws IOException {
        return new TypedTable<K, V>(this.getTable(name), this.codecRegistry, keyType, valueType, cleanupPolicy);
    }

    @Override
    public ArrayList<Table> listTables() {
        ArrayList<Table> returnList = new ArrayList<Table>();
        for (ColumnFamilyHandle handle : this.handleTable.values()) {
            returnList.add(new RDBTable(this.db, handle, this.writeOptions, this.rdbMetrics));
        }
        return returnList;
    }

    @Override
    public void flush() throws IOException {
        try (FlushOptions flushOptions = new FlushOptions();){
            flushOptions.setWaitForFlush(true);
            this.db.flush(flushOptions);
        }
        catch (RocksDBException e) {
            throw RDBStore.toIOException("Unable to Flush RocksDB data", e);
        }
    }

    @Override
    public DBCheckpoint getCheckpoint(boolean flush) throws IOException {
        if (flush) {
            this.flush();
        }
        return this.checkPointManager.createCheckpoint(this.checkpointsParentDir);
    }

    @Override
    public File getDbLocation() {
        return this.dbLocation;
    }

    @Override
    public Map<Integer, String> getTableNames() {
        HashMap<Integer, String> tableNames = new HashMap<Integer, String>();
        StringCodec stringCodec = new StringCodec();
        for (ColumnFamilyHandle columnFamilyHandle : this.columnFamilyHandles) {
            try {
                tableNames.put(columnFamilyHandle.getID(), stringCodec.fromPersistedFormat(columnFamilyHandle.getName()));
            }
            catch (IOException | RocksDBException e) {
                LOG.error("Unexpected exception while reading column family handle name", (Throwable)e);
            }
        }
        return tableNames;
    }

    @Override
    public CodecRegistry getCodecRegistry() {
        return this.codecRegistry;
    }

    @Override
    public DBUpdatesWrapper getUpdatesSince(long sequenceNumber) throws SequenceNumberNotFoundException {
        DBUpdatesWrapper dbUpdatesWrapper = new DBUpdatesWrapper();
        try {
            TransactionLogIterator transactionLogIterator = this.db.getUpdatesSince(sequenceNumber);
            boolean checkValidStartingSeqNumber = true;
            while (transactionLogIterator.isValid()) {
                TransactionLogIterator.BatchResult result = transactionLogIterator.getBatch();
                long currSequenceNumber = result.sequenceNumber();
                if (checkValidStartingSeqNumber && currSequenceNumber > 1L + sequenceNumber) {
                    throw new SequenceNumberNotFoundException("Unable to read data from RocksDB wal to get delta updates. It may have already beenflushed to SSTs.");
                }
                checkValidStartingSeqNumber = false;
                if (currSequenceNumber <= sequenceNumber) {
                    transactionLogIterator.next();
                    continue;
                }
                dbUpdatesWrapper.addWriteBatch(result.writeBatch().data(), result.sequenceNumber());
                transactionLogIterator.next();
            }
        }
        catch (RocksDBException e) {
            LOG.error("Unable to get delta updates since sequenceNumber {} ", (Object)sequenceNumber, (Object)e);
        }
        return dbUpdatesWrapper;
    }

    @VisibleForTesting
    public RocksDB getDb() {
        return this.db;
    }

    public RDBMetrics getMetrics() {
        return this.rdbMetrics;
    }
}

