/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.storage;

import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.hadoop.hdds.client.BlockID;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.scm.XceiverClientManager;
import org.apache.hadoop.hdds.scm.XceiverClientReply;
import org.apache.hadoop.hdds.scm.XceiverClientSpi;
import org.apache.hadoop.hdds.scm.container.common.helpers.StorageContainerException;
import org.apache.hadoop.hdds.scm.pipeline.Pipeline;
import org.apache.hadoop.hdds.scm.storage.BufferPool;
import org.apache.hadoop.hdds.scm.storage.CommitWatcher;
import org.apache.hadoop.hdds.scm.storage.ContainerProtocolCalls;
import org.apache.hadoop.ozone.common.Checksum;
import org.apache.hadoop.ozone.common.ChecksumData;
import org.apache.hadoop.ozone.common.ChunkBuffer;
import org.apache.hadoop.ozone.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.ozone.shaded.com.google.common.base.Preconditions;
import org.apache.ratis.thirdparty.com.google.protobuf.ByteString;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BlockOutputStream
extends OutputStream {
    public static final Logger LOG = LoggerFactory.getLogger(BlockOutputStream.class);
    private AtomicReference<BlockID> blockID;
    private final ContainerProtos.BlockData.Builder containerBlockData;
    private XceiverClientManager xceiverClientManager;
    private XceiverClientSpi xceiverClient;
    private final int bytesPerChecksum;
    private int chunkIndex;
    private final AtomicLong chunkOffset = new AtomicLong();
    private final long streamBufferFlushSize;
    private final long streamBufferMaxSize;
    private final BufferPool bufferPool;
    private final AtomicReference<IOException> ioException;
    private final ExecutorService responseExecutor;
    private long totalDataFlushedLength;
    private long writtenDataLength;
    private List<ChunkBuffer> bufferList;
    private final CommitWatcher commitWatcher;
    private final List<DatanodeDetails> failedServers;
    private final Checksum checksum;

    public BlockOutputStream(BlockID blockID, XceiverClientManager xceiverClientManager, Pipeline pipeline, long streamBufferFlushSize, long streamBufferMaxSize, BufferPool bufferPool, ContainerProtos.ChecksumType checksumType, int bytesPerChecksum) throws IOException {
        this.blockID = new AtomicReference<BlockID>(blockID);
        ContainerProtos.KeyValue keyValue = ContainerProtos.KeyValue.newBuilder().setKey("TYPE").setValue("KEY").build();
        this.containerBlockData = ContainerProtos.BlockData.newBuilder().setBlockID(blockID.getDatanodeBlockIDProtobuf()).addMetadata(keyValue);
        this.xceiverClientManager = xceiverClientManager;
        this.xceiverClient = xceiverClientManager.acquireClient(pipeline);
        this.streamBufferFlushSize = streamBufferFlushSize;
        this.streamBufferMaxSize = streamBufferMaxSize;
        this.bufferPool = bufferPool;
        this.bytesPerChecksum = bytesPerChecksum;
        this.responseExecutor = Executors.newSingleThreadExecutor();
        this.commitWatcher = new CommitWatcher(bufferPool, this.xceiverClient);
        this.bufferList = null;
        this.totalDataFlushedLength = 0L;
        this.writtenDataLength = 0L;
        this.failedServers = new ArrayList<DatanodeDetails>(0);
        this.ioException = new AtomicReference<Object>(null);
        this.checksum = new Checksum(checksumType, bytesPerChecksum);
    }

    public BlockID getBlockID() {
        return this.blockID.get();
    }

    public long getTotalAckDataLength() {
        return this.commitWatcher.getTotalAckDataLength();
    }

    public long getWrittenDataLength() {
        return this.writtenDataLength;
    }

    public List<DatanodeDetails> getFailedServers() {
        return this.failedServers;
    }

    @VisibleForTesting
    public XceiverClientSpi getXceiverClient() {
        return this.xceiverClient;
    }

    @VisibleForTesting
    public long getTotalDataFlushedLength() {
        return this.totalDataFlushedLength;
    }

    @VisibleForTesting
    public BufferPool getBufferPool() {
        return this.bufferPool;
    }

    public IOException getIoException() {
        return this.ioException.get();
    }

    @VisibleForTesting
    public Map<Long, List<ChunkBuffer>> getCommitIndex2flushedDataMap() {
        return this.commitWatcher.getCommitIndex2flushedDataMap();
    }

    @Override
    public void write(int b) throws IOException {
        this.checkOpen();
        byte[] buf = new byte[]{(byte)b};
        this.write(buf, 0, 1);
    }

    @Override
    public void write(byte[] b, int off, int len) throws IOException {
        this.checkOpen();
        if (b == null) {
            throw new NullPointerException();
        }
        if (off < 0 || off > b.length || len < 0 || off + len > b.length || off + len < 0) {
            throw new IndexOutOfBoundsException();
        }
        if (len == 0) {
            return;
        }
        while (len > 0) {
            ChunkBuffer currentBuffer = this.bufferPool.allocateBufferIfNeeded(this.bytesPerChecksum);
            int writeLen = Math.min(currentBuffer.remaining(), len);
            currentBuffer.put(b, off, writeLen);
            if (!currentBuffer.hasRemaining()) {
                this.writeChunk(currentBuffer);
            }
            off += writeLen;
            len -= writeLen;
            this.writtenDataLength += (long)writeLen;
            if (this.shouldFlush()) {
                this.updateFlushLength();
                this.executePutBlock(false, false);
            }
            if (!this.isBufferPoolFull()) continue;
            this.handleFullBuffer();
        }
    }

    private boolean shouldFlush() {
        return this.bufferPool.computeBufferData() % this.streamBufferFlushSize == 0L;
    }

    private void updateFlushLength() {
        this.totalDataFlushedLength = this.writtenDataLength;
    }

    private boolean isBufferPoolFull() {
        return this.bufferPool.computeBufferData() == this.streamBufferMaxSize;
    }

    public void writeOnRetry(long len) throws IOException {
        if (len == 0L) {
            return;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Retrying write length {} for blockID {}", (Object)len, this.blockID);
        }
        Preconditions.checkArgument(len <= this.streamBufferMaxSize);
        int count = 0;
        while (len > 0L) {
            ChunkBuffer buffer = this.bufferPool.getBuffer(count);
            long writeLen = Math.min((long)buffer.position(), len);
            if (!buffer.hasRemaining()) {
                this.writeChunk(buffer);
            }
            len -= writeLen;
            ++count;
            this.writtenDataLength += writeLen;
            if (this.writtenDataLength % this.streamBufferFlushSize == 0L) {
                this.updateFlushLength();
                this.executePutBlock(false, false);
            }
            if (this.writtenDataLength != this.streamBufferMaxSize) continue;
            this.handleFullBuffer();
        }
    }

    private void handleFullBuffer() throws IOException {
        try {
            this.checkOpen();
            if (!this.commitWatcher.getFutureMap().isEmpty()) {
                this.waitOnFlushFutures();
            }
        }
        catch (InterruptedException | ExecutionException e) {
            this.setIoException(e);
            this.adjustBuffersOnException();
            throw this.getIoException();
        }
        this.watchForCommit(true);
    }

    private void adjustBuffersOnException() {
        this.commitWatcher.releaseBuffersOnException();
    }

    private void watchForCommit(boolean bufferFull) throws IOException {
        this.checkOpen();
        try {
            List<DatanodeDetails> dnList;
            XceiverClientReply reply;
            XceiverClientReply xceiverClientReply = reply = bufferFull ? this.commitWatcher.watchOnFirstIndex() : this.commitWatcher.watchOnLastIndex();
            if (reply != null && !(dnList = reply.getDatanodes()).isEmpty()) {
                Pipeline pipe = this.xceiverClient.getPipeline();
                LOG.warn("Failed to commit BlockId {} on {}. Failed nodes: {}", new Object[]{this.blockID, pipe, dnList});
                this.failedServers.addAll(dnList);
            }
        }
        catch (IOException ioe) {
            this.setIoException(ioe);
            throw this.getIoException();
        }
    }

    private CompletableFuture<ContainerProtos.ContainerCommandResponseProto> executePutBlock(boolean close, boolean force) throws IOException {
        CompletionStage flushFuture;
        List<ChunkBuffer> byteBufferList;
        this.checkOpen();
        long flushPos = this.totalDataFlushedLength;
        if (!force) {
            Preconditions.checkNotNull(this.bufferList);
            byteBufferList = this.bufferList;
            this.bufferList = null;
            Preconditions.checkNotNull(byteBufferList);
        } else {
            byteBufferList = null;
        }
        try {
            ContainerProtos.BlockData blockData = this.containerBlockData.build();
            XceiverClientReply asyncReply = ContainerProtocolCalls.putBlockAsync(this.xceiverClient, blockData, close);
            CompletableFuture<ContainerProtos.ContainerCommandResponseProto> future = asyncReply.getResponse();
            flushFuture = ((CompletableFuture)future.thenApplyAsync(e -> {
                try {
                    this.validateResponse((ContainerProtos.ContainerCommandResponseProto)e);
                }
                catch (IOException sce) {
                    throw new CompletionException(sce);
                }
                if (this.getIoException() == null && !force) {
                    BlockID responseBlockID = BlockID.getFromProtobuf(e.getPutBlock().getCommittedBlockLength().getBlockID());
                    Preconditions.checkState(this.blockID.get().getContainerBlockID().equals(responseBlockID.getContainerBlockID()));
                    this.blockID.set(responseBlockID);
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Adding index " + asyncReply.getLogIndex() + " commitMap size " + this.commitWatcher.getCommitInfoMapSize() + " flushLength " + flushPos + " numBuffers " + byteBufferList.size() + " blockID " + this.blockID + " bufferPool size" + this.bufferPool.getSize() + " currentBufferIndex " + this.bufferPool.getCurrentBufferIndex());
                    }
                    this.commitWatcher.updateCommitInfoMap(asyncReply.getLogIndex(), byteBufferList);
                }
                return e;
            }, (Executor)this.responseExecutor)).exceptionally(e -> {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("putBlock failed for blockID {} with exception {}", this.blockID, (Object)e.getLocalizedMessage());
                }
                CompletionException ce = new CompletionException((Throwable)e);
                this.setIoException(ce);
                throw ce;
            });
        }
        catch (IOException | InterruptedException | ExecutionException e2) {
            throw new IOException("Unexpected Storage Container Exception: " + e2.toString(), e2);
        }
        this.commitWatcher.getFutureMap().put(flushPos, (CompletableFuture<ContainerProtos.ContainerCommandResponseProto>)flushFuture);
        return flushFuture;
    }

    @Override
    public void flush() throws IOException {
        if (this.xceiverClientManager != null && this.xceiverClient != null && this.bufferPool != null && this.bufferPool.getSize() > 0) {
            try {
                this.handleFlush(false);
            }
            catch (InterruptedException | ExecutionException e) {
                this.setIoException(e);
                this.adjustBuffersOnException();
                throw this.getIoException();
            }
        }
    }

    private void writeChunk(ChunkBuffer buffer) throws IOException {
        if (this.bufferList == null) {
            this.bufferList = new ArrayList<ChunkBuffer>();
        }
        this.bufferList.add(buffer);
        this.writeChunkToContainer(buffer.duplicate(0, buffer.position()));
    }

    private void handleFlush(boolean close) throws IOException, InterruptedException, ExecutionException {
        this.checkOpen();
        if (this.totalDataFlushedLength < this.writtenDataLength) {
            ChunkBuffer currentBuffer = this.bufferPool.getCurrentBuffer();
            Preconditions.checkArgument(currentBuffer.position() > 0);
            if (currentBuffer.hasRemaining()) {
                this.writeChunk(currentBuffer);
            }
            this.updateFlushLength();
            this.executePutBlock(close, false);
        } else if (close) {
            this.executePutBlock(true, true);
        }
        this.waitOnFlushFutures();
        this.watchForCommit(false);
        this.checkOpen();
    }

    @Override
    public void close() throws IOException {
        if (this.xceiverClientManager != null && this.xceiverClient != null && this.bufferPool != null && this.bufferPool.getSize() > 0) {
            try {
                this.handleFlush(true);
            }
            catch (InterruptedException | ExecutionException e) {
                this.setIoException(e);
                this.adjustBuffersOnException();
                throw this.getIoException();
            }
            finally {
                this.cleanup(false);
            }
        }
    }

    private void waitOnFlushFutures() throws InterruptedException, ExecutionException {
        CompletableFuture<Void> combinedFuture = CompletableFuture.allOf(this.commitWatcher.getFutureMap().values().toArray(new CompletableFuture[this.commitWatcher.getFutureMap().size()]));
        combinedFuture.get();
    }

    private void validateResponse(ContainerProtos.ContainerCommandResponseProto responseProto) throws IOException {
        try {
            IOException exception = this.getIoException();
            if (exception != null) {
                throw exception;
            }
            ContainerProtocolCalls.validateContainerResponse(responseProto);
        }
        catch (StorageContainerException sce) {
            this.setIoException(sce);
            throw sce;
        }
    }

    private void setIoException(Exception e) {
        IOException ioe = this.getIoException();
        if (ioe == null) {
            IOException exception = new IOException("Unexpected Storage Container Exception: " + e.toString(), e);
            this.ioException.compareAndSet(null, exception);
        } else {
            LOG.debug("Previous request had already failed with " + ioe.toString() + " so subsequent request also encounters" + " Storage Container Exception ", (Throwable)e);
        }
    }

    public void cleanup(boolean invalidateClient) {
        if (this.xceiverClientManager != null) {
            this.xceiverClientManager.releaseClient(this.xceiverClient, invalidateClient);
        }
        this.xceiverClientManager = null;
        this.xceiverClient = null;
        this.commitWatcher.cleanup();
        if (this.bufferList != null) {
            this.bufferList.clear();
        }
        this.bufferList = null;
        this.responseExecutor.shutdown();
    }

    private void checkOpen() throws IOException {
        if (this.isClosed()) {
            throw new IOException("BlockOutputStream has been closed.");
        }
        if (this.getIoException() != null) {
            this.adjustBuffersOnException();
            throw this.getIoException();
        }
    }

    public boolean isClosed() {
        return this.xceiverClient == null;
    }

    private void writeChunkToContainer(ChunkBuffer chunk) throws IOException {
        int effectiveChunkSize = chunk.remaining();
        long offset = this.chunkOffset.getAndAdd(effectiveChunkSize);
        ByteString data = chunk.toByteString(this.bufferPool.byteStringConversion());
        ChecksumData checksumData = this.checksum.computeChecksum(chunk);
        ContainerProtos.ChunkInfo chunkInfo = ContainerProtos.ChunkInfo.newBuilder().setChunkName(this.blockID.get().getLocalID() + "_chunk_" + ++this.chunkIndex).setOffset(offset).setLen(effectiveChunkSize).setChecksumData(checksumData.getProtoBufMessage()).build();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Writing chunk {} length {} at offset {}", new Object[]{chunkInfo.getChunkName(), effectiveChunkSize, offset});
        }
        try {
            XceiverClientReply asyncReply = ContainerProtocolCalls.writeChunkAsync(this.xceiverClient, chunkInfo, this.blockID.get(), data);
            CompletableFuture<ContainerProtos.ContainerCommandResponseProto> future = asyncReply.getResponse();
            ((CompletableFuture)future.thenApplyAsync(e -> {
                try {
                    this.validateResponse((ContainerProtos.ContainerCommandResponseProto)e);
                }
                catch (IOException sce) {
                    future.completeExceptionally(sce);
                }
                return e;
            }, (Executor)this.responseExecutor)).exceptionally(e -> {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("writing chunk failed " + chunkInfo.getChunkName() + " blockID " + this.blockID + " with exception " + e.getLocalizedMessage());
                }
                CompletionException ce = new CompletionException((Throwable)e);
                this.setIoException(ce);
                throw ce;
            });
        }
        catch (IOException | InterruptedException | ExecutionException e2) {
            throw new IOException("Unexpected Storage Container Exception: " + e2.toString(), e2);
        }
        this.containerBlockData.addChunks(chunkInfo);
    }

    @VisibleForTesting
    public void setXceiverClient(XceiverClientSpi xceiverClient) {
        this.xceiverClient = xceiverClient;
    }
}

