/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;
import java.nio.file.Path;
import java.util.Calendar;
import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.TimeZone;
import javax.management.ObjectName;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdds.annotation.InterfaceAudience;
import org.apache.hadoop.hdds.annotation.InterfaceStability;
import org.apache.hadoop.hdds.client.BlockID;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.metrics2.util.MBeans;
import org.apache.hadoop.net.DNS;
import org.apache.hadoop.net.NetUtils;
import org.apache.hadoop.ozone.shaded.com.google.common.base.Preconditions;
import org.apache.hadoop.ozone.shaded.com.google.common.net.HostAndPort;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Stable
public final class HddsUtils {
    private static final Logger LOG = LoggerFactory.getLogger(HddsUtils.class);
    public static final String OZONE_SCM_SERVICE_ID = "OzoneScmService";
    public static final String OZONE_SCM_SERVICE_INSTANCE_ID = "OzoneScmServiceInstance";
    private static final TimeZone UTC_ZONE = TimeZone.getTimeZone("UTC");
    private static final String MULTIPLE_SCM_NOT_YET_SUPPORTED = "ozone.scm.names must contain a single hostname. Multiple SCM hosts are currently unsupported";
    private static final int NO_PORT = -1;

    private HddsUtils() {
    }

    public static InetSocketAddress getScmAddressForClients(Configuration conf) {
        Optional<String> host = HddsUtils.getHostNameFromConfigKeys(conf, "ozone.scm.client.address");
        if (!host.isPresent()) {
            host = Optional.of(HddsUtils.getSingleSCMAddress(conf).getHostName());
        }
        int port = HddsUtils.getPortNumberFromConfigKeys(conf, "ozone.scm.client.address").orElse(9860);
        return NetUtils.createSocketAddr((String)(host.get() + ":" + port));
    }

    public static InetSocketAddress getScmAddressForBlockClients(Configuration conf) {
        Optional<String> host = HddsUtils.getHostNameFromConfigKeys(conf, "ozone.scm.block.client.address", "ozone.scm.client.address");
        if (!host.isPresent()) {
            host = Optional.of(HddsUtils.getSingleSCMAddress(conf).getHostName());
        }
        int port = HddsUtils.getPortNumberFromConfigKeys(conf, "ozone.scm.block.client.address").orElse(9863);
        return NetUtils.createSocketAddr((String)(host.get() + ":" + port));
    }

    public static Optional<String> getHostNameFromConfigKeys(Configuration conf, String ... keys) {
        for (String key : keys) {
            String value = conf.getTrimmed(key);
            Optional<String> hostName = HddsUtils.getHostName(value);
            if (!hostName.isPresent()) continue;
            return hostName;
        }
        return Optional.empty();
    }

    public static Optional<String> getHostName(String value) {
        if (value == null || value.isEmpty()) {
            return Optional.empty();
        }
        String hostname = value.replaceAll("\\:[0-9]+$", "");
        if (hostname.length() == 0) {
            return Optional.empty();
        }
        return Optional.of(hostname);
    }

    public static OptionalInt getHostPort(String value) {
        if (value == null || value.isEmpty()) {
            return OptionalInt.empty();
        }
        int port = HostAndPort.fromString(value).getPortOrDefault(-1);
        if (port == -1) {
            return OptionalInt.empty();
        }
        return OptionalInt.of(port);
    }

    public static OptionalInt getPortNumberFromConfigKeys(Configuration conf, String ... keys) {
        for (String key : keys) {
            String value = conf.getTrimmed(key);
            OptionalInt hostPort = HddsUtils.getHostPort(value);
            if (!hostPort.isPresent()) continue;
            return hostPort;
        }
        return OptionalInt.empty();
    }

    public static Collection<InetSocketAddress> getSCMAddresses(Configuration conf) {
        Collection names = conf.getTrimmedStringCollection("ozone.scm.names");
        if (names.isEmpty()) {
            throw new IllegalArgumentException("ozone.scm.names need to be a set of valid DNS names or IP addresses. Empty address list found.");
        }
        HashSet<InetSocketAddress> addresses = new HashSet<InetSocketAddress>(names.size());
        for (String address : names) {
            Optional<String> hostname = HddsUtils.getHostName(address);
            if (!hostname.isPresent()) {
                throw new IllegalArgumentException("Invalid hostname for SCM: " + address);
            }
            int port = HddsUtils.getHostPort(address).orElse(9861);
            InetSocketAddress addr = NetUtils.createSocketAddr((String)hostname.get(), (int)port);
            addresses.add(addr);
        }
        return addresses;
    }

    public static InetSocketAddress getReconAddresses(Configuration conf) {
        String name = conf.get("ozone.recon.address");
        if (StringUtils.isEmpty((CharSequence)name)) {
            return null;
        }
        Optional<String> hostname = HddsUtils.getHostName(name);
        if (!hostname.isPresent()) {
            throw new IllegalArgumentException("Invalid hostname for Recon: " + name);
        }
        int port = HddsUtils.getHostPort(name).orElse(9891);
        return NetUtils.createSocketAddr((String)hostname.get(), (int)port);
    }

    public static InetSocketAddress getSingleSCMAddress(Configuration conf) {
        Collection<InetSocketAddress> singleton = HddsUtils.getSCMAddresses(conf);
        Preconditions.checkArgument(singleton.size() == 1, MULTIPLE_SCM_NOT_YET_SUPPORTED);
        return singleton.iterator().next();
    }

    public static String getHostName(Configuration conf) throws UnknownHostException {
        String name = conf.get("dfs.datanode.hostname");
        if (name == null) {
            String dnsInterface = conf.get("hadoop.security.dns.interface");
            String nameServer = conf.get("hadoop.security.dns.nameserver");
            boolean fallbackToHosts = false;
            if (dnsInterface == null) {
                dnsInterface = conf.get("dfs.datanode.dns.interface");
                dnsInterface = conf.get("dfs.datanode.dns.interface");
                nameServer = conf.get("dfs.datanode.dns.nameserver");
            } else {
                fallbackToHosts = true;
            }
            name = DNS.getDefaultHost((String)dnsInterface, (String)nameServer, (boolean)fallbackToHosts);
        }
        return name;
    }

    public static boolean isReadOnly(ContainerProtos.ContainerCommandRequestProto proto) {
        switch (proto.getCmdType()) {
            case ReadContainer: 
            case ReadChunk: 
            case ListBlock: 
            case GetBlock: 
            case GetSmallFile: 
            case ListContainer: 
            case ListChunk: 
            case GetCommittedBlockLength: {
                return true;
            }
        }
        return false;
    }

    public static boolean requireBlockToken(ContainerProtos.Type cmdType) {
        switch (cmdType) {
            case ReadChunk: 
            case GetBlock: 
            case GetSmallFile: 
            case WriteChunk: 
            case PutBlock: 
            case PutSmallFile: {
                return true;
            }
        }
        return false;
    }

    public static BlockID getBlockID(ContainerProtos.ContainerCommandRequestProto msg) {
        switch (msg.getCmdType()) {
            case ReadChunk: {
                if (msg.hasReadChunk()) {
                    return BlockID.getFromProtobuf(msg.getReadChunk().getBlockID());
                }
                return null;
            }
            case GetBlock: {
                if (msg.hasGetBlock()) {
                    return BlockID.getFromProtobuf(msg.getGetBlock().getBlockID());
                }
                return null;
            }
            case WriteChunk: {
                if (msg.hasWriteChunk()) {
                    return BlockID.getFromProtobuf(msg.getWriteChunk().getBlockID());
                }
                return null;
            }
            case PutBlock: {
                if (msg.hasPutBlock()) {
                    return BlockID.getFromProtobuf(msg.getPutBlock().getBlockData().getBlockID());
                }
                return null;
            }
            case PutSmallFile: {
                if (msg.hasPutSmallFile()) {
                    return BlockID.getFromProtobuf(msg.getPutSmallFile().getBlock().getBlockData().getBlockID());
                }
                return null;
            }
            case GetSmallFile: {
                if (msg.hasGetSmallFile()) {
                    return BlockID.getFromProtobuf(msg.getGetSmallFile().getBlock().getBlockID());
                }
                return null;
            }
        }
        return null;
    }

    public static ObjectName registerWithJmxProperties(String serviceName, String mBeanName, Map<String, String> jmxProperties, Object mBean) {
        try {
            Method registerMethod = MBeans.class.getMethod("register", String.class, String.class, Map.class, Object.class);
            return (ObjectName)registerMethod.invoke(null, serviceName, mBeanName, jmxProperties, mBean);
        }
        catch (IllegalAccessException | NoSuchMethodException | InvocationTargetException e) {
            if (LOG.isTraceEnabled()) {
                LOG.trace("Registering MBean {} without additional properties {}", (Object)mBeanName, jmxProperties);
            }
            return MBeans.register((String)serviceName, (String)mBeanName, (Object)mBean);
        }
    }

    public static long getUtcTime() {
        return Calendar.getInstance(UTC_ZONE).getTimeInMillis();
    }

    public static void validatePath(Path path, Path ancestor) {
        Preconditions.checkNotNull(path, "Path should not be null");
        Preconditions.checkNotNull(ancestor, "Ancestor should not be null");
        Preconditions.checkArgument(path.normalize().startsWith(ancestor.normalize()), "Path should be a descendant of %s", (Object)ancestor);
    }

    static String getPassword(Configuration conf, String alias) {
        String password = null;
        try {
            char[] passchars = conf.getPassword(alias);
            if (passchars != null) {
                password = new String(passchars);
            }
        }
        catch (IOException ioe) {
            LOG.warn("Setting password to null since IOException is caught when getting password", (Throwable)ioe);
            password = null;
        }
        return password;
    }
}

