/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.ozone;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BlockLocation;
import org.apache.hadoop.fs.CreateFlag;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileAlreadyExistsException;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathIsNotEmptyDirectoryException;
import org.apache.hadoop.fs.ozone.BasicKeyInfo;
import org.apache.hadoop.fs.ozone.BasicOzoneClientAdapterImpl;
import org.apache.hadoop.fs.ozone.FileStatusAdapter;
import org.apache.hadoop.fs.ozone.OzoneClientAdapter;
import org.apache.hadoop.fs.ozone.OzoneClientAdapterFactory;
import org.apache.hadoop.fs.ozone.OzoneFSInputStream;
import org.apache.hadoop.fs.ozone.Statistic;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hdds.annotation.InterfaceAudience;
import org.apache.hadoop.hdds.annotation.InterfaceStability;
import org.apache.hadoop.ozone.om.exceptions.OMException;
import org.apache.hadoop.ozone.shaded.com.google.common.base.Preconditions;
import org.apache.hadoop.ozone.shaded.org.apache.http.client.utils.URIBuilder;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.util.Progressable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public class BasicOzoneFileSystem
extends FileSystem {
    static final Logger LOG = LoggerFactory.getLogger(BasicOzoneFileSystem.class);
    private URI uri;
    private String userName;
    private Path workingDir;
    private OzoneClientAdapter adapter;
    private static final Pattern URL_SCHEMA_PATTERN = Pattern.compile("([^\\.]+)\\.([^\\.]+)\\.{0,1}(.*)");
    private static final String URI_EXCEPTION_TEXT = "Ozone file system URL should be one of the following formats: o3fs://bucket.volume/key  OR o3fs://bucket.volume.om-host.example.com/key  OR o3fs://bucket.volume.om-host.example.com:5678/key";

    public void initialize(URI name, Configuration conf) throws IOException {
        super.initialize(name, conf);
        this.setConf(conf);
        Objects.requireNonNull(name.getScheme(), "No scheme provided in " + name);
        Preconditions.checkArgument(this.getScheme().equals(name.getScheme()), "Invalid scheme provided in " + name);
        String authority = name.getAuthority();
        if (authority == null) {
            throw new IllegalArgumentException(URI_EXCEPTION_TEXT);
        }
        Matcher matcher = URL_SCHEMA_PATTERN.matcher(authority);
        if (!matcher.matches()) {
            throw new IllegalArgumentException(URI_EXCEPTION_TEXT);
        }
        String bucketStr = matcher.group(1);
        String volumeStr = matcher.group(2);
        String remaining = matcher.groupCount() == 3 ? matcher.group(3) : null;
        String omHost = null;
        int omPort = -1;
        if (!this.isEmpty(remaining)) {
            String[] parts = remaining.split(":");
            if (parts.length > 2) {
                throw new IllegalArgumentException(URI_EXCEPTION_TEXT);
            }
            omHost = parts[0];
            if (parts.length == 2) {
                try {
                    omPort = Integer.parseInt(parts[1]);
                }
                catch (NumberFormatException e) {
                    throw new IllegalArgumentException(URI_EXCEPTION_TEXT);
                }
            }
        }
        try {
            this.uri = new URIBuilder().setScheme("o3fs").setHost(authority).build();
            LOG.trace("Ozone URI for ozfs initialization is {}", (Object)this.uri);
            boolean defaultValue = BasicOzoneFileSystem.class.getClassLoader().getResource("ozonefs.txt") != null;
            boolean isolatedClassloader = conf.getBoolean("ozone.fs.isolated-classloader", defaultValue);
            this.adapter = this.createAdapter(conf, bucketStr, volumeStr, omHost, omPort, isolatedClassloader);
            try {
                this.userName = UserGroupInformation.getCurrentUser().getShortUserName();
            }
            catch (IOException e) {
                this.userName = "hdfs";
            }
            this.workingDir = new Path("/user", this.userName).makeQualified(this.uri, this.workingDir);
        }
        catch (URISyntaxException ue) {
            String msg = "Invalid Ozone endpoint " + name;
            LOG.error(msg, (Throwable)ue);
            throw new IOException(msg, ue);
        }
    }

    protected OzoneClientAdapter createAdapter(Configuration conf, String bucketStr, String volumeStr, String omHost, int omPort, boolean isolatedClassloader) throws IOException {
        if (isolatedClassloader) {
            return OzoneClientAdapterFactory.createAdapter(volumeStr, bucketStr);
        }
        return new BasicOzoneClientAdapterImpl(omHost, omPort, conf, volumeStr, bucketStr);
    }

    public void close() throws IOException {
        try {
            this.adapter.close();
        }
        finally {
            super.close();
        }
    }

    public URI getUri() {
        return this.uri;
    }

    public String getScheme() {
        return "o3fs";
    }

    public FSDataInputStream open(Path f, int bufferSize) throws IOException {
        this.incrementCounter(Statistic.INVOCATION_OPEN);
        this.statistics.incrementReadOps(1);
        LOG.trace("open() path:{}", (Object)f);
        String key = this.pathToKey(f);
        InputStream inputStream = this.adapter.readFile(key);
        return new FSDataInputStream(this.createFSInputStream(inputStream));
    }

    protected InputStream createFSInputStream(InputStream inputStream) {
        return new OzoneFSInputStream(inputStream, this.statistics);
    }

    protected void incrementCounter(Statistic statistic) {
    }

    public FSDataOutputStream create(Path f, FsPermission permission, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        LOG.trace("create() path:{}", (Object)f);
        this.incrementCounter(Statistic.INVOCATION_CREATE);
        this.statistics.incrementWriteOps(1);
        String key = this.pathToKey(f);
        return this.createOutputStream(key, replication, overwrite, true);
    }

    public FSDataOutputStream createNonRecursive(Path path, FsPermission permission, EnumSet<CreateFlag> flags, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        this.incrementCounter(Statistic.INVOCATION_CREATE_NON_RECURSIVE);
        this.statistics.incrementWriteOps(1);
        String key = this.pathToKey(path);
        return this.createOutputStream(key, replication, flags.contains(CreateFlag.OVERWRITE), false);
    }

    private FSDataOutputStream createOutputStream(String key, short replication, boolean overwrite, boolean recursive) throws IOException {
        return new FSDataOutputStream((OutputStream)this.adapter.createFile(key, replication, overwrite, recursive), this.statistics);
    }

    public FSDataOutputStream append(Path f, int bufferSize, Progressable progress) throws IOException {
        throw new UnsupportedOperationException("append() Not implemented by the " + ((Object)((Object)this)).getClass().getSimpleName() + " FileSystem implementation");
    }

    public boolean rename(Path src, Path dst) throws IOException {
        FileStatus dstStatus;
        FileStatus srcStatus;
        this.incrementCounter(Statistic.INVOCATION_RENAME);
        this.statistics.incrementWriteOps(1);
        super.checkPath(src);
        super.checkPath(dst);
        String srcPath = src.toUri().getPath();
        String dstPath = dst.toUri().getPath();
        if (srcPath.equals(dstPath)) {
            return true;
        }
        LOG.trace("rename() from:{} to:{}", (Object)src, (Object)dst);
        if (src.isRoot()) {
            LOG.trace("Cannot rename the root of a filesystem");
            return false;
        }
        try {
            srcStatus = this.getFileStatus(src);
        }
        catch (FileNotFoundException fnfe) {
            return false;
        }
        if (srcStatus.isDirectory()) {
            Path dstParent;
            for (dstParent = dst.getParent(); dstParent != null && !src.equals((Object)dstParent); dstParent = dstParent.getParent()) {
            }
            Preconditions.checkArgument(dstParent == null, "Cannot rename a directory to its own subdirectory");
        }
        try {
            dstStatus = this.getFileStatus(dst);
        }
        catch (FileNotFoundException fnde) {
            dstStatus = null;
        }
        if (dstStatus == null) {
            dstStatus = this.getFileStatus(dst.getParent());
            if (!dstStatus.isDirectory()) {
                throw new IOException(String.format("Failed to rename %s to %s, %s is a file", src, dst, dst.getParent()));
            }
        } else {
            if (srcStatus.getPath().equals((Object)dstStatus.getPath())) {
                return !srcStatus.isDirectory();
            }
            if (dstStatus.isDirectory()) {
                FileStatus[] statuses;
                dst = new Path(dst, src.getName());
                dstPath = dst.toUri().getPath();
                try {
                    statuses = this.listStatus(dst);
                }
                catch (FileNotFoundException fnde) {
                    statuses = null;
                }
                if (statuses != null && statuses.length > 0) {
                    throw new FileAlreadyExistsException(String.format("Failed to rename %s to %s, file already exists or not empty!", src, dst));
                }
            } else {
                throw new FileAlreadyExistsException(String.format("Failed to rename %s to %s, file already exists!", src, dst));
            }
        }
        if (srcStatus.isDirectory() && dstPath.toString().startsWith(srcPath.toString() + "/")) {
            LOG.trace("Cannot rename a directory to a subdirectory of self");
            return false;
        }
        RenameIterator iterator = new RenameIterator(src, dst);
        boolean result = iterator.iterate();
        if (result) {
            this.createFakeParentDirectory(src);
        }
        return result;
    }

    private boolean innerDelete(Path f, boolean recursive) throws IOException {
        LOG.trace("delete() path:{} recursive:{}", (Object)f, (Object)recursive);
        try {
            DeleteIterator iterator = new DeleteIterator(f, recursive);
            if (f.isRoot()) {
                LOG.warn("Cannot delete root directory.");
                return false;
            }
            return iterator.iterate();
        }
        catch (FileNotFoundException e) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Couldn't delete {} - does not exist", (Object)f);
            }
            return false;
        }
    }

    public boolean delete(Path f, boolean recursive) throws IOException {
        boolean result;
        FileStatus status;
        this.incrementCounter(Statistic.INVOCATION_DELETE);
        this.statistics.incrementWriteOps(1);
        LOG.debug("Delete path {} - recursive {}", (Object)f, (Object)recursive);
        try {
            status = this.getFileStatus(f);
        }
        catch (FileNotFoundException ex) {
            LOG.warn("delete: Path does not exist: {}", (Object)f);
            return false;
        }
        String key = this.pathToKey(f);
        if (status.isDirectory()) {
            LOG.debug("delete: Path is a directory: {}", (Object)f);
            result = this.innerDelete(f, recursive);
        } else {
            LOG.debug("delete: Path is a file: {}", (Object)f);
            result = this.adapter.deleteObject(key);
        }
        if (result) {
            this.createFakeParentDirectory(f);
        }
        return result;
    }

    private void createFakeParentDirectory(Path f) throws IOException {
        Path parent = f.getParent();
        if (parent != null && !parent.isRoot()) {
            this.createFakeDirectoryIfNecessary(parent);
        }
    }

    private void createFakeDirectoryIfNecessary(Path f) throws IOException {
        String key = this.pathToKey(f);
        if (!key.isEmpty() && !this.o3Exists(f)) {
            LOG.debug("Creating new fake directory at {}", (Object)f);
            String dirKey = this.addTrailingSlashIfNeeded(key);
            this.adapter.createDirectory(dirKey);
        }
    }

    private boolean o3Exists(Path f) throws IOException {
        Path path = this.makeQualified(f);
        try {
            this.getFileStatus(path);
            return true;
        }
        catch (FileNotFoundException ex) {
            return false;
        }
    }

    public FileStatus[] listStatus(Path f) throws IOException {
        List tmpStatusList;
        this.incrementCounter(Statistic.INVOCATION_LIST_STATUS);
        this.statistics.incrementReadOps(1);
        LOG.trace("listStatus() path:{}", (Object)f);
        int numEntries = 1024;
        LinkedList statuses = new LinkedList();
        String startKey = "";
        do {
            if ((tmpStatusList = this.adapter.listStatus(this.pathToKey(f), false, startKey, numEntries, this.uri, this.workingDir, this.getUsername()).stream().map(this::convertFileStatus).collect(Collectors.toList())).isEmpty()) continue;
            if (startKey.isEmpty()) {
                statuses.addAll(tmpStatusList);
            } else {
                statuses.addAll(tmpStatusList.subList(1, tmpStatusList.size()));
            }
            startKey = this.pathToKey(((FileStatus)statuses.getLast()).getPath());
        } while (tmpStatusList.size() == numEntries);
        return statuses.toArray(new FileStatus[0]);
    }

    public void setWorkingDirectory(Path newDir) {
        this.workingDir = newDir;
    }

    public Path getWorkingDirectory() {
        return this.workingDir;
    }

    public Token<?> getDelegationToken(String renewer) throws IOException {
        return this.adapter.getDelegationToken(renewer);
    }

    public String getCanonicalServiceName() {
        return this.adapter.getCanonicalServiceName();
    }

    public String getUsername() {
        return this.userName;
    }

    private boolean mkdir(Path path) throws IOException {
        return this.adapter.createDirectory(this.pathToKey(path));
    }

    public boolean mkdirs(Path f, FsPermission permission) throws IOException {
        LOG.trace("mkdir() path:{} ", (Object)f);
        String key = this.pathToKey(f);
        if (this.isEmpty(key)) {
            return false;
        }
        return this.mkdir(f);
    }

    public FileStatus getFileStatus(Path f) throws IOException {
        FileStatus fileStatus;
        block2: {
            this.incrementCounter(Statistic.INVOCATION_GET_FILE_STATUS);
            this.statistics.incrementReadOps(1);
            LOG.trace("getFileStatus() path:{}", (Object)f);
            Path qualifiedPath = f.makeQualified(this.uri, this.workingDir);
            String key = this.pathToKey(qualifiedPath);
            fileStatus = null;
            try {
                fileStatus = this.convertFileStatus(this.adapter.getFileStatus(key, this.uri, qualifiedPath, this.getUsername()));
            }
            catch (OMException ex) {
                if (!ex.getResult().equals((Object)OMException.ResultCodes.KEY_NOT_FOUND)) break block2;
                throw new FileNotFoundException("File not found. path:" + f);
            }
        }
        return fileStatus;
    }

    public BlockLocation[] getFileBlockLocations(FileStatus fileStatus, long start, long len) throws IOException {
        if (fileStatus instanceof LocatedFileStatus) {
            return ((LocatedFileStatus)fileStatus).getBlockLocations();
        }
        return super.getFileBlockLocations(fileStatus, start, len);
    }

    public short getDefaultReplication() {
        return this.adapter.getDefaultReplication();
    }

    public String pathToKey(Path path) {
        Objects.requireNonNull(path, "Path can not be null!");
        if (!path.isAbsolute()) {
            path = new Path(this.workingDir, path);
        }
        String key = path.toUri().getPath().substring(1);
        LOG.trace("path for key:{} is:{}", (Object)key, (Object)path);
        return key;
    }

    private String addTrailingSlashIfNeeded(String key) {
        if (!this.isEmpty(key) && !key.endsWith("/")) {
            return key + "/";
        }
        return key;
    }

    public String toString() {
        return "OzoneFileSystem{URI=" + this.uri + ", " + "workingDir=" + this.workingDir + ", " + "userName=" + this.userName + ", " + "statistics=" + this.statistics + "}";
    }

    public OzoneClientAdapter getAdapter() {
        return this.adapter;
    }

    public boolean isEmpty(CharSequence cs) {
        return cs == null || cs.length() == 0;
    }

    public boolean isNumber(String number) {
        try {
            Integer.parseInt(number);
        }
        catch (NumberFormatException ex) {
            return false;
        }
        return true;
    }

    private FileStatus convertFileStatus(FileStatusAdapter fileStatusAdapter) {
        Path symLink = null;
        try {
            fileStatusAdapter.getSymlink();
        }
        catch (Exception exception) {
            // empty catch block
        }
        FileStatus fileStatus = new FileStatus(fileStatusAdapter.getLength(), fileStatusAdapter.isDir(), (int)fileStatusAdapter.getBlockReplication(), fileStatusAdapter.getBlocksize(), fileStatusAdapter.getModificationTime(), fileStatusAdapter.getAccessTime(), new FsPermission(fileStatusAdapter.getPermission()), fileStatusAdapter.getOwner(), fileStatusAdapter.getGroup(), symLink, fileStatusAdapter.getPath());
        BlockLocation[] blockLocations = fileStatusAdapter.getBlockLocations();
        if (blockLocations == null || blockLocations.length == 0) {
            return fileStatus;
        }
        return new LocatedFileStatus(fileStatus, blockLocations);
    }

    private abstract class OzoneListingIterator {
        private final Path path;
        private final FileStatus status;
        private String pathKey;
        private Iterator<BasicKeyInfo> keyIterator;

        OzoneListingIterator(Path path) throws IOException {
            this.path = path;
            this.status = BasicOzoneFileSystem.this.getFileStatus(path);
            this.pathKey = BasicOzoneFileSystem.this.pathToKey(path);
            if (this.status.isDirectory()) {
                this.pathKey = BasicOzoneFileSystem.this.addTrailingSlashIfNeeded(this.pathKey);
            }
            this.keyIterator = BasicOzoneFileSystem.this.adapter.listKeys(this.pathKey);
        }

        abstract boolean processKey(String var1) throws IOException;

        boolean iterate() throws IOException {
            LOG.trace("Iterating path {}", (Object)this.path);
            if (this.status.isDirectory()) {
                LOG.trace("Iterating directory:{}", (Object)this.pathKey);
                while (this.keyIterator.hasNext()) {
                    BasicKeyInfo key = this.keyIterator.next();
                    LOG.trace("iterating key:{}", (Object)key.getName());
                    if (this.processKey(key.getName())) continue;
                    return false;
                }
                return true;
            }
            LOG.trace("iterating file:{}", (Object)this.path);
            return this.processKey(this.pathKey);
        }

        String getPathKey() {
            return this.pathKey;
        }

        boolean pathIsDirectory() {
            return this.status.isDirectory();
        }

        FileStatus getStatus() {
            return this.status;
        }
    }

    private class DeleteIterator
    extends OzoneListingIterator {
        private boolean recursive;

        DeleteIterator(Path f, boolean recursive) throws IOException {
            super(f);
            this.recursive = recursive;
            if (this.getStatus().isDirectory() && !this.recursive && BasicOzoneFileSystem.this.listStatus(f).length != 0) {
                throw new PathIsNotEmptyDirectoryException(f.toString());
            }
        }

        @Override
        boolean processKey(String key) throws IOException {
            if (key.equals("")) {
                LOG.trace("Skipping deleting root directory");
                return true;
            }
            LOG.trace("deleting key:{}", (Object)key);
            boolean succeed = BasicOzoneFileSystem.this.adapter.deleteObject(key);
            return this.recursive || succeed;
        }
    }

    private class RenameIterator
    extends OzoneListingIterator {
        private final String srcKey;
        private final String dstKey;

        RenameIterator(Path srcPath, Path dstPath) throws IOException {
            super(srcPath);
            this.srcKey = BasicOzoneFileSystem.this.pathToKey(srcPath);
            this.dstKey = BasicOzoneFileSystem.this.pathToKey(dstPath);
            LOG.trace("rename from:{} to:{}", (Object)this.srcKey, (Object)this.dstKey);
        }

        @Override
        boolean processKey(String key) throws IOException {
            String newKeyName = this.dstKey.concat(key.substring(this.srcKey.length()));
            BasicOzoneFileSystem.this.adapter.renameKey(key, newKeyName);
            return true;
        }
    }
}

