/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hive.ql.exec.tez;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicReference;

import javax.security.auth.login.LoginException;

import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.hive.conf.HiveConf.ConfVars;
import org.apache.hadoop.hive.llap.LlapUtil;
import org.apache.hadoop.hive.llap.coordinator.LlapCoordinator;
import org.apache.hadoop.hive.llap.impl.LlapProtocolClientImpl;
import org.apache.hadoop.hive.llap.security.LlapTokenClient;
import org.apache.hadoop.hive.llap.security.LlapTokenIdentifier;
import org.apache.hadoop.hive.llap.tez.LlapProtocolClientProxy;
import org.apache.hadoop.hive.llap.tezplugins.LlapContainerLauncher;
import org.apache.hadoop.hive.llap.tezplugins.LlapTaskCommunicator;
import org.apache.hadoop.hive.llap.tezplugins.LlapTaskSchedulerService;
import org.apache.hadoop.hive.ql.exec.Utilities;
import org.apache.hadoop.hive.ql.exec.tez.monitoring.TezJobMonitor;
import org.apache.hadoop.hive.ql.metadata.HiveException;
import org.apache.hadoop.hive.ql.session.KillQuery;
import org.apache.hadoop.hive.ql.session.SessionState;
import org.apache.hadoop.hive.ql.session.SessionState.LogHelper;
import org.apache.hadoop.hive.ql.wm.WmContext;
import org.apache.hadoop.hive.shims.Utils;
import org.apache.hadoop.mapred.JobContext;
import org.apache.hadoop.registry.client.api.RegistryOperations;
import org.apache.hadoop.security.Credentials;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.yarn.api.records.LocalResource;
import org.apache.hadoop.yarn.api.records.LocalResourceType;
import org.apache.hadoop.yarn.conf.YarnConfiguration;
import org.apache.tez.client.TezClient;
import org.apache.tez.common.TezUtils;
import org.apache.tez.common.security.TokenCache;
import org.apache.tez.dag.api.PreWarmVertex;
import org.apache.tez.dag.api.SessionNotRunning;
import org.apache.tez.dag.api.TezConfiguration;
import org.apache.tez.dag.api.TezException;
import org.apache.tez.dag.api.UserPayload;
import org.apache.tez.mapreduce.hadoop.DeprecatedKeys;
import org.apache.tez.mapreduce.hadoop.MRHelpers;
import org.apache.tez.mapreduce.hadoop.MRJobConfig;
import org.apache.tez.serviceplugins.api.ContainerLauncherDescriptor;
import org.apache.tez.serviceplugins.api.ServicePluginsDescriptor;
import org.apache.tez.serviceplugins.api.TaskCommunicatorDescriptor;
import org.apache.tez.serviceplugins.api.TaskSchedulerDescriptor;
import org.codehaus.jackson.annotate.JsonProperty;
import org.codehaus.jackson.map.annotate.JsonSerialize;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

/**
 * The basic implementation of TezSession.
 */
@JsonSerialize
public class TezSessionState implements TezSession {

  protected static final Logger LOG = LoggerFactory.getLogger(TezSessionState.class.getName());
  private static final String TEZ_DIR = "_tez_session_dir";
  public static final String LLAP_SERVICE = "LLAP";
  private static final String LLAP_SCHEDULER = LlapTaskSchedulerService.class.getName();
  private static final String LLAP_LAUNCHER = LlapContainerLauncher.class.getName();
  private static final String LLAP_TASK_COMMUNICATOR = LlapTaskCommunicator.class.getName();

  protected final HiveConf conf;
  private Path tezScratchDir;
  protected LocalResource appJarLr;
  private TezClient session;
  private Future<TezClient> sessionFuture;
  /** Console used for user feedback during async session opening. */
  private LogHelper console;
  @JsonProperty("sessionId")
  private String sessionId;
  protected final DagUtils utils;
  @JsonProperty("queueName")
  private String queueName;
  @JsonProperty("defaultQueue")
  private boolean defaultQueue = false;
  @JsonProperty("user")
  private String user;

  private AtomicReference<String> ownerThread = new AtomicReference<>(null);


  private HiveResources resources;
  @JsonProperty("doAsEnabled")
  private boolean doAsEnabled;
  private boolean isLegacyLlapMode;
  protected WmContext wmContext;
  protected KillQuery killQuery;

  private static final Cache<String, String> shaCache = CacheBuilder.newBuilder().maximumSize(100).build();
  /**
   * Constructor. We do not automatically connect, because we only want to
   * load tez classes when the user has tez installed.
   */
  public TezSessionState(DagUtils utils, HiveConf conf) {
    this.utils = utils;
    this.conf = conf;
  }

  @Override
  public String toString() {
    return "sessionId=" + sessionId + ", queueName=" + queueName + ", user=" + user
        + ", doAs=" + doAsEnabled + ", isOpen=" + isOpen() + ", isDefault=" + defaultQueue;
  }

  /**
   * Constructor. We do not automatically connect, because we only want to
   * load tez classes when the user has tez installed.
   */
  public TezSessionState(String sessionId, HiveConf conf) {
    this(DagUtils.getInstance(), conf);
    this.sessionId = sessionId;
  }

  public boolean isOpening() {
    if (session != null || sessionFuture == null) {
      return false;
    }
    try {
      TezClient session = sessionFuture.get(0, TimeUnit.NANOSECONDS);
      if (session == null) {
        return false;
      }
      this.session = session;
    } catch (InterruptedException e) {
      Thread.currentThread().interrupt();
      return false;
    } catch (ExecutionException e) {
      throw new RuntimeException(e);
    } catch (CancellationException e) {
      return false;
    } catch (TimeoutException e) {
      return true;
    }
    return false;
  }

  public boolean isOpen() {
    if (session != null) {
      return true;
    }
    if (sessionFuture == null) {
      return false;
    }
    try {
      TezClient session = sessionFuture.get(0, TimeUnit.NANOSECONDS);
      if (session == null) {
        return false;
      }
      this.session = session;
    } catch (InterruptedException e) {
      Thread.currentThread().interrupt();
      return false;
    } catch (ExecutionException e) {
      throw new RuntimeException(e);
    } catch (TimeoutException | CancellationException e) {
      return false;
    }
    return true;
  }


  /**
   * Get all open sessions. Only used to clean up at shutdown.
   * @return List<TezSessionState>
   */
  public static String makeSessionId() {
    return UUID.randomUUID().toString();
  }

  @Override
  public void open() throws IOException, LoginException, URISyntaxException, TezException {
    open(false);
  }

  public boolean reconnect(String applicationId, long amAgeMs)
      throws IOException, LoginException, URISyntaxException, TezException  {
    this.queueName = conf.get(TezConfiguration.TEZ_QUEUE_NAME);
    this.doAsEnabled = conf.getBoolVar(HiveConf.ConfVars.HIVE_SERVER2_ENABLE_DOAS);
    this.user = Utils.getUGI().getShortUserName();

    // TODO: we currently have to create the client as if it was new, to have the correct
    //       local object state. So, go thru most of the same things that open does.

    Path scratchDir = reconnectTezDir(sessionId, null);
    if (scratchDir == null) {
      LOG.warn("Cannot find the original session scratchDir, will create a new one");
      scratchDir = createTezDir(sessionId, null);
    }
    tezScratchDir = scratchDir;
    scratchDir = reconnectTezDir(sessionId, "resources");
    if (scratchDir == null) {
      LOG.warn("Cannot find the original resources scratchDir, will create a new one");
      scratchDir = createTezDir(sessionId, "resources");
    }
    this.resources = new HiveResources(scratchDir);
    LOG.info("Created new resources: " + resources);

    // We won't actually localize anything here; the files should already be there.
    // The NotFromConf files are unknown - if any are needed later and were localized by
    // the previous HS2, this one will try to localize them again and find they're already there.
    ensureLocalResources(conf, null);

    // exec and LLAP jars are definitely already localized... we just need to init the fields.
    // TODO: perhaps we could improve this method to share some fields when multiple
    //       sessions are being reconnected at the same time.

    final Map<String, LocalResource> commonLocalResources = addExecJarAndLocalResources();

    final boolean llapMode = addLlapJarsIfNeeded(commonLocalResources);

    // generate basic tez config
    final TezConfiguration tezConfig = createTezConfig();
    tezConfig.set(TezConfiguration.TEZ_AM_STAGING_DIR, tezScratchDir.toUri().toString());
    final Credentials llapCredentials = createLlapCredentials(llapMode, tezConfig);
    final ServicePluginsDescriptor spd = createServicePluginDescriptor(llapMode, tezConfig);

    // Do not initialize prewarm here. Either it's already prewarmed or we don't care (for now).

    TezClient session = createTezClientObject(
        tezConfig, commonLocalResources, llapCredentials, spd);

    // Note: if this ever moved on a threadpool, see isOnThread stuff
    //       in open() w.r.t. how to propagate errors correctly.
    // Can this ever return a different client? Not as of this writing. Hmm.
    session = session.getClient(applicationId);
    this.session = session;
    return true;
  }

  protected boolean isLlapMode() {
    return "llap".equalsIgnoreCase(HiveConf.getVar(
        conf, HiveConf.ConfVars.HIVE_EXECUTION_MODE));
  }

  protected final TezClient createTezClientObject(TezConfiguration tezConfig,
      Map<String, LocalResource> commonLocalResources,
      Credentials llapCredentials, ServicePluginsDescriptor spd) {
    return TezClient.newBuilder("HIVE-" + sessionId, tezConfig).setIsSession(true)
        .setLocalResources(commonLocalResources).setCredentials(llapCredentials)
        .setServicePluginDescriptor(spd).build();
  }

  /**
   * Creates a tez session. A session is tied to either a cli/hs2 session. You can
   * submit multiple DAGs against a session (as long as they are executed serially).
   */
  @Override
  public void open(String[] additionalFilesNotFromConf)
      throws IOException, LoginException, URISyntaxException, TezException {
    openInternal(additionalFilesNotFromConf, false, null, null, false);
  }


  @Override
  public void open(HiveResources resources)
      throws LoginException, IOException, URISyntaxException, TezException {
    openInternal(null, false, null, resources, false);
  }

  @Override
  public void open(boolean isPoolInit)
      throws LoginException, IOException, URISyntaxException, TezException {
    String[] noFiles = null;
    openInternal(noFiles, false, null, null, isPoolInit);
  }

  @Override
  public void beginOpen(String[] additionalFiles, LogHelper console)
      throws IOException, LoginException, URISyntaxException, TezException {
    openInternal(additionalFiles, true, console, null, false);
  }

  protected void openInternal(String[] additionalFilesNotFromConf,
      boolean isAsync, LogHelper console, HiveResources resources, boolean isPoolInit)
          throws IOException, LoginException, URISyntaxException, TezException {
    initQueueAndUser();

    // Create the tez tmp dir and a directory for Hive resources.
    tezScratchDir = createTezDir(sessionId, null);
    if (resources != null) {
      // If we are getting the resources externally, don't relocalize anything.
      this.resources = resources;
      LOG.info("Setting resources to " + resources);
    } else {
      this.resources = new HiveResources(createTezDir(sessionId, "resources"));
      ensureLocalResources(conf, additionalFilesNotFromConf);
      LOG.info("Created new resources: " + resources);
    }


    // configuration for the application master
    final Map<String, LocalResource> commonLocalResources = addExecJarAndLocalResources();
    final boolean llapMode = addLlapJarsIfNeeded(commonLocalResources);

    // Create environment for AM.
    // TODO: this is unused since AMConfiguration was replaced with TezConfiguration.
    Map<String, String> amEnv = new HashMap<String, String>();
    MRHelpers.updateEnvBasedOnMRAMEnv(conf, amEnv);

    // and finally we're ready to create and start the session
    // generate basic tez config
    final TezConfiguration tezConfig = createTezConfig();
    tezConfig.set(TezConfiguration.TEZ_AM_STAGING_DIR, tezScratchDir.toUri().toString());

    Credentials llapCredentials = createLlapCredentials(llapMode, tezConfig);
    ServicePluginsDescriptor spd = createServicePluginDescriptor(llapMode, tezConfig);

    // container prewarming. tell the am how many containers we need
    if (HiveConf.getBoolVar(conf, ConfVars.HIVE_PREWARM_ENABLED)) {
      int n = HiveConf.getIntVar(conf, ConfVars.HIVE_PREWARM_NUM_CONTAINERS);
      n = Math.max(tezConfig.getInt(
          TezConfiguration.TEZ_AM_SESSION_MIN_HELD_CONTAINERS,
          TezConfiguration.TEZ_AM_SESSION_MIN_HELD_CONTAINERS_DEFAULT), n);
      tezConfig.setInt(TezConfiguration.TEZ_AM_SESSION_MIN_HELD_CONTAINERS, n);
    }

    TezClient session = createTezClientObject(
        tezConfig, commonLocalResources, llapCredentials, spd);

    LOG.info("Opening new Tez Session (id: " + sessionId
        + ", scratch dir: " + tezScratchDir + ")");

    TezJobMonitor.initShutdownHook();
    if (!isAsync) {
      startSessionAndContainers(session, conf, commonLocalResources, tezConfig, false);
      setTezClient(session);
    } else {
      FutureTask<TezClient> sessionFuture = new FutureTask<>(new Callable<TezClient>() {
        @Override
        public TezClient call() throws Exception {
          TezClient result = null;
          try {
            result = startSessionAndContainers(
                session, conf, commonLocalResources, tezConfig, true);
          } catch (Throwable t) {
            // The caller has already stopped the session.
            LOG.error("Failed to start Tez session", t);
            throw (t instanceof Exception) ? (Exception) t : new Exception(t);
          }
          // Check interrupt at the last moment in case we get cancelled quickly.
          // This is not bulletproof but should allow us to close session in most cases.
          if (Thread.interrupted()) {
            LOG.info("Interrupted while starting Tez session");
            closeAndIgnoreExceptions(result);
            return null;
          }
          return result;
        }
      });
      new Thread(sessionFuture, "Tez session start thread").start();
      // We assume here nobody will try to get session before open() returns.
      this.console = console;
      this.sessionFuture = sessionFuture;
    }
  }

  private Map<String, LocalResource> addExecJarAndLocalResources()
      throws IOException, LoginException, URISyntaxException {
    final Map<String, LocalResource> commonLocalResources = new HashMap<String, LocalResource>();
    appJarLr = createJarLocalResource(utils.getExecJarPathLocal(conf));
    commonLocalResources.put(DagUtils.getBaseName(appJarLr), appJarLr);
    for (LocalResource lr : this.resources.localizedResources) {
      commonLocalResources.put(DagUtils.getBaseName(lr), lr);
    }
    return commonLocalResources;
  }

  protected static ServicePluginsDescriptor createServicePluginDescriptor(boolean llapMode,
      TezConfiguration tezConfig) throws IOException {
    if (!llapMode) return ServicePluginsDescriptor.create(true);
    // TODO Change this to not serialize the entire Configuration - minor.
    UserPayload servicePluginPayload = TezUtils.createUserPayloadFromConf(tezConfig);
    // we need plugins to handle llap and uber mode
    return ServicePluginsDescriptor.create(true,
        new TaskSchedulerDescriptor[] { TaskSchedulerDescriptor.create(
            LLAP_SERVICE, LLAP_SCHEDULER).setUserPayload(servicePluginPayload) },
        new ContainerLauncherDescriptor[] { ContainerLauncherDescriptor.create(
            LLAP_SERVICE, LLAP_LAUNCHER) },
        new TaskCommunicatorDescriptor[] { TaskCommunicatorDescriptor.create(
            LLAP_SERVICE, LLAP_TASK_COMMUNICATOR).setUserPayload(servicePluginPayload) });
  }

  protected final Credentials createLlapCredentials(boolean llapMode,
      TezConfiguration tezConfig) throws IOException {
    if (!isKerberosEnabled(tezConfig)) {
      return null;
    }
    Credentials llapCredentials = new Credentials();
    if (llapMode) {
      llapCredentials.addToken(LlapTokenIdentifier.KIND_NAME, getLlapToken(user, tezConfig));
    }
    String protoPath = conf.getVar(ConfVars.HIVE_PROTO_EVENTS_BASE_PATH);
    if (protoPath != null && !protoPath.isEmpty()) {
      TokenCache.obtainTokensForFileSystems(llapCredentials, new Path[] {new Path(protoPath)}, tezConfig);
    }
    return llapCredentials;
  }

  protected final TezConfiguration createTezConfig() throws IOException {
    TezConfiguration tezConfig = new TezConfiguration(true);
    tezConfig.addResource(conf);
    setupTezParamsBasedOnMR(tezConfig);
    conf.stripHiddenConfigurations(tezConfig);
    setupSessionAcls(tezConfig, conf);
    return tezConfig;
  }

  protected final boolean addLlapJarsIfNeeded(Map<String, LocalResource> commonLocalResources)
      throws IOException, LoginException {
    if (!isLlapMode()) {
      return false;
    }
    // localize llap client jars
    addJarLRByClass(LlapTaskSchedulerService.class, commonLocalResources);
    addJarLRByClass(LlapProtocolClientImpl.class, commonLocalResources);
    addJarLRByClass(LlapProtocolClientProxy.class, commonLocalResources);
    addJarLRByClass(RegistryOperations.class, commonLocalResources);
    return true;
  }

  protected final void initQueueAndUser() throws LoginException, IOException {
    // TODO Why is the queue name set again. It has already been setup via setQueueName. Do only one of the two.
    String confQueueName = conf.get(TezConfiguration.TEZ_QUEUE_NAME);
    if (queueName != null && !queueName.equals(confQueueName)) {
      LOG.warn("Resetting a queue name that was already set: was "
          + queueName + ", now " + confQueueName);
    }
    this.queueName = confQueueName;
    this.doAsEnabled = conf.getBoolVar(HiveConf.ConfVars.HIVE_SERVER2_ENABLE_DOAS);

    // TODO This - at least for the session pool - will always be the hive user. How does doAs above this affect things ?
    UserGroupInformation ugi = Utils.getUGI();
    user = ugi.getShortUserName();
    LOG.info("User of session id " + sessionId + " is " + user);
  }

  /**
   * Check if Kerberos authentication is enabled.
   * This is used by:
   * - HS2 (upon Tez session creation)
   * In secure scenarios HS2 might either be logged on (by Kerberos) by itself or by a launcher
   * script it was forked from. In the latter case UGI.getLoginUser().isFromKeytab() returns false,
   * hence UGI.getLoginUser().hasKerberosCredentials() is a tightest setting we can check against.
   */
  private boolean isKerberosEnabled(Configuration conf) {
    try {
      return UserGroupInformation.getLoginUser().hasKerberosCredentials() &&
          HiveConf.getBoolVar(conf, ConfVars.LLAP_USE_KERBEROS);
    } catch (IOException e) {
      return false;
    }
  }

  private static Token<LlapTokenIdentifier> getLlapToken(
      String user, final Configuration conf) throws IOException {
    // TODO: parts of this should be moved out of TezSession to reuse the clients, but there's
    //       no good place for that right now (HIVE-13698).
    // TODO: De-link from SessionState. A TezSession can be linked to different Hive Sessions via the pool.
    SessionState session = SessionState.get();
    boolean isInHs2 = session != null && session.isHiveServerQuery();
    Token<LlapTokenIdentifier> token = null;
    // For Tez, we don't use appId to distinguish the tokens.

    LlapCoordinator coordinator = null;
    if (isInHs2) {
      // We are in HS2, get the token locally.
      // TODO: coordinator should be passed in; HIVE-13698. Must be initialized for now.
      coordinator = LlapCoordinator.getInstance();
      if (coordinator == null) {
        throw new IOException("LLAP coordinator not initialized; cannot get LLAP tokens");
      }
      // Signing is not required for Tez.
      token = coordinator.getLocalTokenClient(conf, user).createToken(null, null, false);
    } else {
      // We are not in HS2; always create a new client for now.
      token = new LlapTokenClient(conf).getDelegationToken(null);
    }
    if (LOG.isInfoEnabled()) {
      LOG.info("Obtained a LLAP token: " + token);
    }
    return token;
  }

  private TezClient startSessionAndContainers(TezClient session, HiveConf conf,
      Map<String, LocalResource> commonLocalResources, TezConfiguration tezConfig,
      boolean isOnThread) throws TezException, IOException {
    session.start();
    boolean isSuccessful = false;
    try {
      if (HiveConf.getBoolVar(conf, ConfVars.HIVE_PREWARM_ENABLED)) {
        int n = HiveConf.getIntVar(conf, ConfVars.HIVE_PREWARM_NUM_CONTAINERS);
        LOG.info("Prewarming " + n + " containers  (id: " + sessionId
            + ", scratch dir: " + tezScratchDir + ")");
        PreWarmVertex prewarmVertex = utils.createPreWarmVertex(
            tezConfig, n, commonLocalResources);
        try {
          session.preWarm(prewarmVertex);
        } catch (IOException ie) {
          if (!isOnThread && ie.getMessage().contains("Interrupted while waiting")) {
            LOG.warn("Hive Prewarm threw an exception ", ie);
          } else {
            throw ie;
          }
        }
      }
      try {
        session.waitTillReady();
      } catch (InterruptedException ie) {
        if (isOnThread) {
          throw new IOException(ie);
          //ignore
        }
      }
      isSuccessful = true;
      // sessionState.getQueueName() comes from cluster wide configured queue names.
      // sessionState.getConf().get("tez.queue.name") is explicitly set by user in a session.
      // TezSessionPoolManager sets tez.queue.name if user has specified one or use the one from
      // cluster wide queue names.
      // There is no way to differentiate how this was set (user vs system).
      // Unset this after opening the session so that reopening of session uses the correct queue
      // names i.e, if client has not died and if the user has explicitly set a queue name
      // then reopened session will use user specified queue name else default cluster queue names.
      conf.unset(TezConfiguration.TEZ_QUEUE_NAME);
      return session;
    } finally {
      if (isOnThread && !isSuccessful) {
        closeAndIgnoreExceptions(session);
      }
    }
  }

  private static void closeAndIgnoreExceptions(TezClient session) {
    try {
      session.stop();
    } catch (SessionNotRunning nr) {
      // Ignore.
    } catch (IOException | TezException ex) {
      LOG.info("Failed to close Tez session after failure to initialize: " + ex.getMessage());
    }
  }

  public void endOpen() throws InterruptedException, CancellationException {
    if (this.session != null || this.sessionFuture == null) {
      return;
    }
    try {
      TezClient session = this.sessionFuture.get();
      if (session == null) {
        throw new RuntimeException("Initialization was interrupted");
      }
      setTezClient(session);
    } catch (ExecutionException e) {
      throw new RuntimeException(e);
    }
  }

  /**
   * This takes settings from MR and applies them to the appropriate Tez configuration. This is
   * similar to what Pig on Tez does (refer MRToTezHelper.java).
   *
   * @param conf configuration with MR settings
   */
  private void setupTezParamsBasedOnMR(TezConfiguration conf) {

    String env = conf.get(MRJobConfig.MR_AM_ADMIN_USER_ENV);
    if (conf.get(MRJobConfig.MR_AM_ENV) != null) {
      env = (env == null) ? conf.get(MRJobConfig.MR_AM_ENV) : env + "," + conf.get(MRJobConfig.MR_AM_ENV);
    }
    if (env != null) {
      conf.setIfUnset(TezConfiguration.TEZ_AM_LAUNCH_ENV, env);
    }

    conf.setIfUnset(TezConfiguration.TEZ_AM_LAUNCH_CMD_OPTS,
        org.apache.tez.mapreduce.hadoop.MRHelpers.getJavaOptsForMRAM(conf));

    String queueName = conf.get(JobContext.QUEUE_NAME, YarnConfiguration.DEFAULT_QUEUE_NAME);
    conf.setIfUnset(TezConfiguration.TEZ_QUEUE_NAME, queueName);

    int amMemMB = conf.getInt(MRJobConfig.MR_AM_VMEM_MB, MRJobConfig.DEFAULT_MR_AM_VMEM_MB);
    conf.setIfUnset(TezConfiguration.TEZ_AM_RESOURCE_MEMORY_MB, "" + amMemMB);

    int amCores = conf.getInt(MRJobConfig.MR_AM_CPU_VCORES, MRJobConfig.DEFAULT_MR_AM_CPU_VCORES);
    conf.setIfUnset(TezConfiguration.TEZ_AM_RESOURCE_CPU_VCORES, "" + amCores);

    conf.setIfUnset(TezConfiguration.TEZ_AM_MAX_APP_ATTEMPTS, ""
        + conf.getInt(MRJobConfig.MR_AM_MAX_ATTEMPTS, MRJobConfig.DEFAULT_MR_AM_MAX_ATTEMPTS));

    conf.setIfUnset(TezConfiguration.TEZ_AM_VIEW_ACLS,
        conf.get(MRJobConfig.JOB_ACL_VIEW_JOB, MRJobConfig.DEFAULT_JOB_ACL_VIEW_JOB));

    conf.setIfUnset(TezConfiguration.TEZ_AM_MODIFY_ACLS,
        conf.get(MRJobConfig.JOB_ACL_MODIFY_JOB, MRJobConfig.DEFAULT_JOB_ACL_MODIFY_JOB));


    // Refer to org.apache.tez.mapreduce.hadoop.MRHelpers.processDirectConversion.
    ArrayList<Map<String, String>> maps = new ArrayList<Map<String, String>>(2);
    maps.add(DeprecatedKeys.getMRToTezRuntimeParamMap());
    maps.add(DeprecatedKeys.getMRToDAGParamMap());

    boolean preferTez = true; // Can make this configurable.

    for (Map<String, String> map : maps) {
      for (Map.Entry<String, String> dep : map.entrySet()) {
        if (conf.get(dep.getKey()) != null) {
          // TODO Deprecation reason does not seem to reflect in the config ?
          // The ordering is important in case of keys which are also deprecated.
          // Unset will unset the deprecated keys and all its variants.
          final String mrValue = conf.get(dep.getKey());
          final String tezValue = conf.get(dep.getValue());
          conf.unset(dep.getKey());
          if (tezValue == null) {
            conf.set(dep.getValue(), mrValue, "TRANSLATED_TO_TEZ");
          } else if (!preferTez) {
            conf.set(dep.getValue(), mrValue, "TRANSLATED_TO_TEZ_AND_MR_OVERRIDE");
          }
          LOG.debug("Config: mr(unset):" + dep.getKey() + ", mr initial value="
              + mrValue
              + ", tez(original):" + dep.getValue() + "=" + tezValue
              + ", tez(final):" + dep.getValue() + "=" + conf.get(dep.getValue()));
        }
      }
    }
  }

  private void setupSessionAcls(Configuration tezConf, HiveConf hiveConf) throws
      IOException {

    // TODO: De-link from SessionState. A TezSession can be linked to different Hive Sessions via the pool.
    String user = SessionState.getUserFromAuthenticator();
    UserGroupInformation loginUserUgi = UserGroupInformation.getLoginUser();
    String loginUser =
        loginUserUgi == null ? null : loginUserUgi.getShortUserName();
    boolean addHs2User =
        HiveConf.getBoolVar(hiveConf, ConfVars.HIVETEZHS2USERACCESS);

    String viewStr = Utilities.getAclStringWithHiveModification(tezConf,
            TezConfiguration.TEZ_AM_VIEW_ACLS, addHs2User, user, loginUser);
    String modifyStr = Utilities.getAclStringWithHiveModification(tezConf,
            TezConfiguration.TEZ_AM_MODIFY_ACLS, addHs2User, user, loginUser);

    if (LOG.isDebugEnabled()) {
      // TODO: De-link from SessionState. A TezSession can be linked to different Hive Sessions via the pool.
      LOG.debug(
          "Setting Tez Session access for sessionId={} with viewAclString={}, modifyStr={}",
          SessionState.get().getSessionId(), viewStr, modifyStr);
    }

    tezConf.set(TezConfiguration.TEZ_AM_VIEW_ACLS, viewStr);
    tezConf.set(TezConfiguration.TEZ_AM_MODIFY_ACLS, modifyStr);
  }

  /** This is called in openInternal and in TezTask.updateSession to localize conf resources. */
  @Override
  public void ensureLocalResources(Configuration conf, String[] newFilesNotFromConf)
          throws IOException, LoginException, URISyntaxException, TezException {
    if (resources == null) {
      throw new AssertionError("Ensure called on an unitialized (or closed) session " + sessionId);
    }
    String dir = resources.dagResourcesDir.toString();
    resources.localizedResources.clear();

    // Always localize files from conf; duplicates are handled on FS level.
    // TODO: we could do the same thing as below and only localize if missing.
    //       That could be especially valuable given that this almost always the same set.
    List<LocalResource> lrs = utils.localizeTempFilesFromConf(dir, conf);
    if (lrs != null) {
      resources.localizedResources.addAll(lrs);
    }

    // Localize the non-conf resources that are missing from the current list.
    Map<String, LocalResource> newResources = null;
    if (newFilesNotFromConf != null && newFilesNotFromConf.length > 0) {
      boolean hasResources = !resources.additionalFilesNotFromConf.isEmpty();
      if (hasResources) {
        for (String s : newFilesNotFromConf) {
          hasResources = resources.additionalFilesNotFromConf.keySet().contains(s);
          if (!hasResources) {
            break;
          }
        }
      }
      if (!hasResources) {
        String[] skipFilesFromConf = DagUtils.getTempFilesFromConf(conf);
        newResources = utils.localizeTempFiles(dir, conf, newFilesNotFromConf, skipFilesFromConf);
        if (newResources != null) {
          resources.localizedResources.addAll(newResources.values());
          resources.additionalFilesNotFromConf.putAll(newResources);
        }
      } else {
        resources.localizedResources.addAll(resources.additionalFilesNotFromConf.values());
      }
    }

    // Finally, add the files to the existing AM (if any). The old code seems to do this twice,
    // first for all the new resources regardless of type; and then for all the session resources
    // that are not of type file (see branch-1 calls to addAppMasterLocalFiles: from updateSession
    // and with resourceMap from submit).
    // TODO: Do we really need all this nonsense?
    if (session != null) {
      if (newResources != null && !newResources.isEmpty()) {
        session.addAppMasterLocalFiles(DagUtils.createTezLrMap(null, newResources.values()));
      }
      if (!resources.localizedResources.isEmpty()) {
        session.addAppMasterLocalFiles(
            DagUtils.getResourcesUpdatableForAm(resources.localizedResources));
      }
    }
  }

  /**
   * Close a tez session. Will cleanup any tez/am related resources. After closing a session no
   * further DAGs can be executed against it. Only called by session management classes; some
   * sessions should not simply be closed by users - e.g. pool sessions need to be restarted.
   *
   * @param keepDagFilesDir
   *          whether or not to remove the scratch dir at the same time.
   * @throws Exception
   */
  @Override
  public void close(boolean keepDagFilesDir) throws Exception {
    console = null;
    appJarLr = null;

    try {
      if (session != null) {
        LOG.info("Closing Tez Session");
        closeClient(session);
        session = null;
      } else if (sessionFuture != null) {
        sessionFuture.cancel(true);
        TezClient asyncSession = null;
        try {
          asyncSession = sessionFuture.get(); // In case it was done and noone looked at it.
        } catch (ExecutionException | CancellationException e) {
          // ignore
        } catch (InterruptedException e) {
          Thread.currentThread().interrupt();
          // ignore
        }
        sessionFuture = null;
        if (asyncSession != null) {
          LOG.info("Closing Tez Session");
          closeClient(asyncSession);
        }
      }
    } finally {
      try {
        cleanupScratchDir();
      } finally {
        if (!keepDagFilesDir) {
          cleanupDagResources();
        }
      }
    }
  }

  private void closeClient(TezClient client) throws TezException,
      IOException {
    try {
      client.stop();
    } catch (SessionNotRunning nr) {
      // ignore
    }
  }

  protected final void cleanupScratchDir() throws IOException {
    if (tezScratchDir != null) {
      FileSystem fs = tezScratchDir.getFileSystem(conf);
      fs.delete(tezScratchDir, true);
      tezScratchDir = null;
    }
  }

  protected final void cleanupDagResources() throws IOException {
    LOG.info("Attemting to clean up resources for " + sessionId + ": " + resources);
    if (resources != null) {
      FileSystem fs = resources.dagResourcesDir.getFileSystem(conf);
      fs.delete(resources.dagResourcesDir, true);
      resources = null;
    }
  }

  @Override
  public String getSessionId() {
    return sessionId;
  }

  protected final void setTezClient(TezClient session) {
    this.session = session;
  }

  @Override
  public TezClient getTezClient() {
    if (session == null && sessionFuture != null) {
      if (!sessionFuture.isDone()) {
        console.printInfo("Waiting for Tez session and AM to be ready...");
      }
      try {
        session = sessionFuture.get();
      } catch (InterruptedException e) {
        console.printInfo("Interrupted while waiting for the session");
        Thread.currentThread().interrupt();
        return null;
      } catch (ExecutionException e) {
        console.printInfo("Failed to get session");
        throw new RuntimeException(e);
      } catch (CancellationException e) {
        console.printInfo("Cancelled while waiting for the session");
        return null;
      }
    }
    return session;
  }

  public LocalResource getAppJarLr() {
    return appJarLr;
  }

  /**
   * createTezDir creates a temporary directory in the scratchDir folder to
   * be used with Tez. Assumes scratchDir exists.
   */
  private Path createTezDir(String sessionId, String suffix) throws IOException {
    Path tezDir = getScratchDirPath(sessionId, suffix, false);
    FileSystem fs = tezDir.getFileSystem(conf);
    FsPermission fsPermission = new FsPermission(
        HiveConf.getVar(conf, HiveConf.ConfVars.SCRATCHDIRPERMISSION));
    fs.mkdirs(tezDir, fsPermission);
    // Make sure the path is normalized (we expect validation to pass since we just created it).
    tezDir = DagUtils.validateTargetDir(tezDir, conf).getPath();

    // Directory removal will be handled by cleanup at the SessionState level.
    return tezDir;
  }

  private Path reconnectTezDir(String sessionId, String suffix) throws IOException {
    // Make sure the path is normalized (we expect validation to pass since we just created it).
    Path dir = getScratchDirPath(sessionId, suffix, false);
    FileStatus dirFs = DagUtils.validateTargetDir(dir, conf);
    if (dirFs != null) {
      return dirFs.getPath();
    }
    // Perhaps the original session open used a different config.
    dir = getScratchDirPath(sessionId, suffix, true);
    if (dir == null) return null;
    dirFs = DagUtils.validateTargetDir(dir, conf);
    return (dirFs != null) ? dirFs.getPath() : null;
  }

  private Path getScratchDirPath(String sessionId, String suffix, boolean isAlternative) {
    // TODO: De-link from SessionState. A TezSession can be linked to different Hive Sessions via the pool.
    SessionState sessionState = SessionState.get();
    if (sessionState == null && isAlternative) return null; // No alternative.
    String hdfsScratchDir = (isAlternative || sessionState == null)
        ? HiveConf.getVar(conf, ConfVars.SCRATCHDIR) : sessionState.getHdfsScratchDirURIString();
    Path tezDir = new Path(hdfsScratchDir, TEZ_DIR);
    return new Path(tezDir, sessionId + ((suffix == null) ? "" : ("-" + suffix)));
  }

  /**
   * Returns a local resource representing a jar.
   * This resource will be used to execute the plan on the cluster.
   * @param localJarPath Local path to the jar to be localized.
   * @return LocalResource corresponding to the localized hive exec resource.
   * @throws IOException when any file system related call fails.
   * @throws LoginException when we are unable to determine the user.
   * @throws URISyntaxException when current jar location cannot be determined.
   */
  protected final LocalResource createJarLocalResource(String localJarPath)
      throws IOException, LoginException, IllegalArgumentException {
    // TODO Reduce the number of lookups that happen here. This shouldn't go to HDFS for each call.
    // The hiveJarDir can be determined once per client.
    FileStatus destDirStatus = utils.getHiveJarDirectory(conf);
    assert destDirStatus != null;
    Path destDirPath = destDirStatus.getPath();

    Path localFile = new Path(localJarPath);
    String sha = getSha(localFile);

    String destFileName = localFile.getName();

    // Now, try to find the file based on SHA and name. Currently we require exact name match.
    // We could also allow cutting off versions and other stuff provided that SHA matches...
    destFileName = FilenameUtils.removeExtension(destFileName) + "-" + sha
        + FilenameUtils.EXTENSION_SEPARATOR + FilenameUtils.getExtension(destFileName);

    if (LOG.isDebugEnabled()) {
      LOG.debug("The destination file name for [" + localJarPath + "] is " + destFileName);
    }

    // TODO: if this method is ever called on more than one jar, getting the dir and the
    //       list need to be refactored out to be done only once.
    Path destFile = new Path(destDirPath.toString() + "/" + destFileName);
    return utils.localizeResource(localFile, destFile, LocalResourceType.FILE, conf);
  }

  private String getKey(final FileStatus fileStatus) {
    return fileStatus.getPath() + ":" + fileStatus.getLen() + ":" + fileStatus.getModificationTime();
  }

  private void addJarLRByClassName(String className, final Map<String, LocalResource> lrMap) throws
      IOException, LoginException {
    Class<?> clazz;
    try {
      clazz = Class.forName(className);
    } catch (ClassNotFoundException e) {
      throw new IOException("Cannot find " + className + " in classpath", e);
    }
    addJarLRByClass(clazz, lrMap);
  }

  private void addJarLRByClass(Class<?> clazz, final Map<String, LocalResource> lrMap) throws IOException,
      LoginException {
    final File jar =
        new File(Utilities.jarFinderGetJar(clazz));
    final String localJarPath = jar.toURI().toURL().toExternalForm();
    final LocalResource jarLr = createJarLocalResource(localJarPath);
    lrMap.put(DagUtils.getBaseName(jarLr), jarLr);
  }

  private String getSha(final Path localFile) throws IOException, IllegalArgumentException {
    FileSystem localFs = FileSystem.getLocal(conf);
    FileStatus fileStatus = localFs.getFileStatus(localFile);
    String key = getKey(fileStatus);
    String sha256 = shaCache.getIfPresent(key);
    if (sha256 == null) {
      FSDataInputStream is = null;
      try {
        is = localFs.open(localFile);
        long start = System.currentTimeMillis();
        sha256 = DigestUtils.sha256Hex(is);
        long end = System.currentTimeMillis();
        LOG.info("Computed sha: {} for file: {} of length: {} in {} ms", sha256, localFile,
          LlapUtil.humanReadableByteCount(fileStatus.getLen()), end - start);
        shaCache.put(key, sha256);
      } finally {
        if (is != null) {
          is.close();
        }
      }
    }
    return sha256;
  }

  @Override
  public void setQueueName(String queueName) {
    this.queueName = queueName;
  }

  @Override
  public String getQueueName() {
    return queueName;
  }

  @Override
  public void setDefault() {
    defaultQueue = true;
  }

  @Override
  public boolean isDefault() {
    return defaultQueue;
  }

  @Override
  public HiveConf getConf() {
    return conf;
  }

  public List<LocalResource> getLocalizedResources() {
    if (resources == null) return new ArrayList<>();
    return new ArrayList<>(resources.localizedResources);
  }

  public String getUser() {
    return user;
  }

  @Override
  public boolean getDoAsEnabled() {
    return doAsEnabled;
  }

  /** Mark session as free for use from TezTask, for safety/debugging purposes. */
  @Override
  public void unsetOwnerThread() {
    if (ownerThread.getAndSet(null) == null) {
      throw new AssertionError("Not in use");
    }
  }

  /** Mark session as being in use from TezTask, for safety/debugging purposes. */
  @Override
  public void setOwnerThread() {
    String newName = Thread.currentThread().getName();
    do {
      String oldName = ownerThread.get();
      if (oldName != null) {
        throw new AssertionError("Tez session is already in use from "
            + oldName + "; cannot use from " + newName);
      }
    } while (!ownerThread.compareAndSet(null, newName));
  }

  @Override
  public void setLegacyLlapMode(boolean value) {
    this.isLegacyLlapMode = value;
  }

  @Override
  public boolean getLegacyLlapMode() {
    return this.isLegacyLlapMode;
  }

  @Override
  public void returnToSessionManager() throws Exception {
    // By default, TezSessionPoolManager handles this for both pool and non-pool session.
    TezSessionPoolManager.getInstance().returnSession(this);
  }

  @Override
  public TezSession reopen() throws Exception {
    // By default, TezSessionPoolManager handles this for both pool and non-pool session.
    return TezSessionPoolManager.getInstance().reopen(this);
  }

  @Override
  public void destroy() throws Exception {
    // By default, TezSessionPoolManager handles this for both pool and non-pool session.
    TezSessionPoolManager.getInstance().destroy(this);
  }

  @Override
  public WmContext getWmContext() {
    return wmContext;
  }

  public void setWmContext(final WmContext wmContext) {
    this.wmContext = wmContext;
  }

  public void setKillQuery(final KillQuery killQuery) {
    this.killQuery = killQuery;
  }

  public HiveResources extractHiveResources() {
    HiveResources result = resources;
    resources = null;
    return result;
  }

  @Override
  public Path replaceHiveResources(HiveResources resources, boolean isAsync) {
    Path dir = null;
    if (this.resources != null) {
      dir = this.resources.dagResourcesDir;
      if (!isAsync) {
        try {
          dir.getFileSystem(conf).delete(dir, true);
        } catch (Exception ex) {
          LOG.error("Failed to delete the old resources directory "
              + dir + "; ignoring " + ex.getLocalizedMessage());
        }
        dir = null;
      }
    }
    this.resources = resources;
    return dir;
  }

  @Override
  public boolean killQuery(String reason) throws HiveException {
    if (killQuery == null || wmContext == null) return false;
    String queryId = wmContext.getQueryId();
    if (queryId == null) return false;
    LOG.info("Killing the query {}: {}", queryId, reason);
    killQuery.killQuery(queryId, reason, conf, true);
    return true;
  }
}
