/*
 * Decompiled with CFR 0.152.
 */
package org.apache.sentry.service.thrift;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.MalformedURLException;
import java.net.ServerSocket;
import java.security.PrivilegedExceptionAction;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import javax.security.auth.Subject;
import javax.security.auth.callback.CallbackHandler;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.net.NetUtils;
import org.apache.hadoop.security.SaslRpcServer;
import org.apache.hadoop.security.SecurityUtil;
import org.apache.sentry.Command;
import org.apache.sentry.api.common.SentryServiceUtil;
import org.apache.sentry.api.service.thrift.SentryMetrics;
import org.apache.sentry.core.common.utils.SigUtils;
import org.apache.sentry.provider.db.service.persistent.HMSFollower;
import org.apache.sentry.provider.db.service.persistent.LeaderStatusMonitor;
import org.apache.sentry.provider.db.service.persistent.SentryStoreFactory;
import org.apache.sentry.provider.db.service.persistent.SentryStoreInterface;
import org.apache.sentry.service.common.ServiceConstants;
import org.apache.sentry.service.thrift.GSSCallback;
import org.apache.sentry.service.thrift.HiveSimpleConnectionFactory;
import org.apache.sentry.service.thrift.ProcessorFactory;
import org.apache.sentry.service.thrift.SentryKerberosContext;
import org.apache.sentry.service.thrift.SentryServiceState;
import org.apache.sentry.service.thrift.SentryStateBank;
import org.apache.sentry.service.web.SentryWebServer;
import org.apache.thrift.TMultiplexedProcessor;
import org.apache.thrift.TProcessor;
import org.apache.thrift.protocol.TBinaryProtocol;
import org.apache.thrift.protocol.TProtocolFactory;
import org.apache.thrift.server.TServer;
import org.apache.thrift.server.TServerEventHandler;
import org.apache.thrift.server.TThreadPoolServer;
import org.apache.thrift.transport.TSaslServerTransport;
import org.apache.thrift.transport.TServerSocket;
import org.apache.thrift.transport.TServerTransport;
import org.apache.thrift.transport.TTransportFactory;
import org.eclipse.jetty.util.MultiException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sentry.com.codahale.metrics.Gauge;

public class SentryService
implements Callable,
SigUtils.SigListener {
    private static final Logger LOGGER = LoggerFactory.getLogger(SentryService.class);
    private HiveSimpleConnectionFactory hiveConnectionFactory;
    private static final String SENTRY_SERVICE_THREAD_NAME = "sentry-service";
    private static final String HMSFOLLOWER_THREAD_NAME = "hms-follower";
    private static final String STORE_CLEANER_THREAD_NAME = "store-cleaner";
    private static final String SERVICE_SHUTDOWN_THREAD_NAME = "service-shutdown";
    private final Configuration conf;
    private final InetSocketAddress address;
    private final int maxThreads;
    private final int minThreads;
    private final boolean kerberos;
    private final String principal;
    private final String[] principalParts;
    private final String keytab;
    private final ExecutorService serviceExecutor;
    private ScheduledExecutorService hmsFollowerExecutor = null;
    private HMSFollower hmsFollower = null;
    private Future serviceStatus;
    private TServer thriftServer;
    private Status status;
    private SentryWebServer sentryWebServer;
    private final long maxMessageSize;
    private final SentryStoreInterface sentryStore;
    private ScheduledExecutorService sentryStoreCleanService;
    private final LeaderStatusMonitor leaderMonitor;

    public SentryService(Configuration conf) throws Exception {
        this.conf = conf;
        int port = conf.getInt("sentry.service.server.rpc-port", 8038);
        if (port == 0) {
            port = SentryService.findFreePort();
            conf.setInt("sentry.service.server.rpc-port", port);
        }
        this.address = NetUtils.createSocketAddr((String)conf.get("sentry.service.server.rpc-address", "0.0.0.0"), (int)port);
        LOGGER.info("Configured on address {}", (Object)this.address);
        this.kerberos = "kerberos".equalsIgnoreCase(conf.get("sentry.service.security.mode", "kerberos").trim());
        this.maxThreads = conf.getInt("sentry.service.server-max-threads", 500);
        this.minThreads = conf.getInt("sentry.service.server-min-threads", 10);
        this.maxMessageSize = conf.getLong("sentry.policy.server.thrift.max.message.size", 0x6400000L);
        if (this.kerberos) {
            try {
                String rawPrincipal = (String)Preconditions.checkNotNull((Object)conf.get("sentry.service.server.principal"), (Object)"sentry.service.server.principal is required");
                this.principal = SecurityUtil.getServerPrincipal((String)rawPrincipal, (InetAddress)this.address.getAddress());
            }
            catch (IOException io) {
                throw new RuntimeException("Can't translate kerberos principal'", io);
            }
            LOGGER.info("Using kerberos principal: {}", (Object)this.principal);
            this.principalParts = SaslRpcServer.splitKerberosName((String)this.principal);
            Preconditions.checkArgument((this.principalParts.length == 3 ? 1 : 0) != 0, (Object)("Kerberos principal should have 3 parts: " + this.principal));
            this.keytab = (String)Preconditions.checkNotNull((Object)conf.get("sentry.service.server.keytab"), (Object)"sentry.service.server.keytab is required");
            File keytabFile = new File(this.keytab);
            Preconditions.checkState((keytabFile.isFile() && keytabFile.canRead() ? 1 : 0) != 0, (String)"Keytab %s does not exist or is not readable.", (Object[])new Object[]{this.keytab});
        } else {
            this.principal = null;
            this.principalParts = null;
            this.keytab = null;
        }
        ThreadFactory sentryServiceThreadFactory = new ThreadFactoryBuilder().setNameFormat(SENTRY_SERVICE_THREAD_NAME).build();
        this.serviceExecutor = Executors.newSingleThreadExecutor(sentryServiceThreadFactory);
        this.sentryStore = SentryStoreFactory.create(conf, LOGGER);
        this.sentryStore.setPersistUpdateDeltas(SentryServiceUtil.isHDFSSyncEnabled((Configuration)conf));
        this.leaderMonitor = LeaderStatusMonitor.getLeaderStatusMonitor(conf);
        this.status = Status.NOT_STARTED;
        String sigName = conf.get("sentry.ha.standby.signal");
        if (sigName != null && !sigName.isEmpty()) {
            LOGGER.info("Registering signal handler {} for HA", (Object)sigName);
            try {
                SigUtils.registerSigListener((String)sigName, (SigUtils.SigListener)this);
            }
            catch (Exception e) {
                LOGGER.error("Failed to register signal", (Throwable)e);
            }
        }
    }

    public String call() throws Exception {
        SentryKerberosContext kerberosContext = null;
        try {
            this.status = Status.STARTED;
            if (this.kerberos) {
                kerberosContext = new SentryKerberosContext(this.principal, this.keytab, true);
                Subject.doAs(kerberosContext.getSubject(), new PrivilegedExceptionAction<Void>(){

                    @Override
                    public Void run() throws Exception {
                        SentryService.this.runServer();
                        return null;
                    }
                });
            } else {
                this.runServer();
            }
        }
        catch (Exception t) {
            LOGGER.error("Error starting server", (Throwable)t);
            throw new Exception("Error starting server", t);
        }
        finally {
            if (kerberosContext != null) {
                kerberosContext.shutDown();
            }
            this.status = Status.NOT_STARTED;
        }
        return null;
    }

    private void runServer() throws Exception {
        this.startSentryStoreCleaner(this.conf);
        this.startHMSFollower(this.conf);
        Iterable processorFactories = ServiceConstants.ConfUtilties.CLASS_SPLITTER.split((CharSequence)this.conf.get("sentry.service.processor.factories", "org.apache.sentry.api.service.thrift.SentryPolicyStoreProcessorFactory,org.apache.sentry.api.generic.thrift.SentryGenericPolicyProcessorFactory").trim());
        TMultiplexedProcessor processor = new TMultiplexedProcessor();
        boolean registeredProcessor = false;
        for (String processorFactory : processorFactories) {
            Class clazz = this.conf.getClassByName(processorFactory);
            if (!ProcessorFactory.class.isAssignableFrom(clazz)) {
                throw new IllegalArgumentException("Processor Factory " + processorFactory + " is not a " + ProcessorFactory.class.getName());
            }
            try {
                Constructor constructor = clazz.getConstructor(Configuration.class);
                LOGGER.info("ProcessorFactory being used: " + clazz.getCanonicalName());
                ProcessorFactory factory = (ProcessorFactory)constructor.newInstance(this.conf);
                boolean registerStatus = factory.register(processor, this.sentryStore);
                if (!registerStatus) {
                    LOGGER.error("Failed to register " + clazz.getCanonicalName());
                }
                registeredProcessor = registerStatus || registeredProcessor;
            }
            catch (Exception e) {
                throw new IllegalStateException("Could not create " + processorFactory, e);
            }
        }
        if (!registeredProcessor) {
            throw new IllegalStateException("Failed to register any processors from " + processorFactories);
        }
        this.addSentryServiceGauge();
        TServerSocket serverTransport = new TServerSocket(this.address);
        TTransportFactory transportFactory = null;
        if (this.kerberos) {
            TSaslServerTransport.Factory saslTransportFactory = new TSaslServerTransport.Factory();
            saslTransportFactory.addServerDefinition(SaslRpcServer.AuthMethod.KERBEROS.getMechanismName(), this.principalParts[0], this.principalParts[1], (Map)ServiceConstants.ServerConfig.SASL_PROPERTIES, (CallbackHandler)((Object)new GSSCallback(this.conf)));
            transportFactory = saslTransportFactory;
        } else {
            transportFactory = new TTransportFactory();
        }
        TThreadPoolServer.Args args = ((TThreadPoolServer.Args)((TThreadPoolServer.Args)((TThreadPoolServer.Args)new TThreadPoolServer.Args((TServerTransport)serverTransport).processor((TProcessor)processor)).transportFactory(transportFactory)).protocolFactory((TProtocolFactory)new TBinaryProtocol.Factory(true, true, this.maxMessageSize, this.maxMessageSize))).minWorkerThreads(this.minThreads).maxWorkerThreads(this.maxThreads);
        this.thriftServer = new TThreadPoolServer(args);
        LOGGER.info("Serving on {}", (Object)this.address);
        this.startSentryWebServer();
        LOGGER.info("Sentry service is ready to serve client requests");
        System.out.println("Sentry service is ready to serve client requests");
        SentryStateBank.enableState("SentryService", SentryServiceState.SERVICE_RUNNING);
        this.thriftServer.serve();
    }

    private void startHMSFollower(Configuration conf) throws Exception {
        boolean syncPolicyStore = SentryServiceUtil.isSyncPolicyStoreEnabled((Configuration)conf);
        if (!SentryServiceUtil.isHDFSSyncEnabled((Configuration)conf) && !syncPolicyStore) {
            LOGGER.info("HMS follower is not started because HDFS sync is disabled and perm sync is disabled");
            return;
        }
        String metastoreURI = SentryServiceUtil.getHiveMetastoreURI();
        if (metastoreURI == null) {
            LOGGER.info("Metastore uri is not configured. Do not start HMSFollower");
            return;
        }
        LOGGER.info("Starting HMSFollower to HMS {}", (Object)metastoreURI);
        Preconditions.checkState((this.hmsFollower == null ? 1 : 0) != 0);
        Preconditions.checkState((this.hmsFollowerExecutor == null ? 1 : 0) != 0);
        Preconditions.checkState((this.hiveConnectionFactory == null ? 1 : 0) != 0);
        this.hiveConnectionFactory = new HiveSimpleConnectionFactory(conf, new HiveConf());
        this.hiveConnectionFactory.init();
        this.hmsFollower = new HMSFollower(conf, this.sentryStore, this.leaderMonitor, this.hiveConnectionFactory);
        long initDelay = conf.getLong("sentry.hmsfollower.init.delay.mills", 0L);
        long period = conf.getLong("sentry.hmsfollower.interval.mills", 500L);
        try {
            ThreadFactory hmsFollowerThreadFactory = new ThreadFactoryBuilder().setNameFormat(HMSFOLLOWER_THREAD_NAME).build();
            this.hmsFollowerExecutor = Executors.newScheduledThreadPool(1, hmsFollowerThreadFactory);
            this.hmsFollowerExecutor.scheduleAtFixedRate(this.hmsFollower, initDelay, period, TimeUnit.MILLISECONDS);
        }
        catch (IllegalArgumentException e) {
            LOGGER.error(String.format("Could not start HMSFollower due to illegal argument. period is %s ms", period), (Throwable)e);
            throw e;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void stopHMSFollower(Configuration conf) {
        if (this.hmsFollowerExecutor == null || this.hmsFollower == null) {
            Preconditions.checkState((this.hmsFollower == null ? 1 : 0) != 0);
            Preconditions.checkState((this.hmsFollowerExecutor == null ? 1 : 0) != 0);
            LOGGER.debug("Skip shuting down hmsFollowerExecutor and closing hmsFollower because they are not created");
            return;
        }
        Preconditions.checkNotNull((Object)this.hmsFollowerExecutor);
        Preconditions.checkNotNull((Object)this.hmsFollower);
        Preconditions.checkNotNull((Object)this.hiveConnectionFactory);
        long timeoutValue = conf.getLong("sentry.hmsfollower.interval.mills", 500L);
        try {
            SentryServiceUtil.shutdownAndAwaitTermination((ExecutorService)this.hmsFollowerExecutor, (String)"hmsFollowerExecutor", (long)timeoutValue, (TimeUnit)TimeUnit.MILLISECONDS, (Logger)LOGGER);
        }
        finally {
            try {
                this.hiveConnectionFactory.close();
            }
            catch (Exception e) {
                LOGGER.error("Can't close HiveConnectionFactory", (Throwable)e);
            }
            this.hmsFollowerExecutor = null;
            this.hiveConnectionFactory = null;
            try {
                this.hmsFollower.close();
            }
            catch (Exception ex) {
                LOGGER.error("HMSFollower.close() failed", (Throwable)ex);
            }
            finally {
                this.hmsFollower = null;
            }
        }
    }

    private void startSentryStoreCleaner(Configuration conf) {
        Preconditions.checkState((this.sentryStoreCleanService == null ? 1 : 0) != 0);
        long storeCleanPeriodSecs = conf.getLong("sentry.store.clean.period.seconds", 43200L);
        if (storeCleanPeriodSecs <= 0L) {
            return;
        }
        try {
            Runnable storeCleaner = new Runnable(){

                @Override
                public void run() {
                    if (SentryService.this.leaderMonitor.isLeader()) {
                        SentryService.this.sentryStore.purgeDeltaChangeTables();
                        SentryService.this.sentryStore.purgeNotificationIdTable();
                    }
                }
            };
            ThreadFactory sentryStoreCleanerThreadFactory = new ThreadFactoryBuilder().setNameFormat(STORE_CLEANER_THREAD_NAME).build();
            this.sentryStoreCleanService = Executors.newSingleThreadScheduledExecutor(sentryStoreCleanerThreadFactory);
            this.sentryStoreCleanService.scheduleWithFixedDelay(storeCleaner, 0L, storeCleanPeriodSecs, TimeUnit.SECONDS);
            LOGGER.info("sentry store cleaner is scheduled with interval {} seconds", (Object)storeCleanPeriodSecs);
        }
        catch (IllegalArgumentException e) {
            LOGGER.error("Could not start SentryStoreCleaner due to illegal argument", (Throwable)e);
            this.sentryStoreCleanService = null;
        }
    }

    private void stopSentryStoreCleaner() {
        Preconditions.checkNotNull((Object)this.sentryStoreCleanService);
        try {
            SentryServiceUtil.shutdownAndAwaitTermination((ExecutorService)this.sentryStoreCleanService, (String)"sentryStoreCleanService", (long)10L, (TimeUnit)TimeUnit.SECONDS, (Logger)LOGGER);
        }
        finally {
            this.sentryStoreCleanService = null;
        }
    }

    private void addSentryServiceGauge() {
        SentryMetrics.getInstance().addSentryServiceGauges(this);
    }

    private void startSentryWebServer() throws Exception {
        if (this.conf.getBoolean("sentry.service.web.enable", ServiceConstants.ServerConfig.SENTRY_WEB_ENABLE_DEFAULT.booleanValue())) {
            this.sentryWebServer = new SentryWebServer(this.conf);
            this.sentryWebServer.start();
        }
    }

    private void stopSentryWebServer() throws Exception {
        if (this.sentryWebServer != null) {
            this.sentryWebServer.stop();
            this.sentryWebServer = null;
        }
    }

    public InetSocketAddress getAddress() {
        return this.address;
    }

    public synchronized boolean isRunning() {
        return this.status == Status.STARTED && this.thriftServer != null && this.thriftServer.isServing();
    }

    public synchronized void start() throws Exception {
        if (this.status != Status.NOT_STARTED) {
            throw new IllegalStateException("Cannot start when " + (Object)((Object)this.status));
        }
        LOGGER.info("Attempting to start...");
        this.serviceStatus = this.serviceExecutor.submit(this);
    }

    public synchronized void stop() throws Exception {
        MultiException exception = null;
        LOGGER.info("Attempting to stop...");
        this.leaderMonitor.close();
        if (this.isRunning()) {
            LOGGER.info("Attempting to stop sentry thrift service...");
            try {
                this.thriftServer.stop();
                this.thriftServer = null;
                this.status = Status.NOT_STARTED;
            }
            catch (Exception e) {
                LOGGER.error("Error while stopping sentry thrift service", (Throwable)e);
                exception = this.addMultiException(exception, e);
            }
        } else {
            this.thriftServer = null;
            this.status = Status.NOT_STARTED;
            LOGGER.info("Sentry thrift service is already stopped...");
        }
        if (this.isWebServerRunning()) {
            try {
                LOGGER.info("Attempting to stop sentry web service...");
                this.stopSentryWebServer();
            }
            catch (Exception e) {
                LOGGER.error("Error while stopping sentry web service", (Throwable)e);
                exception = this.addMultiException(exception, e);
            }
        } else {
            LOGGER.info("Sentry web service is already stopped...");
        }
        this.stopHMSFollower(this.conf);
        this.stopSentryStoreCleaner();
        if (exception != null) {
            exception.ifExceptionThrow();
        }
        SentryStateBank.disableState("SentryService", SentryServiceState.SERVICE_RUNNING);
        LOGGER.info("Stopped...");
    }

    @VisibleForTesting
    public synchronized void becomeStandby() {
        this.leaderMonitor.deactivate();
    }

    private MultiException addMultiException(MultiException exception, Exception e) {
        MultiException newException = exception;
        if (newException == null) {
            newException = new MultiException();
        }
        newException.add((Throwable)e);
        return newException;
    }

    private boolean isWebServerRunning() {
        return this.sentryWebServer != null && this.sentryWebServer.isAlive();
    }

    private static int findFreePort() {
        int attempts = 0;
        while (attempts++ <= 1000) {
            try {
                ServerSocket s = new ServerSocket(0);
                int port = s.getLocalPort();
                s.close();
                return port;
            }
            catch (IOException iOException) {
            }
        }
        throw new IllegalStateException("Unable to find a port after 1000 attempts");
    }

    public static Configuration loadConfig(String configFileName) throws MalformedURLException {
        File configFile = null;
        if (configFileName == null) {
            throw new IllegalArgumentException("Usage: conffile path/to/sentry-service.xml");
        }
        configFile = new File(configFileName);
        if (!configFile.isFile() || !configFile.canRead()) {
            throw new IllegalArgumentException("Cannot read configuration file " + configFile);
        }
        Configuration conf = new Configuration(true);
        conf.addResource(configFile.toURI().toURL(), true);
        return conf;
    }

    public Configuration getConf() {
        return this.conf;
    }

    public void setThriftEventHandler(TServerEventHandler eventHandler) throws IllegalStateException {
        if (this.thriftServer == null) {
            throw new IllegalStateException("Server is not initialized or stopped");
        }
        this.thriftServer.setServerEventHandler(eventHandler);
    }

    public TServerEventHandler getThriftEventHandler() throws IllegalStateException {
        if (this.thriftServer == null) {
            throw new IllegalStateException("Server is not initialized or stopped");
        }
        return this.thriftServer.getEventHandler();
    }

    public Gauge<Boolean> getIsActiveGauge() {
        return new Gauge<Boolean>(){

            @Override
            public Boolean getValue() {
                return SentryService.this.leaderMonitor.isLeader();
            }
        };
    }

    public Gauge<Long> getBecomeActiveCount() {
        return new Gauge<Long>(){

            @Override
            public Long getValue() {
                return SentryService.this.leaderMonitor.getLeaderCount();
            }
        };
    }

    public void onSignal(String signalName) {
        this.leaderMonitor.deactivate();
    }

    @VisibleForTesting
    public void restartHMSFollower(Configuration newConf) throws Exception {
        this.stopHMSFollower(this.conf);
        this.startHMSFollower(newConf);
    }

    public static class CommandImpl
    implements Command {
        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void run(String[] args) throws Exception {
            GnuParser parser = new GnuParser();
            Options options = new Options();
            options.addOption("c", "conffile", true, "Sentry Service configuration file");
            CommandLine commandLine = parser.parse(options, args);
            String configFileName = commandLine.getOptionValue("conffile");
            File configFile = null;
            if (configFileName == null || commandLine.hasOption("h") || commandLine.hasOption("help")) {
                HelpFormatter formatter = new HelpFormatter();
                formatter.printHelp("sentry --command service", options);
                System.exit(-1);
            } else {
                configFile = new File(configFileName);
                if (!configFile.isFile() || !configFile.canRead()) {
                    throw new IllegalArgumentException("Cannot read configuration file " + configFile);
                }
            }
            Configuration serverConf = SentryService.loadConfig(configFileName);
            final SentryService server = new SentryService(serverConf);
            server.start();
            ThreadFactory serviceShutdownThreadFactory = new ThreadFactoryBuilder().setNameFormat(SentryService.SERVICE_SHUTDOWN_THREAD_NAME).build();
            Runtime.getRuntime().addShutdownHook(serviceShutdownThreadFactory.newThread(new Runnable(){

                @Override
                public void run() {
                    LOGGER.info("ShutdownHook shutting down server");
                    try {
                        server.stop();
                    }
                    catch (Throwable t) {
                        LOGGER.error("Error stopping SentryService", t);
                        System.exit(1);
                    }
                }
            }));
            try {
                server.serviceStatus.get();
            }
            finally {
                server.serviceExecutor.shutdown();
            }
        }
    }

    private static enum Status {
        NOT_STARTED,
        STARTED;

    }
}

