/*
 * Decompiled with CFR 0.152.
 */
package org.apache.sentry.service.thrift;

import com.google.common.collect.ImmutableMap;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hive.metastore.api.Database;
import org.apache.hadoop.hive.metastore.api.Partition;
import org.apache.hadoop.hive.metastore.api.Table;
import org.apache.sentry.api.service.thrift.SentryMetrics;
import org.apache.sentry.hdfs.PathsUpdate;
import org.apache.sentry.hdfs.SentryMalformedPathException;
import org.apache.sentry.service.thrift.HMSClient;
import org.apache.sentry.service.thrift.HiveConnectionFactory;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class FullUpdateInitializer
implements AutoCloseable {
    private static final String FULL_UPDATE_INITIALIZER_THREAD_NAME = "hms-fetch-%d";
    private final ExecutorService threadPool;
    private final int maxPartitionsPerCall;
    private final int maxTablesPerCall;
    private final Deque<Future<CallResult>> results = new ConcurrentLinkedDeque<Future<CallResult>>();
    private final int maxRetries;
    private final int waitDurationMillis;
    private final long printSnapshotFetchTimeInterval;
    private int totalNumberOfDatabasesFetched;
    private int totalNumberOfTablesFetched;
    private int totalNumberOfPartitionsFetched;
    private static final Logger LOGGER = LoggerFactory.getLogger(FullUpdateInitializer.class);
    private static final ObjectMapping emptyObjectMapping = new ObjectMapping(Collections.emptyMap());
    private final HiveConnectionFactory clientFactory;

    static String pathFromURI(String uri) {
        try {
            return PathsUpdate.parsePath((String)uri);
        }
        catch (SentryMalformedPathException e) {
            LOGGER.warn(String.format("Ignoring invalid uri %s: %s", uri, e.getReason()));
            return null;
        }
    }

    FullUpdateInitializer(HiveConnectionFactory clientFactory, Configuration conf) {
        this.clientFactory = clientFactory;
        this.maxPartitionsPerCall = conf.getInt("sentry.hdfs.sync.metastore.cache.max-partitions-per-rpc", 100);
        this.maxTablesPerCall = conf.getInt("sentry.hdfs.sync.metastore.cache.max-tables-per-rpc", 100);
        this.maxRetries = conf.getInt("sentry.hdfs.sync.metastore.cache.retry.max.num", 1);
        this.waitDurationMillis = conf.getInt("sentry.hdfs.sync.metastore.cache.retry.wait.duration.millis", 1000);
        this.printSnapshotFetchTimeInterval = conf.getInt("sentry.hdfs.sync.metastore.cache.print-snapshot-fetch-interval.millis", 300000);
        ThreadFactory fullUpdateInitThreadFactory = new ThreadFactoryBuilder().setNameFormat(FULL_UPDATE_INITIALIZER_THREAD_NAME).setDaemon(false).build();
        this.threadPool = Executors.newFixedThreadPool(conf.getInt("sentry.hdfs.sync.metastore.cache.init.threads", 10), fullUpdateInitThreadFactory);
    }

    Map<String, Collection<String>> getFullHMSSnapshot() throws Exception {
        List allDbStr;
        HMSClient c = null;
        try (HMSClient client = this.clientFactory.connect();){
            c = client;
            LOGGER.debug("Fetch all db names");
            allDbStr = client.getClient().getAllDatabases();
            SentryMetrics.getInstance().databaseCount.inc(allDbStr.size());
            LOGGER.info("Total number of db names fetched = {}", (Object)allDbStr.size());
        }
        catch (Exception e) {
            if (c != null) {
                c.invalidate();
            }
            throw e;
        }
        for (String dbName : allDbStr) {
            this.results.add(this.threadPool.submit(new DbTask(dbName)));
        }
        HashMap<String, Collection<String>> fullSnapshot = new HashMap<String, Collection<String>>();
        long printMessageTime = System.currentTimeMillis();
        while (!this.results.isEmpty()) {
            Future<CallResult> result = this.results.pop();
            CallResult callResult = result.get();
            if (!callResult.success()) {
                throw callResult.getFailure();
            }
            Map<String, Set<String>> objectMapping = callResult.getObjectMapping().getObjects();
            for (Map.Entry<String, Set<String>> entry : objectMapping.entrySet()) {
                String key = entry.getKey();
                Set<String> val = entry.getValue();
                Set existingSet = (Set)fullSnapshot.get(key);
                if (existingSet == null) {
                    fullSnapshot.put(key, val);
                    continue;
                }
                existingSet.addAll(val);
            }
            if (System.currentTimeMillis() - printMessageTime <= this.printSnapshotFetchTimeInterval) continue;
            long totalNumberOfDatabases = SentryMetrics.getInstance().databaseCount.getCount();
            long totalNumberOfTables = SentryMetrics.getInstance().tableCount.getCount();
            long totalNumberOfPartitions = SentryMetrics.getInstance().partitionCount.getCount();
            double percentageDatabasesFetched = totalNumberOfDatabases > 0L ? (double)this.totalNumberOfDatabasesFetched / (double)totalNumberOfDatabases * 100.0 : 0.0;
            double percentageTablesFetched = totalNumberOfTables > 0L ? (double)this.totalNumberOfTablesFetched / (double)totalNumberOfTables * 100.0 : 0.0;
            double percentagePartitionsFetched = totalNumberOfPartitions > 0L ? (double)this.totalNumberOfPartitionsFetched / (double)totalNumberOfPartitions * 100.0 : 0.0;
            String snapshotFetchStatusString = String.format("Fetching full hms snapshot: databases fetched=%d (%.2f%%); tables fetched=%d (%.2f%%); partitions fetched=%d (%.2f%%); total number of databases=%d; total number of tables=%d total number of partitions=%d", this.totalNumberOfDatabasesFetched, percentageDatabasesFetched, this.totalNumberOfTablesFetched, percentageTablesFetched, this.totalNumberOfPartitionsFetched, percentagePartitionsFetched, totalNumberOfDatabases, totalNumberOfTables, totalNumberOfPartitions);
            LOGGER.info(snapshotFetchStatusString);
            printMessageTime = System.currentTimeMillis();
        }
        return fullSnapshot;
    }

    @Override
    public void close() {
        this.threadPool.shutdownNow();
        try {
            this.threadPool.awaitTermination(1L, TimeUnit.SECONDS);
        }
        catch (InterruptedException ignored) {
            LOGGER.warn("Interrupted shutdown");
            Thread.currentThread().interrupt();
        }
    }

    static String safeIntern(String arg) {
        return arg != null ? arg.intern() : null;
    }

    private class DbTask
    extends BaseTask {
        private final String dbName;

        DbTask(String dbName) {
            this.dbName = FullUpdateInitializer.safeIntern(dbName.toLowerCase());
        }

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        @Override
        ObjectMapping doTask() throws Exception {
            long startTime = System.currentTimeMillis();
            HMSClient c = null;
            try (HMSClient client = FullUpdateInitializer.this.clientFactory.connect();){
                c = client;
                LOGGER.debug("Fetching database object for db = {}", (Object)this.dbName);
                Database db = client.getClient().getDatabase(this.dbName);
                FullUpdateInitializer.this.totalNumberOfDatabasesFetched++;
                if (!this.dbName.equalsIgnoreCase(db.getName())) {
                    LOGGER.warn("Database name {} does not match {}", (Object)db.getName(), (Object)this.dbName);
                    ObjectMapping objectMapping = emptyObjectMapping;
                    return objectMapping;
                }
                LOGGER.debug("Fetch all table names for db = {}", (Object)this.dbName);
                List allTblStr = client.getClient().getAllTables(this.dbName);
                LOGGER.info("For db = {} total number of table names fetched = {}", (Object)this.dbName, (Object)allTblStr.size());
                SentryMetrics.getInstance().tableCount.inc(allTblStr.size());
                for (int i = 0; i < allTblStr.size(); i += FullUpdateInitializer.this.maxTablesPerCall) {
                    List<String> tablesToFetch = allTblStr.subList(i, Math.min(i + FullUpdateInitializer.this.maxTablesPerCall, allTblStr.size()));
                    TableTask tableTask = new TableTask(db, tablesToFetch);
                    FullUpdateInitializer.this.results.add(FullUpdateInitializer.this.threadPool.submit(tableTask));
                }
                String dbPath = FullUpdateInitializer.safeIntern(FullUpdateInitializer.pathFromURI(db.getLocationUri()));
                LOGGER.debug("Completed database task for db = {}. Current task size = {}. Time Taken = {} ms", new Object[]{this.dbName, FullUpdateInitializer.this.results.size(), System.currentTimeMillis() - startTime});
                ObjectMapping objectMapping = dbPath != null ? new ObjectMapping(this.dbName, dbPath) : emptyObjectMapping;
                return objectMapping;
            }
            catch (Exception e) {
                if (c == null) throw e;
                c.invalidate();
                throw e;
            }
        }
    }

    private class TableTask
    extends BaseTask {
        private final String dbName;
        private final List<String> tableNames;

        TableTask(Database db, List<String> tableNames) {
            this.dbName = FullUpdateInitializer.safeIntern(db.getName());
            this.tableNames = tableNames;
        }

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        @Override
        ObjectMapping doTask() throws Exception {
            long startTime = System.currentTimeMillis();
            HMSClient c = null;
            try (HMSClient client = FullUpdateInitializer.this.clientFactory.connect();){
                c = client;
                LOGGER.debug("Fetching table objects for db = {} tables count = {} tables = {}", new Object[]{this.dbName, this.tableNames.size(), this.tableNames});
                List tables = client.getClient().getTableObjectsByName(this.dbName, this.tableNames);
                FullUpdateInitializer.this.totalNumberOfTablesFetched = FullUpdateInitializer.this.totalNumberOfTablesFetched + tables.size();
                HashMap<String, HashSet<String>> objectMapping = new HashMap<String, HashSet<String>>(tables.size());
                for (Table tbl : tables) {
                    if (!tbl.getDbName().equalsIgnoreCase(this.dbName)) {
                        LOGGER.warn(String.format("DB name %s for table %s does not match %s", tbl.getDbName(), tbl.getTableName(), this.dbName));
                        continue;
                    }
                    String tableName = FullUpdateInitializer.safeIntern(tbl.getTableName().toLowerCase());
                    String authzObject = (this.dbName + "." + tableName).intern();
                    LOGGER.debug("Fetch all partition names for db = {} table = {}", (Object)this.dbName, (Object)tableName);
                    List tblPartNames = client.getClient().listPartitionNames(this.dbName, tableName, (short)-1);
                    LOGGER.info("For db = {} table = {} total number of partitions = {}", new Object[]{this.dbName, tableName, tblPartNames.size()});
                    SentryMetrics.getInstance().partitionCount.inc(tblPartNames.size());
                    for (int i = 0; i < tblPartNames.size(); i += FullUpdateInitializer.this.maxPartitionsPerCall) {
                        List<String> partsToFetch = tblPartNames.subList(i, Math.min(i + FullUpdateInitializer.this.maxPartitionsPerCall, tblPartNames.size()));
                        PartitionTask partTask = new PartitionTask(this.dbName, tableName, authzObject, partsToFetch);
                        FullUpdateInitializer.this.results.add(FullUpdateInitializer.this.threadPool.submit(partTask));
                    }
                    String tblPath = FullUpdateInitializer.safeIntern(FullUpdateInitializer.pathFromURI(tbl.getSd().getLocation()));
                    if (tblPath == null) continue;
                    HashSet<String> paths = (HashSet<String>)objectMapping.get(authzObject);
                    if (paths == null) {
                        paths = new HashSet<String>(1);
                        objectMapping.put(authzObject, paths);
                    }
                    paths.add(tblPath);
                }
                LOGGER.debug("Completed table task for db = {} tables = {}. Current task size = {}. Time Taken = {} ms", new Object[]{this.dbName, this.tableNames, FullUpdateInitializer.this.results.size(), System.currentTimeMillis() - startTime});
                ObjectMapping objectMapping2 = new ObjectMapping(Collections.unmodifiableMap(objectMapping));
                return objectMapping2;
            }
            catch (Exception e) {
                if (c == null) throw e;
                c.invalidate();
                throw e;
            }
        }
    }

    private class PartitionTask
    extends BaseTask {
        private final String dbName;
        private final String tblName;
        private final String authName;
        private final List<String> partNames;

        PartitionTask(String dbName, String tblName, String authName, List<String> partNames) {
            this.dbName = FullUpdateInitializer.safeIntern(dbName);
            this.tblName = FullUpdateInitializer.safeIntern(tblName);
            this.authName = FullUpdateInitializer.safeIntern(authName);
            this.partNames = partNames;
        }

        @Override
        ObjectMapping doTask() throws Exception {
            List tblParts;
            long startTime = System.currentTimeMillis();
            HMSClient c = null;
            try {
                HMSClient client = FullUpdateInitializer.this.clientFactory.connect();
                Object object = null;
                try {
                    c = client;
                    LOGGER.debug("Fetching partition objects for db = {} table = {}", (Object)this.dbName, (Object)this.tblName);
                    tblParts = client.getClient().getPartitionsByNames(this.dbName, this.tblName, this.partNames);
                }
                catch (Throwable throwable) {
                    object = throwable;
                    throw throwable;
                }
                finally {
                    if (client != null) {
                        if (object != null) {
                            try {
                                client.close();
                            }
                            catch (Throwable throwable) {
                                ((Throwable)object).addSuppressed(throwable);
                            }
                        } else {
                            client.close();
                        }
                    }
                }
            }
            catch (Exception e) {
                if (c != null) {
                    c.invalidate();
                }
                throw e;
            }
            FullUpdateInitializer.this.totalNumberOfPartitionsFetched = FullUpdateInitializer.this.totalNumberOfPartitionsFetched + tblParts.size();
            ArrayList<String> partitionNames = new ArrayList<String>(tblParts.size());
            for (Partition part : tblParts) {
                if (part != null && part.getSd() != null) {
                    String partPath = FullUpdateInitializer.pathFromURI(part.getSd().getLocation());
                    if (partPath == null) continue;
                    partitionNames.add(partPath.intern());
                    continue;
                }
                LOGGER.info("Partition or its storage descriptor is null while fetching partitions for db = {} table = {}", (Object)this.dbName, (Object)this.tblName);
            }
            LOGGER.debug("Completed partition task for db = {} table = {}. Current task size = {}. Time Taken {} ms", new Object[]{this.dbName, this.tblName, FullUpdateInitializer.this.results.size(), System.currentTimeMillis() - startTime});
            return new ObjectMapping(this.authName, partitionNames);
        }
    }

    private abstract class BaseTask
    implements Callable<CallResult> {
        private final RetryStrategy retryStrategy;

        BaseTask() {
            this.retryStrategy = new RetryStrategy(FullUpdateInitializer.this.maxRetries, FullUpdateInitializer.this.waitDurationMillis);
        }

        @Override
        public CallResult call() throws Exception {
            return this.retryStrategy.exec();
        }

        abstract ObjectMapping doTask() throws Exception;

        private final class RetryStrategy {
            private int retryStrategyMaxRetries = 0;
            private final int retryStrategyWaitDurationMillis;

            private RetryStrategy(int retryStrategyMaxRetries, int retryStrategyWaitDurationMillis) {
                this.retryStrategyMaxRetries = retryStrategyMaxRetries;
                this.retryStrategyWaitDurationMillis = retryStrategyWaitDurationMillis > 0 ? retryStrategyWaitDurationMillis : 1000;
            }

            public CallResult exec() {
                Throwable exception = null;
                try {
                    for (int i = 0; i < this.retryStrategyMaxRetries; ++i) {
                        try {
                            return new CallResult(BaseTask.this.doTask());
                        }
                        catch (TException ex) {
                            LOGGER.debug("Failed to execute task on " + (i + 1) + " attempts. Sleeping for " + this.retryStrategyWaitDurationMillis + " ms. Exception: " + ex.toString(), (Throwable)ex);
                            exception = ex;
                            try {
                                Thread.sleep(this.retryStrategyWaitDurationMillis);
                                continue;
                            }
                            catch (InterruptedException ignored) {
                                LOGGER.warn("Interrupted during update fetch during iteration " + (i + 1));
                                break;
                            }
                        }
                    }
                }
                catch (Exception ex) {
                    exception = ex;
                }
                LOGGER.error("Failed to execute task", exception);
                FullUpdateInitializer.this.threadPool.shutdown();
                return new CallResult((Exception)exception);
            }
        }
    }

    private static final class CallResult {
        private final Exception failure;
        private final boolean successStatus;
        private final ObjectMapping objectMapping;

        CallResult(Exception ex) {
            this.failure = ex;
            this.successStatus = false;
            this.objectMapping = emptyObjectMapping;
        }

        CallResult(ObjectMapping objectMapping) {
            this.failure = null;
            this.successStatus = true;
            this.objectMapping = objectMapping;
        }

        boolean success() {
            return this.successStatus;
        }

        ObjectMapping getObjectMapping() {
            return this.objectMapping;
        }

        public Exception getFailure() {
            return this.failure;
        }
    }

    private static final class ObjectMapping {
        private final Map<String, Set<String>> objects;

        ObjectMapping(Map<String, Set<String>> objects) {
            this.objects = objects;
        }

        ObjectMapping(String authObject, String path) {
            Set<String> values = Collections.singleton(FullUpdateInitializer.safeIntern(path));
            this.objects = ImmutableMap.of((Object)authObject, values);
        }

        ObjectMapping(String authObject, Collection<String> paths) {
            HashSet<String> values = new HashSet<String>(paths);
            this.objects = ImmutableMap.of((Object)authObject, values);
        }

        Map<String, Set<String>> getObjects() {
            return this.objects;
        }
    }
}

