/*
 * Decompiled with CFR 0.152.
 */
package org.apache.sentry.provider.db.service.persistent;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import org.apache.commons.lang.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hive.metastore.TableType;
import org.apache.hadoop.hive.metastore.api.NotificationEvent;
import org.apache.hadoop.hive.metastore.api.PrincipalType;
import org.apache.hive.hcatalog.messaging.HCatEventMessage;
import org.apache.sentry.api.common.SentryServiceUtil;
import org.apache.sentry.api.service.thrift.SentryMetrics;
import org.apache.sentry.api.service.thrift.TSentryAuthorizable;
import org.apache.sentry.binding.hive.conf.HiveAuthzConf;
import org.apache.sentry.binding.metastore.messaging.json.SentryJSONAddPartitionMessage;
import org.apache.sentry.binding.metastore.messaging.json.SentryJSONAlterDatabaseMessage;
import org.apache.sentry.binding.metastore.messaging.json.SentryJSONAlterPartitionMessage;
import org.apache.sentry.binding.metastore.messaging.json.SentryJSONAlterTableMessage;
import org.apache.sentry.binding.metastore.messaging.json.SentryJSONCreateDatabaseMessage;
import org.apache.sentry.binding.metastore.messaging.json.SentryJSONCreateTableMessage;
import org.apache.sentry.binding.metastore.messaging.json.SentryJSONDropDatabaseMessage;
import org.apache.sentry.binding.metastore.messaging.json.SentryJSONDropPartitionMessage;
import org.apache.sentry.binding.metastore.messaging.json.SentryJSONDropTableMessage;
import org.apache.sentry.binding.metastore.messaging.json.SentryJSONMessageDeserializer;
import org.apache.sentry.core.common.exception.SentryInvalidHMSEventException;
import org.apache.sentry.core.common.exception.SentryInvalidInputException;
import org.apache.sentry.core.common.exception.SentryNoSuchObjectException;
import org.apache.sentry.core.common.utils.PathUtils;
import org.apache.sentry.hdfs.PathsUpdate;
import org.apache.sentry.hdfs.PermissionsUpdate;
import org.apache.sentry.hdfs.SentryMalformedPathException;
import org.apache.sentry.hdfs.UniquePathsUpdate;
import org.apache.sentry.hdfs.Updateable;
import org.apache.sentry.hdfs.service.thrift.TPrivilegeChanges;
import org.apache.sentry.hdfs.service.thrift.TPrivilegePrincipal;
import org.apache.sentry.hdfs.service.thrift.TPrivilegePrincipalType;
import org.apache.sentry.provider.db.service.persistent.HMSFollower;
import org.apache.sentry.provider.db.service.persistent.SentryHMSOwnerHandler;
import org.apache.sentry.provider.db.service.persistent.SentryStoreInterface;
import org.apache.sentry.service.common.SentryOwnerPrivilegeType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sentry.com.codahale.metrics.MetricRegistry;
import sentry.com.codahale.metrics.Timer;

final class NotificationProcessor {
    private static final Logger LOGGER = LoggerFactory.getLogger(NotificationProcessor.class);
    private final SentryStoreInterface sentryStore;
    private final SentryJSONMessageDeserializer deserializer;
    private final String authServerName;
    private boolean syncStoreOnCreate = false;
    private boolean syncStoreOnDrop = false;
    private final boolean hdfsSyncEnabled;
    private final boolean OWNER_GRANT_OPTION;
    private final SentryOwnerPrivilegeType OWNER_PRIVILEGE_TYPE;
    private final SentryHMSOwnerHandler ownerPrivilegeProcessor;

    NotificationProcessor(SentryStoreInterface sentryStore, String authServerName, Configuration conf, SentryHMSOwnerHandler ownershipHandler) {
        this.sentryStore = sentryStore;
        this.deserializer = new SentryJSONMessageDeserializer();
        this.authServerName = authServerName;
        this.syncStoreOnCreate = Boolean.parseBoolean(conf.get(HiveAuthzConf.AuthzConfVars.AUTHZ_SYNC_CREATE_WITH_POLICY_STORE.getVar(), HiveAuthzConf.AuthzConfVars.AUTHZ_SYNC_CREATE_WITH_POLICY_STORE.getDefault()));
        this.syncStoreOnDrop = Boolean.parseBoolean(conf.get(HiveAuthzConf.AuthzConfVars.AUTHZ_SYNC_DROP_WITH_POLICY_STORE.getVar(), HiveAuthzConf.AuthzConfVars.AUTHZ_SYNC_DROP_WITH_POLICY_STORE.getDefault()));
        this.hdfsSyncEnabled = SentryServiceUtil.isHDFSSyncEnabledNoCache((Configuration)conf);
        this.ownerPrivilegeProcessor = ownershipHandler;
        this.OWNER_PRIVILEGE_TYPE = SentryOwnerPrivilegeType.get((Configuration)conf);
        this.OWNER_GRANT_OPTION = this.OWNER_PRIVILEGE_TYPE == SentryOwnerPrivilegeType.ALL_WITH_GRANT;
    }

    private static List<String> splitPath(String path) {
        return Lists.newArrayList((Object[])PathUtils.splitPath((String)path));
    }

    @VisibleForTesting
    static Updateable.Update getPermUpdatableOnDrop(TSentryAuthorizable authorizable) throws SentryInvalidInputException {
        PermissionsUpdate update = new PermissionsUpdate(1L, false);
        String authzObj = SentryServiceUtil.getAuthzObj((TSentryAuthorizable)authorizable);
        update.addPrivilegeUpdate(authzObj).putToDelPrivileges(new TPrivilegePrincipal(TPrivilegePrincipalType.ROLE, "__ALL_PRIVS__"), "__ALL_PRIVS__");
        return update;
    }

    @VisibleForTesting
    String getAuthServerName() {
        return this.authServerName;
    }

    @VisibleForTesting
    static Updateable.Update getPermUpdatableOnRename(TSentryAuthorizable oldAuthorizable, TSentryAuthorizable newAuthorizable) throws SentryInvalidInputException {
        String oldAuthz = SentryServiceUtil.getAuthzObj((TSentryAuthorizable)oldAuthorizable);
        String newAuthz = SentryServiceUtil.getAuthzObj((TSentryAuthorizable)newAuthorizable);
        PermissionsUpdate update = new PermissionsUpdate(1L, false);
        TPrivilegeChanges privUpdate = update.addPrivilegeUpdate("__RENAME_PRIV__");
        privUpdate.putToAddPrivileges(new TPrivilegePrincipal(TPrivilegePrincipalType.AUTHZ_OBJ, newAuthz), newAuthz);
        privUpdate.putToDelPrivileges(new TPrivilegePrincipal(TPrivilegePrincipalType.AUTHZ_OBJ, oldAuthz), oldAuthz);
        return update;
    }

    @VisibleForTesting
    void setSyncStoreOnCreate(boolean value) {
        this.syncStoreOnCreate = value;
    }

    @VisibleForTesting
    void setSyncStoreOnDrop(boolean value) {
        this.syncStoreOnDrop = value;
    }

    boolean processNotificationEvent(NotificationEvent event) throws Exception {
        LOGGER.debug("Processing event with id:{} and Type:{}", (Object)event.getEventId(), (Object)event.getEventType());
        HCatEventMessage.EventType eventType = HCatEventMessage.EventType.valueOf((String)event.getEventType());
        Timer timer = SentryMetrics.getInstance().getTimer(MetricRegistry.name(HMSFollower.class, eventType.toString().toLowerCase()));
        try (Timer.Context ignored = timer.time();){
            switch (eventType) {
                case CREATE_DATABASE: {
                    boolean bl = this.processCreateDatabase(event);
                    return bl;
                }
                case DROP_DATABASE: {
                    boolean bl = this.processDropDatabase(event);
                    return bl;
                }
                case ALTER_DATABASE: {
                    boolean bl = this.processAlterDatabase(event);
                    return bl;
                }
                case CREATE_TABLE: {
                    boolean bl = this.processCreateTable(event);
                    return bl;
                }
                case DROP_TABLE: {
                    boolean bl = this.processDropTable(event);
                    return bl;
                }
                case ALTER_TABLE: {
                    boolean bl = this.processAlterTable(event);
                    return bl;
                }
                case ADD_PARTITION: {
                    boolean bl = this.processAddPartition(event);
                    return bl;
                }
                case DROP_PARTITION: {
                    boolean bl = this.processDropPartition(event);
                    return bl;
                }
                case ALTER_PARTITION: {
                    boolean bl = this.processAlterPartition(event);
                    return bl;
                }
            }
            LOGGER.error("Notification with ID:{} has invalid event type: {}", (Object)event.getEventId(), (Object)event.getEventType());
            boolean bl = false;
            return bl;
        }
    }

    private boolean processCreateDatabase(NotificationEvent event) throws Exception {
        SentryJSONCreateDatabaseMessage message = this.deserializer.getCreateDatabaseMessage(event.getMessage());
        String dbName = message.getDB();
        String location = message.getLocation();
        if (dbName == null || location == null) {
            LOGGER.warn("Create database event has incomplete information. dbName: {} location: {}", (Object)StringUtils.defaultIfBlank((String)dbName, (String)"null"), (Object)StringUtils.defaultIfBlank((String)location, (String)"null"));
            return false;
        }
        if (this.syncStoreOnCreate) {
            this.dropSentryDbPrivileges(dbName, event);
        }
        if (this.OWNER_PRIVILEGE_TYPE != SentryOwnerPrivilegeType.NONE) {
            PrincipalType ownerType = message.getOwnerType();
            String ownerName = message.getOwnerName();
            if (ownerType == null || Strings.isNullOrEmpty((String)ownerName)) {
                LOGGER.warn(String.format("Create database event has incomplete OWNER information. dbName: %s, ownerType: %s, ownerName: %s ", dbName, ownerType, ownerName));
            } else {
                try {
                    this.grantDatabaseOwnerPrivilege(message);
                }
                catch (Exception e) {
                    LOGGER.error("Cannot grant OWNER privileges on database '{}': {}", (Object)dbName, (Object)e.getMessage());
                }
            }
        }
        if (this.hdfsSyncEnabled) {
            List<String> locations = Collections.singletonList(location);
            this.addPaths(dbName, locations, event);
            return true;
        }
        return false;
    }

    private boolean processDropDatabase(NotificationEvent event) throws Exception {
        SentryJSONDropDatabaseMessage dropDatabaseMessage = this.deserializer.getDropDatabaseMessage(event.getMessage());
        String dbName = dropDatabaseMessage.getDB();
        String location = dropDatabaseMessage.getLocation();
        if (dbName == null) {
            LOGGER.warn("Drop database event has incomplete information: dbName = null");
            return false;
        }
        if (this.syncStoreOnDrop) {
            this.dropSentryDbPrivileges(dbName, event);
        }
        if (this.hdfsSyncEnabled) {
            List<String> locations = Collections.singletonList(location);
            this.removePaths(dbName, locations, event);
            return true;
        }
        return false;
    }

    private boolean processCreateTable(NotificationEvent event) throws Exception {
        SentryJSONCreateTableMessage createTableMessage = this.deserializer.getCreateTableMessage(event.getMessage());
        String dbName = createTableMessage.getDB();
        String tableName = createTableMessage.getTable();
        String tableType = createTableMessage.getTableType();
        String location = createTableMessage.getLocation();
        if (dbName == null || tableName == null || !this.isVirtualView(tableType) && location == null) {
            LOGGER.warn(String.format("Create table event has incomplete information. dbName = %s, tableName = %s, location = %s", StringUtils.defaultIfBlank((String)dbName, (String)"null"), StringUtils.defaultIfBlank((String)tableName, (String)"null"), StringUtils.defaultIfBlank((String)location, (String)"null")));
            return false;
        }
        if (this.syncStoreOnCreate) {
            this.dropSentryTablePrivileges(dbName, tableName, event);
        }
        if (this.OWNER_PRIVILEGE_TYPE != SentryOwnerPrivilegeType.NONE) {
            PrincipalType ownerType = createTableMessage.getOwnerType();
            String ownerName = createTableMessage.getOwnerName();
            if (ownerType == null || Strings.isNullOrEmpty((String)ownerName)) {
                LOGGER.warn(String.format("Create table event has incomplete OWNER information. dbName: %s, tableType: %s, tableName: %s, ownerType: %s, ownerName: %s ", dbName, tableType, tableName, ownerType, ownerName));
            } else {
                try {
                    this.grantTableOwnerPrivilege(createTableMessage);
                }
                catch (Exception e) {
                    LOGGER.error("Cannot grant OWNER privileges on table/view '{}.{}': {}", new Object[]{dbName, tableName, e.getMessage()});
                }
            }
        }
        if (!this.isVirtualView(tableType) && this.hdfsSyncEnabled) {
            String authzObj = SentryServiceUtil.getAuthzObj((String)dbName, (String)tableName);
            List<String> locations = Collections.singletonList(location);
            this.addPaths(authzObj, locations, event);
            return true;
        }
        return false;
    }

    private boolean processDropTable(NotificationEvent event) throws Exception {
        SentryJSONDropTableMessage dropTableMessage = this.deserializer.getDropTableMessage(event.getMessage());
        String dbName = dropTableMessage.getDB();
        String tableName = dropTableMessage.getTable();
        if (dbName == null || tableName == null) {
            LOGGER.warn("Drop table event has incomplete information. dbName: {}, tableName: {}", (Object)StringUtils.defaultIfBlank((String)dbName, (String)"null"), (Object)StringUtils.defaultIfBlank((String)tableName, (String)"null"));
            return false;
        }
        if (this.syncStoreOnDrop) {
            this.dropSentryTablePrivileges(dbName, tableName, event);
        }
        if (this.hdfsSyncEnabled) {
            String authzObj = SentryServiceUtil.getAuthzObj((String)dbName, (String)tableName);
            this.removeAllPaths(authzObj, event);
            return true;
        }
        return false;
    }

    private boolean processAlterDatabase(NotificationEvent event) throws Exception {
        SentryJSONAlterDatabaseMessage alterDatabaseMessage = this.deserializer.getAlterDatabaseMessage(event.getMessage());
        String oldDbName = alterDatabaseMessage.getOldDbName();
        String newDbName = alterDatabaseMessage.getNewDbName();
        String oldLocation = alterDatabaseMessage.getOldLocation();
        String newLocation = alterDatabaseMessage.getNewLocation();
        PrincipalType oldOwnerType = alterDatabaseMessage.getOldOwnerType();
        PrincipalType newOwnerType = alterDatabaseMessage.getNewOwnerType();
        String oldOwnerName = alterDatabaseMessage.getOldOwnerName();
        String newOwnerName = alterDatabaseMessage.getNewOwnerName();
        if (oldDbName == null || newDbName == null || oldLocation == null || newLocation == null) {
            LOGGER.warn(String.format("Alter database notification ignored since event has incomplete information. oldDbName = %s, oldLocation = %s, newDbName = %s, newLocation = %s", StringUtils.defaultIfBlank((String)oldDbName, (String)"null"), StringUtils.defaultIfBlank((String)oldLocation, (String)"null"), StringUtils.defaultIfBlank((String)newDbName, (String)"null"), StringUtils.defaultIfBlank((String)newLocation, (String)"null")));
            return false;
        }
        if (oldDbName.equals(newDbName) && oldLocation.equals(newLocation) && Objects.equals(oldOwnerType, newOwnerType) && this.isOwnerNameEquals(oldOwnerType, oldOwnerName, newOwnerName)) {
            LOGGER.debug(String.format("Alter database notification ignored as neither name nor location nor owner has changed: oldAuthzObj = %s, oldLocation = %s, newAuthzObj = %s, newLocation = %s, oldOwnerType = %s, newOwnerType = %s, oldOwnerName = %s, newOwnerName = %s", oldDbName, oldLocation, newDbName, newLocation, oldOwnerType, newOwnerType, oldOwnerName, newOwnerName));
            return false;
        }
        if (!(oldOwnerType == null || oldOwnerType == newOwnerType && this.isOwnerNameEquals(oldOwnerType, oldOwnerName, newOwnerName))) {
            if (newOwnerType == null || Strings.isNullOrEmpty((String)newOwnerName)) {
                LOGGER.warn(String.format("Alter database event has incomplete OWNER information. dbName: %s, newOwnerType: %s, newOwnerName: %s ", newDbName, newOwnerType, newOwnerName));
            } else {
                try {
                    this.transferDatabaseOwnerPrivilege(alterDatabaseMessage);
                }
                catch (Exception e) {
                    LOGGER.error(String.format("Cannot transfer OWNER privileges on database '%s' to '%s.%s': %s", newDbName, newOwnerType, newOwnerName, e.getMessage()));
                }
            }
        }
        if (!this.hdfsSyncEnabled) {
            return false;
        }
        return false;
    }

    private boolean processAlterTable(NotificationEvent event) throws Exception {
        String newAuthzObj;
        String oldAuthzObj;
        SentryJSONAlterTableMessage alterTableMessage = this.deserializer.getAlterTableMessage(event.getMessage());
        String oldDbName = alterTableMessage.getOldDbName();
        String oldTableName = alterTableMessage.getOldTableName();
        String newDbName = alterTableMessage.getNewDbName();
        String newTableName = alterTableMessage.getNewTableName();
        PrincipalType oldOwnerType = alterTableMessage.getOldOwnerType();
        PrincipalType newOwnerType = alterTableMessage.getNewOwnerType();
        String oldOwnerName = alterTableMessage.getOldOwnerName();
        String newOwnerName = alterTableMessage.getNewOwnerName();
        String oldTableType = alterTableMessage.getOldTableType();
        String newTableType = alterTableMessage.getNewTableType();
        if (oldDbName == null || oldTableName == null || newDbName == null || newTableName == null) {
            LOGGER.warn(String.format("Alter table notification ignored since event has incomplete information. oldDbName = %s, oldTableName = %s, newDbName = %s, newTableName = %s", StringUtils.defaultIfBlank((String)oldDbName, (String)"null"), StringUtils.defaultIfBlank((String)oldTableName, (String)"null"), StringUtils.defaultIfBlank((String)newDbName, (String)"null"), StringUtils.defaultIfBlank((String)newTableName, (String)"null")));
            return false;
        }
        if (!newDbName.equalsIgnoreCase(oldDbName) || !oldTableName.equalsIgnoreCase(newTableName)) {
            try {
                this.renamePrivileges(oldDbName, oldTableName, oldTableType, newDbName, newTableName, newTableType);
            }
            catch (SentryNoSuchObjectException e) {
                LOGGER.debug("Rename Sentry privilege ignored as there are no privileges on the table: {}.{}", (Object)oldDbName, (Object)oldTableName);
            }
            catch (Exception e) {
                LOGGER.info("Could not process Alter table event. Event: {}", (Object)event.toString(), (Object)e);
                return false;
            }
        }
        String oldLocation = alterTableMessage.getOldLocation();
        String newLocation = alterTableMessage.getNewLocation();
        if (!this.isVirtualView(oldTableType) && oldLocation == null || !this.isVirtualView(newTableType) && newLocation == null) {
            LOGGER.warn(String.format("HMS path update ignored since event has incomplete path information. oldLocation = %s, newLocation = %s", StringUtils.defaultIfBlank((String)oldLocation, (String)"null"), StringUtils.defaultIfBlank((String)newLocation, (String)"null")));
            return false;
        }
        if (oldDbName.equals(newDbName) && oldTableName.equals(newTableName) && StringUtils.equals((String)oldLocation, (String)newLocation) && Objects.equals(oldOwnerType, newOwnerType) && this.isOwnerNameEquals(oldOwnerType, oldOwnerName, newOwnerName)) {
            LOGGER.debug(String.format("Alter table notification ignored as neither name nor location nor owner has changed: oldAuthzObj = %s, oldLocation = %s, newAuthzObj = %s, newLocation = %s, oldOwnerType = %s, newOwnerType = %s, oldOwnerName = %s, newOwnerName = %s", oldDbName + "." + oldTableName, oldLocation, newDbName + "." + newTableName, newLocation, oldOwnerType, newOwnerType, oldOwnerName, newOwnerName));
            return false;
        }
        if (!StringUtils.equalsIgnoreCase((String)oldTableType, (String)newTableType)) {
            LOGGER.warn(String.format("Alter table notification ignored because old and new table types are different: oldAuthzObj =%s.%s, oldTableType = %s, newAuthzObj= %s.%s newTableType = %s", oldDbName, oldTableName, oldTableType, newDbName, newTableName, newTableType));
            return false;
        }
        if (!(oldOwnerType == null || oldOwnerType == newOwnerType && this.isOwnerNameEquals(oldOwnerType, oldOwnerName, newOwnerName))) {
            if (newOwnerType == null || Strings.isNullOrEmpty((String)newOwnerName)) {
                LOGGER.warn(String.format("Alter table event has incomplete OWNER information. dbName: %s, tableType: %s, tableName: %s, newOwnerType: %s, newOwnerName: %s ", newDbName, newTableType, newTableName, newOwnerType, newOwnerName));
            } else {
                try {
                    this.transferTableOwnerPrivilege(alterTableMessage);
                }
                catch (Exception e) {
                    LOGGER.error(String.format("Cannot transfer OWNER privileges on table/view '%s.%s' to '%s.%s': %s", newDbName, newTableName, newOwnerType, newOwnerName, e.getMessage()));
                }
            }
        }
        if (!this.hdfsSyncEnabled) {
            return false;
        }
        if (!(this.isVirtualView(oldTableType) || (oldAuthzObj = oldDbName + "." + oldTableName).equalsIgnoreCase(newAuthzObj = newDbName + "." + newTableName) && oldLocation.equalsIgnoreCase(newLocation))) {
            this.renameAuthzPath(oldAuthzObj, newAuthzObj, oldLocation, newLocation, event);
            return true;
        }
        return false;
    }

    private boolean processAddPartition(NotificationEvent event) throws Exception {
        if (!this.hdfsSyncEnabled) {
            return false;
        }
        SentryJSONAddPartitionMessage addPartitionMessage = this.deserializer.getAddPartitionMessage(event.getMessage());
        String dbName = addPartitionMessage.getDB();
        String tableName = addPartitionMessage.getTable();
        List locations = addPartitionMessage.getLocations();
        if (dbName == null || tableName == null || locations == null) {
            LOGGER.warn(String.format("Add partition event has incomplete information. dbName = %s, tableName = %s, locations = %s", StringUtils.defaultIfBlank((String)dbName, (String)"null"), StringUtils.defaultIfBlank((String)tableName, (String)"null"), locations != null ? locations.toString() : "null"));
            return false;
        }
        String authzObj = SentryServiceUtil.getAuthzObj((String)dbName, (String)tableName);
        this.addPaths(authzObj, locations, event);
        return true;
    }

    private boolean processDropPartition(NotificationEvent event) throws Exception {
        if (!this.hdfsSyncEnabled) {
            return false;
        }
        SentryJSONDropPartitionMessage dropPartitionMessage = this.deserializer.getDropPartitionMessage(event.getMessage());
        String dbName = dropPartitionMessage.getDB();
        String tableName = dropPartitionMessage.getTable();
        List locations = dropPartitionMessage.getLocations();
        if (dbName == null || tableName == null || locations == null) {
            LOGGER.warn(String.format("Drop partition event has incomplete information. dbName = %s, tableName = %s, location = %s", StringUtils.defaultIfBlank((String)dbName, (String)"null"), StringUtils.defaultIfBlank((String)tableName, (String)"null"), locations != null ? locations.toString() : "null"));
            return false;
        }
        String authzObj = SentryServiceUtil.getAuthzObj((String)dbName, (String)tableName);
        this.removePaths(authzObj, locations, event);
        return true;
    }

    private boolean processAlterPartition(NotificationEvent event) throws Exception {
        if (!this.hdfsSyncEnabled) {
            return false;
        }
        SentryJSONAlterPartitionMessage alterPartitionMessage = this.deserializer.getAlterPartitionMessage(event.getMessage());
        String dbName = alterPartitionMessage.getDB();
        String tableName = alterPartitionMessage.getTable();
        String oldLocation = alterPartitionMessage.getOldLocation();
        String newLocation = alterPartitionMessage.getNewLocation();
        if (dbName == null || tableName == null || oldLocation == null || newLocation == null) {
            LOGGER.warn(String.format("Alter partition event has incomplete information. dbName = %s, tableName = %s, oldLocation = %s, newLocation = %s", StringUtils.defaultIfBlank((String)dbName, (String)"null"), StringUtils.defaultIfBlank((String)tableName, (String)"null"), StringUtils.defaultIfBlank((String)oldLocation, (String)"null"), StringUtils.defaultIfBlank((String)newLocation, (String)"null")));
            return false;
        }
        if (oldLocation.equals(newLocation)) {
            LOGGER.debug(String.format("Alter partition notification ignored aslocation has not changed: AuthzObj = %s, Location = %s", dbName + ".." + tableName, oldLocation));
            return false;
        }
        String oldAuthzObj = dbName + "." + tableName;
        this.renameAuthzPath(oldAuthzObj, oldAuthzObj, oldLocation, newLocation, event);
        return true;
    }

    private void addPaths(String authzObj, Collection<String> locations, NotificationEvent event) throws Exception {
        authzObj = authzObj.toLowerCase();
        UniquePathsUpdate update = new UniquePathsUpdate(event, false);
        HashSet<String> paths = new HashSet<String>(locations.size());
        for (String location : locations) {
            String pathTree = this.getPath(location);
            if (pathTree == null) {
                LOGGER.debug("HMS Path Update [OP : addPath, authzObj : " + authzObj + ", path : " + location + "] - nothing to add, notification event ID: " + event.getEventId() + "]");
                continue;
            }
            LOGGER.debug("HMS Path Update [OP : addPath, authzObj : " + authzObj + ", path : " + location + ", notification event ID: " + event.getEventId() + "]");
            update.newPathChange(authzObj).addToAddPaths(NotificationProcessor.splitPath(pathTree));
            paths.add(pathTree);
        }
        if (!paths.isEmpty()) {
            this.sentryStore.addAuthzPathsMapping(authzObj, paths, update);
        } else {
            LOGGER.info("Received empty paths for {}, not updating sentry store", (Object)authzObj);
        }
    }

    private void removePaths(String authzObj, Collection<String> locations, NotificationEvent event) throws Exception {
        authzObj = authzObj.toLowerCase();
        UniquePathsUpdate update = new UniquePathsUpdate(event, false);
        HashSet<String> paths = new HashSet<String>(locations.size());
        for (String location : locations) {
            String pathTree = this.getPath(location);
            if (pathTree == null) {
                LOGGER.debug("HMS Path Update [OP : removePath, authzObj : " + authzObj + ", path : " + location + "] - nothing to remove, notification event ID: " + event.getEventId() + "]");
                continue;
            }
            LOGGER.debug("HMS Path Update [OP : removePath, authzObj : " + authzObj + ", path : " + location + ", notification event ID: " + event.getEventId() + "]");
            update.newPathChange(authzObj).addToDelPaths(NotificationProcessor.splitPath(pathTree));
            paths.add(pathTree);
        }
        if (!paths.isEmpty()) {
            this.sentryStore.deleteAuthzPathsMapping(authzObj, paths, update);
        } else {
            LOGGER.info("Received empty paths for {}, not updating sentry store", (Object)authzObj);
        }
    }

    private void removeAllPaths(String authzObj, NotificationEvent event) throws Exception {
        authzObj = authzObj.toLowerCase();
        LOGGER.debug("HMS Path Update [OP : removeAllPaths, authzObj : " + authzObj + ", notification event ID: " + event.getEventId() + "]");
        UniquePathsUpdate update = new UniquePathsUpdate(event, false);
        update.newPathChange(authzObj).addToDelPaths((List)Lists.newArrayList((Object[])new String[]{"__ALL_PATHS__"}));
        this.sentryStore.deleteAllAuthzPathsMapping(authzObj, update);
    }

    private void renameAuthzPath(String oldAuthzObj, String newAuthzObj, String oldLocation, String newLocation, NotificationEvent event) throws Exception {
        oldAuthzObj = oldAuthzObj.toLowerCase();
        newAuthzObj = newAuthzObj.toLowerCase();
        String oldPathTree = this.getPath(oldLocation);
        String newPathTree = this.getPath(newLocation);
        LOGGER.debug("HMS Path Update [OP : renameAuthzObject, oldAuthzObj : " + oldAuthzObj + ", newAuthzObj : " + newAuthzObj + ", oldLocation : " + oldLocation + ", newLocation : " + newLocation + ", notification event ID: " + event.getEventId() + "]");
        if (!oldAuthzObj.equalsIgnoreCase(newAuthzObj)) {
            if (oldPathTree != null && newPathTree != null) {
                UniquePathsUpdate update = new UniquePathsUpdate(event, false);
                update.newPathChange(oldAuthzObj).addToDelPaths(NotificationProcessor.splitPath(oldPathTree));
                update.newPathChange(newAuthzObj).addToAddPaths(NotificationProcessor.splitPath(newPathTree));
                if (oldLocation.equals(newLocation)) {
                    this.sentryStore.renameAuthzObj(oldAuthzObj, newAuthzObj, update);
                } else {
                    this.sentryStore.renameAuthzPathsMapping(oldAuthzObj, newAuthzObj, oldPathTree, newPathTree, update);
                }
            } else {
                this.updateAuthzPathsMapping(oldAuthzObj, oldPathTree, newAuthzObj, newPathTree, event);
            }
        } else if (!oldLocation.equals(newLocation)) {
            if (oldPathTree != null && newPathTree != null) {
                UniquePathsUpdate update = new UniquePathsUpdate(event, false);
                update.newPathChange(oldAuthzObj).addToDelPaths(NotificationProcessor.splitPath(oldPathTree));
                update.newPathChange(oldAuthzObj).addToAddPaths(NotificationProcessor.splitPath(newPathTree));
                this.sentryStore.updateAuthzPathsMapping(oldAuthzObj, oldPathTree, newPathTree, update);
            } else {
                this.updateAuthzPathsMapping(oldAuthzObj, oldPathTree, newAuthzObj, newPathTree, event);
            }
        } else {
            LOGGER.error("Update Notification for Auhorizable object {}, with no change, skipping", (Object)oldAuthzObj);
            throw new SentryInvalidHMSEventException("Update Notification for Authorizable objectwith no change");
        }
    }

    private void updateAuthzPathsMapping(String oldAuthzObj, String oldPathTree, String newAuthzObj, String newPathTree, NotificationEvent event) throws Exception {
        if (oldPathTree != null) {
            UniquePathsUpdate update = new UniquePathsUpdate(event, false);
            update.newPathChange(oldAuthzObj).addToDelPaths(NotificationProcessor.splitPath(oldPathTree));
            this.sentryStore.deleteAuthzPathsMapping(oldAuthzObj, Collections.singleton(oldPathTree), update);
        } else if (newPathTree != null) {
            UniquePathsUpdate update = new UniquePathsUpdate(event, false);
            update.newPathChange(newAuthzObj).addToAddPaths(NotificationProcessor.splitPath(newPathTree));
            this.sentryStore.addAuthzPathsMapping(newAuthzObj, Collections.singleton(newPathTree), update);
        }
    }

    private String getPath(String path) {
        try {
            return PathsUpdate.parsePath((String)path);
        }
        catch (SentryMalformedPathException e) {
            LOGGER.error("Unexpected path while parsing {}", (Object)path, (Object)e);
            return null;
        }
    }

    private void dropSentryDbPrivileges(String dbName, NotificationEvent event) {
        try {
            TSentryAuthorizable authorizable = new TSentryAuthorizable(this.authServerName);
            authorizable.setDb(dbName);
            this.sentryStore.dropPrivilege(authorizable, NotificationProcessor.getPermUpdatableOnDrop(authorizable));
        }
        catch (SentryNoSuchObjectException e) {
            LOGGER.debug("Drop Sentry privilege ignored as there are no privileges on the database: {}", (Object)dbName);
        }
        catch (Exception e) {
            LOGGER.error("Could not process Drop database event.Event: " + event.toString(), (Throwable)e);
        }
    }

    private void dropSentryTablePrivileges(String dbName, String tableName, NotificationEvent event) {
        try {
            TSentryAuthorizable authorizable = new TSentryAuthorizable(this.authServerName);
            authorizable.setDb(dbName);
            authorizable.setTable(tableName);
            this.sentryStore.dropPrivilege(authorizable, NotificationProcessor.getPermUpdatableOnDrop(authorizable));
        }
        catch (SentryNoSuchObjectException e) {
            LOGGER.debug("Drop Sentry privilege ignored as there are no privileges on the table: {}.{}", (Object)dbName, (Object)tableName);
        }
        catch (Exception e) {
            LOGGER.error("Could not process Drop table event. Event: " + event.toString(), (Throwable)e);
        }
    }

    private void renamePrivileges(String oldDbName, String oldTableName, String oldTableType, String newDbName, String newTableName, String newTableType) throws Exception {
        Preconditions.checkArgument((boolean)StringUtils.equalsIgnoreCase((String)oldTableType, (String)newTableType), (Object)String.format("oldTableType and newTableType must be equal: %s != %s", oldTableType, newTableType));
        TSentryAuthorizable oldAuthorizable = new TSentryAuthorizable(this.authServerName);
        oldAuthorizable.setDb(oldDbName);
        oldAuthorizable.setTable(oldTableName);
        TSentryAuthorizable newAuthorizable = new TSentryAuthorizable(this.authServerName);
        newAuthorizable.setDb(newDbName);
        newAuthorizable.setTable(newTableName);
        Updateable.Update update = null;
        if (!this.isVirtualView(oldTableType)) {
            update = NotificationProcessor.getPermUpdatableOnRename(oldAuthorizable, newAuthorizable);
        }
        this.sentryStore.renamePrivilege(oldAuthorizable, newAuthorizable, update);
    }

    private void grantDatabaseOwnerPrivilege(SentryJSONCreateDatabaseMessage message) throws Exception {
        String dbName = message.getDB();
        PrincipalType ownerType = message.getOwnerType();
        String ownerName = message.getOwnerName();
        this.ownerPrivilegeProcessor.grantDatabaseOwnerPrivilege(dbName, ownerType, ownerName, this.OWNER_GRANT_OPTION);
    }

    private void grantTableOwnerPrivilege(SentryJSONCreateTableMessage message) throws Exception {
        String dbName = message.getDB();
        String tableType = message.getTableType();
        String tableName = message.getTable();
        PrincipalType ownerType = message.getOwnerType();
        String ownerName = message.getOwnerName();
        this.ownerPrivilegeProcessor.grantTableOwnerPrivilege(dbName, tableType, tableName, ownerType, ownerName, this.OWNER_GRANT_OPTION);
    }

    private void transferTableOwnerPrivilege(SentryJSONAlterTableMessage message) throws Exception {
        Preconditions.checkArgument((boolean)StringUtils.equalsIgnoreCase((String)message.getOldTableType(), (String)message.getNewTableType()), (Object)String.format("oldTableType and newTableType must be equal: %s != %s", message.getOldTableType(), message.getNewTableType()));
        String dbName = message.getNewDbName();
        String tableType = message.getNewTableType();
        String tableName = message.getNewTableName();
        PrincipalType ownerType = message.getNewOwnerType();
        String ownerName = message.getNewOwnerName();
        this.ownerPrivilegeProcessor.dropTableOwnerPrivileges(dbName, tableType, tableName);
        if (this.OWNER_PRIVILEGE_TYPE != SentryOwnerPrivilegeType.NONE) {
            this.ownerPrivilegeProcessor.grantTableOwnerPrivilege(dbName, tableType, tableName, ownerType, ownerName, this.OWNER_GRANT_OPTION);
        }
    }

    private void transferDatabaseOwnerPrivilege(SentryJSONAlterDatabaseMessage message) throws Exception {
        String dbName = message.getNewDbName();
        PrincipalType ownerType = message.getNewOwnerType();
        String ownerName = message.getNewOwnerName();
        this.ownerPrivilegeProcessor.dropDatabaseOwnerPrivileges(dbName);
        if (this.OWNER_PRIVILEGE_TYPE != SentryOwnerPrivilegeType.NONE) {
            this.ownerPrivilegeProcessor.grantDatabaseOwnerPrivilege(dbName, ownerType, ownerName, this.OWNER_GRANT_OPTION);
        }
    }

    private boolean isVirtualView(String tableType) {
        return !Strings.isNullOrEmpty((String)tableType) && tableType.equalsIgnoreCase(TableType.VIRTUAL_VIEW.name());
    }

    private boolean isOwnerNameEquals(PrincipalType ownerType, String oldOwnerName, String newOwnerName) {
        if (ownerType == PrincipalType.ROLE) {
            return StringUtils.equalsIgnoreCase((String)oldOwnerName, (String)newOwnerName);
        }
        return StringUtils.equals((String)oldOwnerName, (String)newOwnerName);
    }
}

