/*
 * Decompiled with CFR 0.152.
 */
package org.apache.sentry.provider.db.service.persistent;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.jdo.JDODataStoreException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hive.metastore.api.NotificationEvent;
import org.apache.sentry.api.common.SentryServiceUtil;
import org.apache.sentry.binding.hive.conf.HiveAuthzConf;
import org.apache.sentry.core.common.utils.PubSub;
import org.apache.sentry.provider.db.audit.SentryAuditLogger;
import org.apache.sentry.provider.db.service.persistent.CounterWait;
import org.apache.sentry.provider.db.service.persistent.LeaderStatusMonitor;
import org.apache.sentry.provider.db.service.persistent.NotificationProcessor;
import org.apache.sentry.provider.db.service.persistent.PathsImage;
import org.apache.sentry.provider.db.service.persistent.SentryHMSOwnerHandler;
import org.apache.sentry.provider.db.service.persistent.SentryStoreInterface;
import org.apache.sentry.service.thrift.HMSFollowerState;
import org.apache.sentry.service.thrift.HiveConnectionFactory;
import org.apache.sentry.service.thrift.HiveNotificationFetcher;
import org.apache.sentry.service.thrift.SentryHMSClient;
import org.apache.sentry.service.thrift.SentryServiceState;
import org.apache.sentry.service.thrift.SentryStateBank;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HMSFollower
implements Runnable,
AutoCloseable,
PubSub.Subscriber {
    private static final Logger LOGGER = LoggerFactory.getLogger(HMSFollower.class);
    private static final String FULL_UPDATE_TRIGGER = "FULL UPDATE TRIGGER: ";
    private static boolean connectedToHms = false;
    private static String SENTRY_SERVICE_USER = System.getProperty("user.name", "SentryService");
    private SentryHMSClient client;
    private final Configuration authzConf;
    private final SentryStoreInterface sentryStore;
    private final NotificationProcessor notificationProcessor;
    private boolean readyToServe;
    private final HiveNotificationFetcher notificationFetcher;
    private final boolean hdfsSyncEnabled;
    private final AtomicBoolean fullUpdateHMS = new AtomicBoolean(false);
    private final LeaderStatusMonitor leaderMonitor;
    private int sentryHMSFetchSize;
    private long hmsImageId = 0L;

    public HMSFollower(Configuration conf, SentryStoreInterface store, LeaderStatusMonitor leaderMonitor, HiveConnectionFactory hiveConnectionFactory) {
        this(conf, store, leaderMonitor, hiveConnectionFactory, null);
    }

    @VisibleForTesting
    public HMSFollower(Configuration conf, SentryStoreInterface store, LeaderStatusMonitor leaderMonitor, HiveConnectionFactory hiveConnectionFactory, String authServerName) {
        LOGGER.info("HMSFollower is being initialized");
        this.readyToServe = false;
        this.authzConf = conf;
        this.leaderMonitor = leaderMonitor;
        this.sentryStore = store;
        if (authServerName == null) {
            authServerName = conf.get(HiveAuthzConf.AuthzConfVars.AUTHZ_SERVER_NAME.getVar(), conf.get(HiveAuthzConf.AuthzConfVars.AUTHZ_SERVER_NAME_DEPRECATED.getVar(), HiveAuthzConf.AuthzConfVars.AUTHZ_SERVER_NAME_DEPRECATED.getDefault()));
        }
        SentryHMSOwnerHandler ownershipHandler = new SentryHMSOwnerHandler(this.sentryStore, new SentryAuditLogger(conf), authServerName, SENTRY_SERVICE_USER);
        this.notificationProcessor = new NotificationProcessor(this.sentryStore, authServerName, this.authzConf, ownershipHandler);
        this.client = new SentryHMSClient(this.authzConf, hiveConnectionFactory);
        this.hdfsSyncEnabled = SentryServiceUtil.isHDFSSyncEnabledNoCache((Configuration)this.authzConf);
        this.notificationFetcher = new HiveNotificationFetcher(hiveConnectionFactory);
        if (conf.getBoolean("sentry.hdfs.sync.full-update-pubsub", false)) {
            LOGGER.info("FULL UPDATE TRIGGER: subscribing to topic " + PubSub.Topic.HDFS_SYNC_HMS.getName());
            PubSub.getInstance().subscribe(PubSub.Topic.HDFS_SYNC_HMS, (PubSub.Subscriber)this);
        }
        this.sentryHMSFetchSize = conf.getInt("sentry.hms.fetch.size", -1);
        if (this.sentryHMSFetchSize < 0) {
            LOGGER.info("Sentry will fetch from HMS max depth");
        } else {
            LOGGER.info("Sentry will fetch from HMS with depth of {}", (Object)this.sentryHMSFetchSize);
        }
        if (!this.hdfsSyncEnabled) {
            try {
                store.clearHmsPathInformation();
            }
            catch (Exception ex) {
                LOGGER.error("Failed to clear HMS path info", (Throwable)ex);
                LOGGER.error("Please manually clear data from SENTRY_PATH_CHANGE/AUTHZ_PATH/AUTHZ_PATHS_MAPPING tables.If not, HDFS ACL's will be inconsistent when HDFS sync feature is enabled back.");
            }
        }
    }

    @VisibleForTesting
    public static boolean isConnectedToHms() {
        return connectedToHms;
    }

    @VisibleForTesting
    void setSentryHmsClient(SentryHMSClient client) {
        this.client = client;
    }

    @Override
    public void close() {
        if (this.client != null) {
            try {
                this.client.disconnect();
                SentryStateBank.disableState("HMSFollower", HMSFollowerState.CONNECTED);
            }
            catch (Exception failure) {
                LOGGER.error("Failed to close the Sentry Hms Client", (Throwable)failure);
            }
        }
        this.notificationFetcher.close();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void run() {
        SentryStateBank.enableState("HMSFollower", HMSFollowerState.STARTED);
        try {
            long lastProcessedNotificationId;
            try {
                lastProcessedNotificationId = this.sentryStore.getLastProcessedNotificationID();
            }
            catch (Exception e) {
                LOGGER.error("Failed to get the last processed notification id from sentry store, Skipping the processing", (Throwable)e);
                SentryStateBank.disableState("HMSFollower", HMSFollowerState.STARTED);
                return;
            }
            this.wakeUpWaitingClientsForSync(lastProcessedNotificationId);
            if (!this.isLeader()) {
                this.close();
                return;
            }
            this.syncupWithHms(lastProcessedNotificationId);
        }
        finally {
            SentryStateBank.disableState("HMSFollower", HMSFollowerState.STARTED);
        }
    }

    private boolean isLeader() {
        return this.leaderMonitor == null || this.leaderMonitor.isLeader();
    }

    @VisibleForTesting
    String getAuthServerName() {
        return this.notificationProcessor.getAuthServerName();
    }

    void syncupWithHms(long notificationId) {
        try {
            this.client.connect();
            connectedToHms = true;
            SentryStateBank.enableState("HMSFollower", HMSFollowerState.CONNECTED);
        }
        catch (Throwable e) {
            LOGGER.error("HMSFollower cannot connect to HMS!!", e);
            return;
        }
        try {
            if (this.hdfsSyncEnabled) {
                if (this.isFullSnapshotRequired(notificationId)) {
                    this.createFullSnapshot();
                    return;
                }
            } else if (this.isSentryOutOfSync(notificationId)) {
                this.sentryStore.setLastProcessedNotificationID(0L);
                notificationId = 0L;
            }
            List<NotificationEvent> notifications = null;
            notifications = this.sentryHMSFetchSize < 0 ? this.notificationFetcher.fetchNotifications(notificationId) : this.notificationFetcher.fetchNotifications(notificationId, this.sentryHMSFetchSize);
            if (this.hdfsSyncEnabled && this.areNotificationsOutOfSync(notifications, notificationId)) {
                this.createFullSnapshot();
                return;
            }
            if (!this.readyToServe) {
                System.out.println("Sentry HMS support is ready");
                this.readyToServe = true;
            }
            this.processNotifications(notifications);
        }
        catch (TException e) {
            LOGGER.error("An error occurred while fetching HMS notifications: ", (Throwable)e);
            this.close();
        }
        catch (Throwable t) {
            LOGGER.error("Exception in HMSFollower! Caused by: " + t.getMessage(), t);
            this.close();
        }
    }

    private boolean isFullSnapshotRequired(long latestSentryNotificationId) throws Exception {
        if (this.sentryStore.isHmsNotificationEmpty()) {
            LOGGER.debug("Sentry Store has no HMS Notifications. Create Full HMS Snapshot. latest sentry notification Id = {}", (Object)latestSentryNotificationId);
            return true;
        }
        if (this.sentryStore.isAuthzPathsSnapshotEmpty()) {
            LOGGER.debug("HDFSSync is enabled and MAuthzPathsMapping table is empty. Need to request a full snapshot");
            return true;
        }
        if (this.isSentryOutOfSync(latestSentryNotificationId)) {
            return true;
        }
        if (this.fullUpdateHMS.compareAndSet(true, false)) {
            LOGGER.info("FULL UPDATE TRIGGER: initiating full HMS snapshot request");
            return true;
        }
        return false;
    }

    private boolean isSentryOutOfSync(long latestSentryNotificationId) throws Exception {
        long currentHmsNotificationId = this.notificationFetcher.getCurrentNotificationId();
        if (currentHmsNotificationId < latestSentryNotificationId) {
            LOGGER.info("The current notification ID on HMS = {} is less than the latest processed Sentry notification ID = {}. Sentry, Out-of-sync", (Object)currentHmsNotificationId, (Object)latestSentryNotificationId);
            return true;
        }
        return false;
    }

    private boolean areNotificationsOutOfSync(Collection<NotificationEvent> events, long latestProcessedId) {
        if (events.isEmpty()) {
            return false;
        }
        List eventList = (List)events;
        long firstNotificationId = ((NotificationEvent)eventList.get(0)).getEventId();
        if (firstNotificationId > latestProcessedId + 1L) {
            LOGGER.info("First HMS event notification Id = {} is greater than latest Sentry processednotification Id = {} + 1. Need to request a full HMS snapshot.", (Object)firstNotificationId, (Object)latestProcessedId);
            return true;
        }
        return false;
    }

    private long createFullSnapshot() throws Exception {
        LOGGER.debug("Attempting to take full HMS snapshot");
        Preconditions.checkState((!SentryStateBank.isEnabled("SentryService", SentryServiceState.FULL_UPDATE_RUNNING) ? 1 : 0) != 0, (Object)"HMSFollower shown loading full snapshot when it should not be.");
        try {
            SentryStateBank.enableState("SentryService", SentryServiceState.FULL_UPDATE_RUNNING);
            PathsImage snapshotInfo = this.client.getFullSnapshot();
            if (snapshotInfo.getPathImage().isEmpty()) {
                LOGGER.debug("Received empty path image from HMS while taking a full snapshot");
                long l = snapshotInfo.getId();
                return l;
            }
            if (!this.isLeader()) {
                LOGGER.info("Not persisting full snapshot since not a leader");
                long l = 0L;
                return l;
            }
            try {
                if (this.hdfsSyncEnabled) {
                    LOGGER.info("Persisting full snapshot for notification Id = {}", (Object)snapshotInfo.getId());
                    this.sentryStore.persistFullPathsImage(snapshotInfo.getPathImage(), snapshotInfo.getId());
                } else {
                    LOGGER.info("HDFSSync is disabled. Not Persisting full snapshot, but only setting last processed notification Id = {}", (Object)snapshotInfo.getId());
                    this.sentryStore.setLastProcessedNotificationID(snapshotInfo.getId());
                }
            }
            catch (Exception failure) {
                LOGGER.error("Received exception while persisting HMS path full snapshot ");
                throw failure;
            }
            this.wakeUpWaitingClientsForSync(snapshotInfo.getId());
            LOGGER.info("Sentry HMS support is ready");
            long l = snapshotInfo.getId();
            return l;
        }
        catch (Exception failure) {
            LOGGER.error("Received exception while creating HMS path full snapshot ");
            throw failure;
        }
        finally {
            SentryStateBank.disableState("SentryService", SentryServiceState.FULL_UPDATE_RUNNING);
        }
    }

    public void processNotifications(Collection<NotificationEvent> events) throws Exception {
        if (events.isEmpty()) {
            return;
        }
        for (NotificationEvent event : events) {
            boolean isNotificationProcessed = false;
            try {
                if (!this.isLeader()) {
                    LOGGER.debug("Not processing notifications since not a leader");
                    return;
                }
                isNotificationProcessed = this.notificationProcessor.processNotificationEvent(event);
            }
            catch (Exception e) {
                if (e.getCause() instanceof JDODataStoreException) {
                    LOGGER.info("Received JDO Storage Exception, Could be because of processing duplicate notification");
                    if (event.getEventId() <= this.sentryStore.getLastProcessedNotificationID()) {
                        LOGGER.error("Received event with Id: {} which is smaller then the ID persisted in store", (Object)event.getEventId());
                        break;
                    }
                }
                LOGGER.error("Processing the notification with ID:{} failed with exception {}", (Object)event.getEventId(), (Object)e);
            }
            if (!isNotificationProcessed) {
                try {
                    LOGGER.debug("Explicitly Persisting Notification ID = {} ", (Object)event.getEventId());
                    this.sentryStore.persistLastProcessedNotificationID(event.getEventId());
                }
                catch (Exception failure) {
                    LOGGER.error("Received exception while persisting the notification ID = {}", (Object)event.getEventId());
                    throw failure;
                }
            }
            this.wakeUpWaitingClientsForSync(event.getEventId());
        }
    }

    private void wakeUpWaitingClientsForSync(long eventId) {
        CounterWait counterWait = this.sentryStore.getCounterWait();
        LOGGER.debug("wakeUpWaitingClientsForSync: eventId = {}, hmsImageId = {}", (Object)eventId, (Object)this.hmsImageId);
        if (counterWait == null) {
            return;
        }
        long lastHMSSnapshotId = this.hmsImageId;
        try {
            lastHMSSnapshotId = this.sentryStore.getLastProcessedImageID();
            LOGGER.debug("wakeUpWaitingClientsForSync: lastHMSSnapshotId = {}", (Object)lastHMSSnapshotId);
        }
        catch (Exception e) {
            counterWait.update(eventId);
            LOGGER.error("Failed to get the last processed HMS image id from sentry store");
            return;
        }
        if (lastHMSSnapshotId > this.hmsImageId) {
            counterWait.reset(eventId);
            this.hmsImageId = lastHMSSnapshotId;
            LOGGER.debug("wakeUpWaitingClientsForSync: reset counterWait with eventId = {}, new hmsImageId = {}", (Object)eventId, (Object)this.hmsImageId);
        }
        LOGGER.debug("wakeUpWaitingClientsForSync: update counterWait with eventId = {}, hmsImageId = {}", (Object)eventId, (Object)this.hmsImageId);
        counterWait.update(eventId);
    }

    public void onMessage(PubSub.Topic topic, String message) {
        Preconditions.checkArgument((topic == PubSub.Topic.HDFS_SYNC_HMS ? 1 : 0) != 0, (String)"Unexpected topic %s instead of %s", (Object[])new Object[]{topic, PubSub.Topic.HDFS_SYNC_HMS});
        LOGGER.info("FULL UPDATE TRIGGER: Received [{}, {}] notification", (Object)topic, (Object)message);
        this.fullUpdateHMS.set(true);
    }
}

