/*
 * Decompiled with CFR 0.152.
 */
package org.apache.sentry.provider.db.service.persistent;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.util.List;
import java.util.concurrent.ThreadFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.security.SecurityUtil;
import org.apache.sentry.service.thrift.JaasConfiguration;
import org.apache.zookeeper.data.ACL;
import org.apache.zookeeper.data.Id;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sentry.org.apache.curator.framework.CuratorFramework;
import sentry.org.apache.curator.framework.CuratorFrameworkFactory;
import sentry.org.apache.curator.framework.api.ACLProvider;
import sentry.org.apache.curator.framework.api.BackgroundPathable;
import sentry.org.apache.curator.framework.imps.CuratorFrameworkState;
import sentry.org.apache.curator.framework.imps.DefaultACLProvider;
import sentry.org.apache.curator.framework.recipes.leader.LeaderSelector;
import sentry.org.apache.curator.framework.recipes.leader.LeaderSelectorListener;
import sentry.org.apache.curator.retry.ExponentialBackoffRetry;

public final class HAContext
implements AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(HAContext.class);
    private static HAContext serverHAContext = null;
    private static boolean aclUnChecked = true;
    private static final String SENTRY_ZK_JAAS_NAME = "SentryClient";
    private static final String SHUTDOWN_THREAD_NAME = "ha-context-shutdown";
    private final String zookeeperQuorum;
    private final String namespace;
    private final boolean zkSecure;
    private final List<ACL> saslACL;
    private final CuratorFramework curatorFramework;

    private HAContext(Configuration conf) throws IOException {
        ACLProvider aclProvider;
        this.zookeeperQuorum = conf.get("sentry.ha.zookeeper.quorum", "");
        int retriesMaxCount = conf.getInt("sentry.ha.zookeeper.session.retries.max.count", 3);
        int sleepMsBetweenRetries = conf.getInt("sentry.ha.zookeeper.session.sleep.between.retries.ms", 100);
        String ns = conf.get("sentry.ha.zookeeper.namespace", "sentry");
        this.namespace = ns.startsWith("/") ? ns.substring(1) : ns;
        this.zkSecure = conf.getBoolean("sentry.ha.zookeeper.security", false);
        this.validateConf();
        if (this.zkSecure) {
            LOGGER.info("Connecting to ZooKeeper with SASL/Kerberos and using 'sasl' ACLs");
            this.setJaasConfiguration(conf);
            System.setProperty("zookeeper.sasl.clientconfig", SENTRY_ZK_JAAS_NAME);
            this.saslACL = Lists.newArrayList();
            this.saslACL.add(new ACL(31, new Id("sasl", HAContext.getServicePrincipal(conf, "sentry.service.server.principal"))));
            this.saslACL.add(new ACL(31, new Id("sasl", HAContext.getServicePrincipal(conf, "sentry.zookeeper.client.principal"))));
            aclProvider = new SASLOwnerACLProvider();
            String allowConnect = conf.get("sentry.service.allow.connect");
            if (!Strings.isNullOrEmpty((String)allowConnect)) {
                for (String principal : allowConnect.split("\\s*,\\s*")) {
                    LOGGER.info("Adding acls for {}", (Object)principal);
                    this.saslACL.add(new ACL(31, new Id("sasl", principal)));
                }
            }
        } else {
            this.saslACL = null;
            LOGGER.info("Connecting to ZooKeeper without authentication");
            aclProvider = new DefaultACLProvider();
        }
        ExponentialBackoffRetry retryPolicy = new ExponentialBackoffRetry(sleepMsBetweenRetries, retriesMaxCount);
        this.curatorFramework = CuratorFrameworkFactory.builder().namespace(this.namespace).connectString(this.zookeeperQuorum).retryPolicy(retryPolicy).aclProvider(aclProvider).build();
    }

    private void start() {
        if (this.curatorFramework.getState() != CuratorFrameworkState.STARTED) {
            this.curatorFramework.start();
        }
    }

    static synchronized HAContext getHAContext(Configuration conf) throws IOException {
        if (serverHAContext != null) {
            return serverHAContext;
        }
        serverHAContext = new HAContext(conf);
        serverHAContext.start();
        ThreadFactory haContextShutdownThreadFactory = new ThreadFactoryBuilder().setDaemon(false).setNameFormat(SHUTDOWN_THREAD_NAME).build();
        Runtime.getRuntime().addShutdownHook(haContextShutdownThreadFactory.newThread(new Runnable(){

            @Override
            public void run() {
                LOGGER.info("ShutdownHook closing curator framework");
                try {
                    if (serverHAContext != null) {
                        serverHAContext.close();
                    }
                }
                catch (Throwable t) {
                    LOGGER.error("Error stopping curator framework", t);
                }
            }
        }));
        return serverHAContext;
    }

    public static HAContext getHAServerContext(Configuration conf) throws Exception {
        HAContext serverContext = HAContext.getHAContext(conf);
        serverContext.checkAndSetACLs();
        return serverContext;
    }

    public static void resetHAContext() {
        HAContext oldContext = serverHAContext;
        if (oldContext != null) {
            try {
                oldContext.close();
            }
            catch (Exception e) {
                LOGGER.error("Failed to close HACOntext", (Throwable)e);
            }
        }
        serverHAContext = null;
    }

    private void validateConf() {
        Preconditions.checkNotNull((Object)this.zookeeperQuorum, (Object)"Zookeeper Quorum should not be null.");
        Preconditions.checkNotNull((Object)this.namespace, (Object)"Zookeeper namespace should not be null.");
    }

    private static String getServicePrincipal(Configuration conf, String confProperty) {
        String principal = (String)Preconditions.checkNotNull((Object)conf.get(confProperty));
        Preconditions.checkArgument((!principal.isEmpty() ? 1 : 0) != 0, (Object)"Server principal is empty.");
        return principal.split("[/@]")[0];
    }

    private void checkAndSetACLs() throws Exception {
        if (this.zkSecure && aclUnChecked) {
            List acls;
            String newNamespace = "/" + this.curatorFramework.getNamespace();
            if (this.curatorFramework.getZookeeperClient().getZooKeeper().exists(newNamespace, null) != null && ((acls = this.curatorFramework.getZookeeperClient().getZooKeeper().getACL(newNamespace, new Stat())).isEmpty() || !((ACL)acls.get(0)).getId().getScheme().equals("sasl"))) {
                LOGGER.info("'sasl' ACLs not set; setting...");
                List children = this.curatorFramework.getZookeeperClient().getZooKeeper().getChildren(newNamespace, null);
                for (String child : children) {
                    this.checkAndSetACLs("/" + child);
                }
                this.curatorFramework.getZookeeperClient().getZooKeeper().setACL(newNamespace, this.saslACL, -1);
            }
            aclUnChecked = false;
        }
    }

    private void checkAndSetACLs(String path) throws Exception {
        LOGGER.info("Setting acls on {}", (Object)path);
        List children = (List)this.curatorFramework.getChildren().forPath(path);
        for (String child : children) {
            this.checkAndSetACLs(path + "/" + child);
        }
        ((BackgroundPathable)this.curatorFramework.setACL().withACL(this.saslACL)).forPath(path);
    }

    private void setJaasConfiguration(Configuration conf) throws IOException {
        if ("false".equalsIgnoreCase(conf.get("sentry.zookeeper.client.ticketcache", "false"))) {
            String keytabFile = conf.get("sentry.zookeeper.client.keytab");
            Preconditions.checkArgument((!keytabFile.isEmpty() ? 1 : 0) != 0, (Object)"Keytab File is empty.");
            String principal = conf.get("sentry.zookeeper.client.principal");
            principal = SecurityUtil.getServerPrincipal((String)principal, (String)conf.get("sentry.service.server.rpc-address", "0.0.0.0"));
            Preconditions.checkArgument((!principal.isEmpty() ? 1 : 0) != 0, (Object)"Kerberos principal is empty.");
            JaasConfiguration.addEntryForKeytab(SENTRY_ZK_JAAS_NAME, principal, keytabFile);
        } else {
            JaasConfiguration.addEntryForTicketCache(SENTRY_ZK_JAAS_NAME);
        }
        javax.security.auth.login.Configuration.setConfiguration(JaasConfiguration.getInstance());
    }

    public LeaderSelector newLeaderSelector(String path, LeaderSelectorListener listener) {
        return new LeaderSelector(this.curatorFramework, path, listener);
    }

    @Override
    public void close() throws Exception {
        this.curatorFramework.close();
    }

    private class SASLOwnerACLProvider
    implements ACLProvider {
        private SASLOwnerACLProvider() {
        }

        @Override
        public List<ACL> getDefaultAcl() {
            return HAContext.this.saslACL;
        }

        @Override
        public List<ACL> getAclForPath(String path) {
            return HAContext.this.saslACL;
        }
    }
}

