/*
 * Decompiled with CFR 0.152.
 */
package org.apache.sentry.hdfs;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;
import java.util.TreeMap;
import java.util.TreeSet;
import org.apache.sentry.hdfs.AuthzPaths;
import org.apache.sentry.hdfs.HMSPathsDumper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HMSPaths
implements AuthzPaths {
    private static final Logger LOG = LoggerFactory.getLogger(HMSPaths.class);
    private volatile Entry root;
    private String[] prefixes;
    private Map<String, Set<Entry>> authzObjToEntries;

    @VisibleForTesting
    static List<String> getPathElements(String path) {
        String trimmedPath = path.trim();
        if (trimmedPath.charAt(0) != '/') {
            throw new IllegalArgumentException("It must be an absolute path: " + trimmedPath);
        }
        ArrayList<String> list = new ArrayList<String>(32);
        int idx = 0;
        int found = trimmedPath.indexOf(47, idx);
        while (found > -1) {
            if (found > idx) {
                list.add(trimmedPath.substring(idx, found));
            }
            idx = found + 1;
            found = trimmedPath.indexOf(47, idx);
        }
        if (idx < trimmedPath.length()) {
            list.add(trimmedPath.substring(idx));
        }
        return list;
    }

    @VisibleForTesting
    static List<List<String>> getPathsElements(List<String> paths) {
        ArrayList<List<String>> pathsElements = new ArrayList<List<String>>(paths.size());
        for (String path : paths) {
            pathsElements.add(HMSPaths.getPathElements(path));
        }
        return pathsElements;
    }

    static List<String> assemblePaths(List<List<String>> pathElements) {
        if (pathElements == null) {
            return Collections.emptyList();
        }
        ArrayList<String> paths = new ArrayList<String>(pathElements.size());
        for (List<String> path : pathElements) {
            StringBuffer sb = new StringBuffer();
            for (String elem : path) {
                sb.append('/').append(elem);
            }
            paths.add(sb.toString());
        }
        return paths;
    }

    public HMSPaths() {
        LOG.info(this.toString() + " (default) Initialized");
    }

    public HMSPaths(String[] pathPrefixes) {
        boolean rootPrefix = false;
        this.prefixes = Arrays.copyOf(pathPrefixes, pathPrefixes.length);
        for (String pathPrefix : pathPrefixes) {
            rootPrefix = rootPrefix || pathPrefix.equals("/");
        }
        if (rootPrefix && pathPrefixes.length > 1) {
            throw new IllegalArgumentException("Root is a path prefix, there cannot be other path prefixes");
        }
        this.root = Entry.createRoot(rootPrefix);
        if (!rootPrefix) {
            for (String pathPrefix : pathPrefixes) {
                this.root.createPrefix(HMSPaths.getPathElements(pathPrefix));
            }
        }
        this.authzObjToEntries = new TreeMap<String, Set<Entry>>(String.CASE_INSENSITIVE_ORDER);
        LOG.info("Sentry managed prefixes: " + this.prefixes.toString());
    }

    void _addAuthzObject(String authzObj, List<String> authzObjPaths) {
        this.addAuthzObject(authzObj, HMSPaths.getPathsElements(authzObjPaths));
    }

    void addAuthzObject(String authzObj, List<List<String>> authzObjPathElements) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Number of Objects: {}", (Object)this.authzObjToEntries.size());
            LOG.debug(String.format("%s addAuthzObject(%s, %s)", this, authzObj, HMSPaths.assemblePaths(authzObjPathElements)));
        }
        Set<Entry> previousEntries = this.authzObjToEntries.get(authzObj);
        HashSet<Entry> newEntries = new HashSet<Entry>(authzObjPathElements.size());
        for (List<String> pathElements : authzObjPathElements) {
            Entry e = this.root.createAuthzObjPath(pathElements, authzObj);
            if (e != null) {
                newEntries.add(e);
                continue;
            }
            LOG.warn(String.format("%s addAuthzObject(%s, %s): Ignoring path %s, no prefix", this, authzObj, HMSPaths.assemblePaths(authzObjPathElements), pathElements));
        }
        this.authzObjToEntries.put(authzObj, newEntries);
        if (previousEntries != null) {
            previousEntries.removeAll(newEntries);
            if (!previousEntries.isEmpty()) {
                for (Entry entry : previousEntries) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Removing stale path {}", (Object)entry.toString());
                    }
                    entry.deleteAuthzObject(authzObj);
                }
            }
        }
    }

    void addPathsToAuthzObject(String authzObj, List<List<String>> authzObjPathElements, boolean createNew) {
        Set<Entry> entries;
        if (LOG.isDebugEnabled()) {
            LOG.debug(String.format("%s addPathsToAuthzObject(%s, %s, %b)", this, authzObj, HMSPaths.assemblePaths(authzObjPathElements), createNew));
        }
        if ((entries = this.authzObjToEntries.get(authzObj)) != null) {
            HashSet<Entry> newEntries = new HashSet<Entry>(authzObjPathElements.size());
            for (List<String> pathElements : authzObjPathElements) {
                Entry e = this.root.createAuthzObjPath(pathElements, authzObj);
                if (e != null) {
                    newEntries.add(e);
                    continue;
                }
                if (!LOG.isDebugEnabled()) continue;
                LOG.debug(String.format("%s addPathsToAuthzObject(%s, %s, %b): Cannot create authz obj for path %s because it is outside of prefix", this, authzObj, HMSPaths.assemblePaths(authzObjPathElements), createNew, pathElements));
            }
            entries.addAll(newEntries);
            if (LOG.isDebugEnabled()) {
                LOG.debug("[addPathsToAuthzObject]Updated path entries for {}", (Object)authzObj);
            }
        } else if (createNew) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("No paths found for Object:{}, Adding new", (Object)authzObj);
            }
            this.addAuthzObject(authzObj, authzObjPathElements);
        } else {
            LOG.warn(String.format("%s addPathsToAuthzObject(%s, %s, %b): Path was not added to AuthzObject, could not find key in authzObjToPath", this, authzObj, HMSPaths.assemblePaths(authzObjPathElements), createNew));
        }
    }

    void _addPathsToAuthzObject(String authzObj, List<String> authzObjPaths) {
        this.addPathsToAuthzObject(authzObj, HMSPaths.getPathsElements(authzObjPaths), false);
    }

    void addPathsToAuthzObject(String authzObj, List<List<String>> authzObjPaths) {
        this.addPathsToAuthzObject(authzObj, authzObjPaths, false);
    }

    void deletePathsFromAuthzObject(String authzObj, List<List<String>> authzObjPathElements) {
        Set<Entry> entries = this.authzObjToEntries.get(authzObj);
        if (entries != null) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("[deletePathsFromAuthzObject] For {}", (Object)authzObj);
            }
            for (List<String> pathElements : authzObjPathElements) {
                Entry entry = this.root.find(pathElements.toArray(new String[pathElements.size()]), false);
                if (entry != null) {
                    entries.remove(entry);
                    entry.deleteAuthzObject(authzObj);
                    continue;
                }
                LOG.warn(String.format("%s deletePathsFromAuthzObject(%s, %s): Path %s was not deleted from AuthzObject, path not registered. This is possible for implicit partition locations", this, authzObj, HMSPaths.assemblePaths(authzObjPathElements), pathElements));
            }
            if (entries.size() == 0) {
                this.authzObjToEntries.remove(authzObj);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("[deletePathsFromAuthzObject] Removing the mapping for {} as the entries are stale", (Object)authzObj);
                }
            }
        } else {
            LOG.warn(String.format("%s deletePathsFromAuthzObject(%s, %s): Path was not deleted from AuthzObject, could not find key in authzObjToPath", this, authzObj, HMSPaths.assemblePaths(authzObjPathElements)));
        }
    }

    void deleteAuthzObject(String authzObj) {
        Set<Entry> entries;
        if (LOG.isDebugEnabled()) {
            LOG.debug(String.format("%s deleteAuthzObject(%s)", this, authzObj));
            LOG.debug("Number of Objects: {}", (Object)this.authzObjToEntries.size());
        }
        if ((entries = this.authzObjToEntries.remove(authzObj)) != null) {
            for (Entry entry : entries) {
                entry.deleteAuthzObject(authzObj);
            }
        }
    }

    Set<String> findAuthzObject(List<String> pathElements) {
        return this.findAuthzObject(pathElements.toArray(new String[0]));
    }

    @Override
    public Set<String> findAuthzObject(String[] pathElements) {
        return this.findAuthzObject(pathElements, true);
    }

    @Override
    public Set<String> findAuthzObjectExactMatches(String[] pathElements) {
        return this.findAuthzObject(pathElements, false);
    }

    public Set<String> findAuthzObject(String[] pathElements, boolean isPartialOk) {
        Set<String> authzObjSet;
        if (LOG.isDebugEnabled()) {
            LOG.debug(String.format("%s findAuthzObject(%s, %b)", this, Arrays.toString(pathElements), isPartialOk));
        }
        if (pathElements == null || pathElements.length == 0) {
            return null;
        }
        Entry entry = this.root.find(pathElements, isPartialOk);
        Set<String> set = authzObjSet = entry != null ? entry.getAuthzObjs() : null;
        if ((authzObjSet == null || authzObjSet.isEmpty()) && LOG.isDebugEnabled()) {
            LOG.debug(String.format("%s findAuthzObject(%s, %b) - no authzObject found", this, Arrays.toString(pathElements), isPartialOk));
        }
        return authzObjSet;
    }

    void renameAuthzObject(String oldName, List<List<String>> oldPathElems, String newName, List<List<String>> newPathElems) {
        Set<Entry> entries;
        List<String> newPathElements;
        if (LOG.isDebugEnabled()) {
            LOG.debug(String.format("%s renameAuthzObject({%s, %s} -> {%s, %s})", this, oldName, HMSPaths.assemblePaths(oldPathElems), newName, HMSPaths.assemblePaths(newPathElems)));
        }
        if (oldPathElems == null || oldPathElems.isEmpty() || newPathElems == null || newPathElems.isEmpty() || newName == null || newName.equals(oldName)) {
            LOG.warn(String.format("%s renameAuthzObject({%s, %s} -> {%s, %s}): invalid inputs, skipping", this, oldName, HMSPaths.assemblePaths(oldPathElems), newName, HMSPaths.assemblePaths(newPathElems)));
            return;
        }
        List<String> oldPathElements = oldPathElems.get(0);
        if (!oldPathElements.equals(newPathElements = newPathElems.get(0))) {
            Entry oldEntry = this.root.find(oldPathElements.toArray(new String[0]), false);
            Entry newParent = this.root.createParent(newPathElements);
            if (oldEntry == null) {
                LOG.warn(String.format("%s Moving old paths for renameAuthzObject({%s, %s} -> {%s, %s}) is skipped. Cannot find entry for old name", this, oldName, HMSPaths.assemblePaths(oldPathElems), newName, HMSPaths.assemblePaths(newPathElems)));
            } else {
                oldEntry.moveTo(newParent, newPathElements.get(newPathElements.size() - 1));
            }
        }
        if ((entries = this.authzObjToEntries.get(oldName)) == null) {
            LOG.warn(String.format("%s renameAuthzObject({%s, %s} -> {%s, %s}): cannot find oldName %s in authzObjToPath", this, oldName, HMSPaths.assemblePaths(oldPathElems), newName, HMSPaths.assemblePaths(newPathElems), oldName));
        } else {
            this.authzObjToEntries.put(newName, entries);
            for (Entry e : entries) {
                e.addAuthzObj(newName);
                if (e.getAuthzObjs().contains(oldName)) {
                    e.removeAuthzObj(oldName);
                    continue;
                }
                LOG.warn(String.format("%s renameAuthzObject({%s, %s} -> {%s, %s}): Unexpected state: authzObjToPath has an entry %s where one of the authz objects does not have oldName", this, oldName, HMSPaths.assemblePaths(oldPathElems), newName, HMSPaths.assemblePaths(newPathElems), e));
            }
        }
        this.deleteAuthzObject(oldName);
    }

    @Override
    public boolean isUnderPrefix(String[] pathElements) {
        return this.root.findPrefixEntry(Lists.newArrayList((Object[])pathElements)) != null;
    }

    String[] getPrefixes() {
        return this.prefixes;
    }

    Entry getRootEntry() {
        return this.root;
    }

    void setRootEntry(Entry root) {
        this.root = root;
    }

    void setAuthzObjToEntryMapping(Map<String, Set<Entry>> mapping) {
        this.authzObjToEntries = mapping;
    }

    public Collection<Entry> getAllEntries() {
        ArrayList<Entry> entries = new ArrayList<Entry>();
        Stack<Entry> stack = new Stack<Entry>();
        stack.push(this.root);
        while (!stack.isEmpty()) {
            Entry entry = (Entry)stack.pop();
            entries.add(entry);
            for (Entry child : entry.childrenValues()) {
                stack.push(child);
            }
        }
        return entries;
    }

    public HMSPathsDumper getPathsDump() {
        return new HMSPathsDumper(this);
    }

    public String toString() {
        return String.format("%s:%s", this.getClass().getSimpleName(), Arrays.toString(this.prefixes));
    }

    public String dumpContent() {
        return this.toString() + ": " + this.getAllEntries();
    }

    @VisibleForTesting
    static class Entry {
        private Entry parent;
        private EntryType type;
        private String pathElement;
        private Object authzObjs;
        private Map<String, Entry> children;

        Entry(Entry parent, String pathElement, EntryType type, String authzObj) {
            this.parent = parent;
            this.type = type;
            this.pathElement = pathElement.intern();
            this.addAuthzObj(authzObj);
        }

        Entry(Entry parent, String pathElement, EntryType type, Collection<String> authzObjs) {
            this.parent = parent;
            this.type = type;
            this.pathElement = pathElement.intern();
            this.addAuthzObjs(authzObjs);
        }

        Entry getChild(String pathElement) {
            if (this.children == null) {
                return null;
            }
            return this.children.get(pathElement);
        }

        void putChild(String pathElement, Entry entry) {
            if (this.children == null) {
                this.children = new HashMap<String, Entry>(2);
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("[putChild]Adding {} as child to {}", (Object)entry.toString(), (Object)this.toString());
            }
            this.children.put(pathElement.intern(), entry);
        }

        Entry removeChild(String pathElement) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("[removeChild]Removing {} from children", (Object)pathElement);
            }
            return this.children.remove(pathElement);
        }

        boolean hasChildren() {
            return this.children != null && !this.children.isEmpty();
        }

        int numChildren() {
            return this.children == null ? 0 : this.children.size();
        }

        Collection<Entry> childrenValues() {
            return this.children != null ? this.children.values() : Collections.emptyList();
        }

        void clearAuthzObjs() {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Clearing authzObjs from {}", (Object)this.toString());
            }
            this.authzObjs = null;
        }

        void removeAuthzObj(String authzObj) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Removing {} from {}", (Object)authzObj, this.authzObjs);
            }
            if (this.authzObjs != null) {
                if (this.authzObjs instanceof Set) {
                    Set authzObjsSet = (Set)this.authzObjs;
                    authzObjsSet.remove(authzObj);
                    if (authzObjsSet.size() == 1) {
                        this.authzObjs = authzObjsSet.iterator().next();
                    }
                } else if (this.authzObjs.equals(authzObj)) {
                    this.authzObjs = null;
                }
            }
        }

        void addAuthzObj(String authzObj) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Adding {} to {}", (Object)authzObj, (Object)this.toString());
            }
            if (authzObj != null) {
                if (this.authzObjs == null) {
                    this.authzObjs = authzObj;
                } else {
                    Set<String> authzObjsSet;
                    if (this.authzObjs instanceof String) {
                        if (this.authzObjs.equals(authzObj)) {
                            return;
                        }
                        this.authzObjs = authzObjsSet = this.newTreeSetWithElement((String)this.authzObjs);
                    } else {
                        authzObjsSet = (Set<String>)this.authzObjs;
                    }
                    authzObjsSet.add(authzObj.intern());
                }
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("Added {} to {}", (Object)authzObj, (Object)this.toString());
            }
        }

        void addAuthzObjs(Collection<String> authzObjs) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Adding {} to {}", authzObjs, (Object)this.toString());
            }
            if (authzObjs != null) {
                for (String authzObj : authzObjs) {
                    this.addAuthzObj(authzObj.intern());
                }
            }
        }

        private void setType(EntryType type) {
            this.type = type;
        }

        protected void removeParent() {
            this.parent = null;
        }

        public String toString() {
            return String.format("Entry[%s:%s -> authObj: %s]", new Object[]{this.type, this.getFullPath(), this.authzObjsToString()});
        }

        private String authzObjsToString() {
            if (this.authzObjs == null) {
                return "";
            }
            if (this.authzObjs instanceof String) {
                return (String)this.authzObjs;
            }
            return Joiner.on((String)",").join((Iterable)((Set)this.authzObjs));
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            Entry other = (Entry)obj;
            if (this.parent == null ? other.parent != null : !this.parent.equals(other.parent)) {
                return false;
            }
            if (this.type == null ? other.type != null : !this.type.equals((Object)other.type)) {
                return false;
            }
            if (this.pathElement == null ? other.pathElement != null : !this.pathElement.equals(other.pathElement)) {
                return false;
            }
            return !(this.authzObjs == null ? other.authzObjs != null : !this.authzObjs.equals(other.authzObjs));
        }

        public int hashCode() {
            int prime = 31;
            int result = 1;
            result = 31 * result + (this.parent == null ? 0 : this.parent.hashCode());
            result = 31 * result + (this.type == null ? 0 : this.type.hashCode());
            result = 31 * result + (this.pathElement == null ? 0 : this.pathElement.hashCode());
            result = 31 * result + (this.authzObjs == null ? 0 : this.authzObjs.hashCode());
            return result;
        }

        private Entry createParent(List<String> pathElements) {
            Entry parent = this;
            if (LOG.isDebugEnabled()) {
                LOG.debug("[createParent]Trying to create entires for {} ", pathElements);
            }
            for (int i = 0; i < pathElements.size() - 1; ++i) {
                String elem = pathElements.get(i);
                Entry child = parent.getChild(elem);
                if (child == null) {
                    child = new Entry(parent, elem, EntryType.DIR, (String)null);
                    parent.putChild(elem, child);
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("[createParent] Entry {} created", (Object)child.toString());
                    }
                }
                parent = child;
            }
            return parent;
        }

        private Entry createChild(List<String> pathElements, EntryType type, String authzObj) {
            String lastPathElement;
            Entry entryParent;
            Entry child;
            if (LOG.isDebugEnabled()) {
                LOG.debug("[createChild] Creating child for {} with path {}", (Object)authzObj, pathElements);
            }
            if ((child = (entryParent = this.createParent(pathElements)).getChild(lastPathElement = pathElements.get(pathElements.size() - 1))) == null) {
                child = new Entry(entryParent, lastPathElement, type, authzObj);
                entryParent.putChild(lastPathElement, child);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Created child entry {}", (Object)child);
                }
            } else if (type == EntryType.AUTHZ_OBJECT && (child.getType() == EntryType.PREFIX || child.getType() == EntryType.AUTHZ_OBJECT)) {
                child.addAuthzObj(authzObj);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("[createChild] Found Child {}, updated authzObj", (Object)child.toString());
                }
            } else if (type == EntryType.AUTHZ_OBJECT && child.getType() == EntryType.DIR) {
                child.addAuthzObj(authzObj);
                child.setType(EntryType.AUTHZ_OBJECT);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("[createChild] Found Child {}, updated authzObj", (Object)child.toString());
                    LOG.debug("[createChild] Updating type to", (Object)child.getType().toString());
                }
            }
            return child;
        }

        public static Entry createRoot(boolean asPrefix) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Creating entry for root");
            }
            return new Entry(null, "/", asPrefix ? EntryType.PREFIX : EntryType.DIR, (String)null);
        }

        private String toPath(List<String> arr) {
            StringBuilder sb = new StringBuilder();
            for (String s : arr) {
                sb.append("/").append(s);
            }
            return sb.toString();
        }

        public Entry createPrefix(List<String> pathElements) {
            Entry prefix;
            if (LOG.isDebugEnabled()) {
                LOG.debug("Creating entries for prefix paths", (Object)pathElements.toString());
            }
            if ((prefix = this.findPrefixEntry(pathElements)) != null) {
                throw new IllegalArgumentException(String.format("%s: createPrefix(%s): cannot add prefix under an existing prefix '%s'", this, pathElements, prefix.getFullPath()));
            }
            return this.createChild(pathElements, EntryType.PREFIX, null);
        }

        public Entry createAuthzObjPath(List<String> pathElements, String authzObj) {
            Entry prefix;
            Entry entry = null;
            if (LOG.isDebugEnabled()) {
                LOG.debug("createAuthzObjPath authzObj:{} paths: {}", (Object)authzObj, pathElements);
            }
            if ((prefix = this.findPrefixEntry(pathElements)) != null) {
                entry = this.createChild(pathElements, EntryType.AUTHZ_OBJECT, authzObj);
            } else if (LOG.isDebugEnabled()) {
                LOG.debug(String.format("%s: createAuthzObjPath(%s, %s): outside of prefix, skipping", this, authzObj, pathElements));
            }
            return entry;
        }

        private void deleteFromParent() {
            if (LOG.isDebugEnabled()) {
                LOG.debug("[deleteFromParent] Attempting to remove path: {}", (Object)this.getFullPath());
            }
            if (this.getParent() != null) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Child in Parent Entry with path: {} is removed", (Object)this.getParent().getFullPath());
                }
                this.getParent().removeChild(this.getPathElement());
                this.getParent().deleteIfDangling();
                this.parent = null;
            } else {
                LOG.warn("Parent for {} not found", (Object)this.toString());
            }
        }

        public void deleteAuthzObject(String authzObj) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("[deleteAuthzObject] Removing authObj:{} from path {}", (Object)authzObj, (Object)this.toString());
            }
            if (!this.getAuthzObjs().contains(authzObj)) {
                return;
            }
            if (this.getParent() != null) {
                if (!this.hasChildren()) {
                    if (this.authzObjs != null) {
                        this.removeAuthzObj(authzObj);
                    }
                    if (this.authzObjs == null) {
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("Deleting path {}", (Object)this.toString());
                        }
                        this.deleteFromParent();
                    }
                } else if (this.getType() == EntryType.AUTHZ_OBJECT) {
                    if (this.authzObjs != null) {
                        this.removeAuthzObj(authzObj);
                    }
                    if (this.getAuthzObjsSize() == 0) {
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("Entry with path: {} is changed to DIR", (Object)this.getFullPath());
                        }
                        this.setType(EntryType.DIR);
                    }
                }
            }
        }

        private void moveTo(Entry newParent, String pathElem) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Moving {} as a child to {}", (Object)this.toString(), (Object)newParent.toString());
            }
            Preconditions.checkNotNull((Object)newParent);
            Preconditions.checkArgument((!pathElem.isEmpty() ? 1 : 0) != 0);
            if (newParent.getChild(pathElem) != null) {
                LOG.warn(String.format("Attempt to move %s to %s: entry with the same name %s already exists", this, newParent, pathElem));
                return;
            }
            this.deleteFromParent();
            this.parent = newParent;
            this.parent.putChild(pathElem, this);
            this.pathElement = pathElem.intern();
        }

        public void delete() {
            if (this.getParent() != null) {
                if (!this.hasChildren()) {
                    this.deleteFromParent();
                } else if (this.getType() == EntryType.AUTHZ_OBJECT) {
                    this.setType(EntryType.DIR);
                    this.clearAuthzObjs();
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Entry with path: {} is changed to DIR", (Object)this.getFullPath());
                    }
                }
            }
        }

        private void deleteIfDangling() {
            if (!this.hasChildren() && this.getType().isRemoveIfDangling()) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Deleting {} as it is dangling", (Object)this.toString());
                }
                this.delete();
            }
        }

        public Entry getParent() {
            return this.parent;
        }

        public EntryType getType() {
            return this.type;
        }

        public String getPathElement() {
            return this.pathElement;
        }

        Set<String> getAuthzObjs() {
            if (this.authzObjs != null) {
                if (this.authzObjs instanceof Set) {
                    return (Set)this.authzObjs;
                }
                return this.newTreeSetWithElement((String)this.authzObjs);
            }
            return Collections.emptySet();
        }

        int getAuthzObjsSize() {
            if (this.authzObjs != null) {
                if (this.authzObjs instanceof Set) {
                    return ((Set)this.authzObjs).size();
                }
                return 1;
            }
            return 0;
        }

        boolean isAuthzObjsEmpty() {
            return this.authzObjs == null;
        }

        private Set<String> newTreeSetWithElement(String el) {
            TreeSet<String> result = new TreeSet<String>(String.CASE_INSENSITIVE_ORDER);
            result.add(el);
            return result;
        }

        public Entry findPrefixEntry(List<String> pathElements) {
            Preconditions.checkArgument((pathElements != null ? 1 : 0) != 0, (Object)"pathElements cannot be NULL");
            return this.getType() == EntryType.PREFIX ? this : this.findPrefixEntry(pathElements, 0);
        }

        private Entry findPrefixEntry(List<String> pathElements, int index) {
            Entry prefixEntry = null;
            if (index == pathElements.size()) {
                prefixEntry = null;
            } else {
                Entry child = this.getChild(pathElements.get(index));
                if (child != null) {
                    prefixEntry = child.getType() == EntryType.PREFIX ? child : child.findPrefixEntry(pathElements, index + 1);
                }
            }
            return prefixEntry;
        }

        public Entry find(String[] pathElements, boolean isPartialMatchOk) {
            Preconditions.checkArgument((pathElements != null && pathElements.length > 0 ? 1 : 0) != 0, (Object)"pathElements cannot be NULL or empty");
            return this.find(pathElements, 0, isPartialMatchOk, null);
        }

        private Entry find(String[] pathElements, int index, boolean isPartialMatchOk, Entry lastAuthObj) {
            Entry found = null;
            if (index == pathElements.length) {
                if (isPartialMatchOk && !this.isAuthzObjsEmpty()) {
                    found = this;
                }
            } else {
                Entry child = this.getChild(pathElements[index]);
                if (child != null) {
                    found = index == pathElements.length - 1 ? (!child.isAuthzObjsEmpty() ? child : lastAuthObj) : child.find(pathElements, index + 1, isPartialMatchOk, !child.isAuthzObjsEmpty() ? child : lastAuthObj);
                } else if (isPartialMatchOk) {
                    found = lastAuthObj;
                }
            }
            return found;
        }

        public String getFullPath() {
            String path = this.getFullPath(this, new StringBuilder()).toString();
            if (path.isEmpty()) {
                path = "/";
            }
            return path;
        }

        private StringBuilder getFullPath(Entry entry, StringBuilder sb) {
            if (entry.getParent() != null) {
                this.getFullPath(entry.getParent(), sb).append("/").append(entry.getPathElement());
            }
            return sb;
        }
    }

    @VisibleForTesting
    static enum EntryType {
        DIR(true),
        PREFIX(false),
        AUTHZ_OBJECT(false);

        private boolean removeIfDangling;

        private EntryType(boolean removeIfDangling) {
            this.removeIfDangling = removeIfDangling;
        }

        public boolean isRemoveIfDangling() {
            return this.removeIfDangling;
        }

        public byte getByte() {
            return (byte)this.toString().charAt(0);
        }

        public static EntryType fromByte(byte b) {
            switch (b) {
                case 68: {
                    return DIR;
                }
                case 80: {
                    return PREFIX;
                }
                case 65: {
                    return AUTHZ_OBJECT;
                }
            }
            return null;
        }
    }
}

