/*
 * Decompiled with CFR 0.152.
 */
package org.apache.sentry.hdfs;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.permission.AclEntry;
import org.apache.hadoop.util.StringUtils;
import org.apache.sentry.hdfs.SentryAuthorizationConstants;
import org.apache.sentry.hdfs.SentryAuthzUpdate;
import org.apache.sentry.hdfs.SentryUpdater;
import org.apache.sentry.hdfs.Updateable;
import org.apache.sentry.hdfs.UpdateableAuthzPaths;
import org.apache.sentry.hdfs.UpdateableAuthzPermissions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SentryAuthorizationInfo
implements Runnable {
    private static final Logger LOG = LoggerFactory.getLogger(SentryAuthorizationInfo.class);
    private static final String SENTRY_AUTHORIZATION_INFO_THREAD_NAME = "sentry-auth-info-refresher";
    private SentryUpdater updater;
    private volatile UpdateableAuthzPaths authzPaths;
    private volatile UpdateableAuthzPermissions authzPermissions;
    private int refreshIntervalMillisec;
    private int staleThresholdMillisec;
    private int retryWaitMillisec;
    private ScheduledExecutorService executor;
    private volatile long lastUpdate;
    private volatile long waitUntil;
    private volatile long lastStaleReport;
    private final ReadWriteLock lock = new ReentrantReadWriteLock();
    private String[][] pathPrefixes;

    public SentryAuthorizationInfo(Configuration conf) throws Exception {
        String[] newPathPrefixes = conf.getTrimmedStrings("sentry.authorization-provider.hdfs-path-prefixes", SentryAuthorizationConstants.HDFS_PATH_PREFIXES_DEFAULT);
        if (newPathPrefixes.length == 0) {
            LOG.warn("There are not HDFS path prefixes configured in [{}], Sentry authorization won't be enforced on any HDFS location", (Object)"sentry.authorization-provider.hdfs-path-prefixes");
        } else {
            this.refreshIntervalMillisec = conf.getInt("sentry.authorization-provider.cache-refresh-interval.ms", 500);
            this.staleThresholdMillisec = conf.getInt("sentry.authorization-provider.cache-stale-threshold.ms", 60000);
            this.retryWaitMillisec = conf.getInt("sentry.authorization-provider.cache-refresh-retry-wait.ms", 30000);
            LOG.info("Sentry authorization will enforced in the following HDFS locations: [{}]", (Object)StringUtils.arrayToString((String[])newPathPrefixes));
            this.setPrefixPaths(newPathPrefixes);
            LOG.info("Refresh interval [{}]ms, retry wait [{}]", (Object)this.refreshIntervalMillisec, (Object)this.retryWaitMillisec);
            LOG.info("stale threshold [{}]ms", (Object)this.staleThresholdMillisec);
            this.authzPaths = new UpdateableAuthzPaths(newPathPrefixes);
            this.authzPermissions = new UpdateableAuthzPermissions();
            this.waitUntil = System.currentTimeMillis();
            this.lastStaleReport = 0L;
            this.updater = new SentryUpdater(conf, this);
        }
    }

    private void setPrefixPaths(String[] pathPrefixes) {
        this.pathPrefixes = new String[pathPrefixes.length][];
        for (int i = 0; i < this.pathPrefixes.length; ++i) {
            Preconditions.checkArgument((boolean)pathPrefixes[i].startsWith("/"), (Object)("Path prefix [" + pathPrefixes[i] + "]does not starting with [" + '/' + "]"));
            this.pathPrefixes[i] = pathPrefixes[i].substring(1).split("/");
        }
    }

    String[][] getPathPrefixes() {
        return this.pathPrefixes;
    }

    UpdateableAuthzPaths getAuthzPaths() {
        return this.authzPaths;
    }

    UpdateableAuthzPermissions getAuthzPermissions() {
        return this.authzPermissions;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private boolean update() {
        SentryAuthzUpdate updates = this.updater.getUpdates();
        if (updates != null) {
            if (updates.isEmpty()) {
                return true;
            }
            LOG.info("Received updates from Sentry Server. Size of PathUpdates {} PermUpdates {}", (Object)updates.getPathUpdates().size(), (Object)updates.getPermUpdates().size());
            LOG.debug("Processing updates " + updates.dumpContent());
            UpdateableAuthzPaths newAuthzPaths = this.processUpdates(updates.getPathUpdates(), this.authzPaths);
            UpdateableAuthzPermissions newAuthzPerms = this.processUpdates(updates.getPermUpdates(), this.authzPermissions);
            if (newAuthzPaths != this.authzPaths || newAuthzPerms != this.authzPermissions) {
                this.lock.writeLock().lock();
                try {
                    if (newAuthzPaths != this.authzPaths) {
                        LOG.info(String.format("FULL Updated paths seq Num [old=%d], [new=%d]", this.authzPaths.getLastUpdatedSeqNum(), newAuthzPaths.getLastUpdatedSeqNum()));
                        this.authzPaths = newAuthzPaths;
                        if (LOG.isTraceEnabled()) {
                            LOG.trace(this.authzPaths.dumpContent());
                        }
                    }
                    if (newAuthzPerms == this.authzPermissions) return true;
                    LOG.info(String.format("FULL Updated perms seq Num [old=%d], [new=%d]", this.authzPermissions.getLastUpdatedSeqNum(), newAuthzPerms.getLastUpdatedSeqNum()));
                    this.authzPermissions = newAuthzPerms;
                    if (!LOG.isTraceEnabled()) return true;
                    LOG.trace(this.authzPermissions.dumpContent());
                    return true;
                }
                finally {
                    this.lock.writeLock().unlock();
                }
            }
            if (!LOG.isDebugEnabled()) return true;
            this.lock.writeLock().lock();
            try {
                LOG.debug(updates.dumpContent());
                if (!LOG.isTraceEnabled()) return true;
                LOG.trace(newAuthzPaths.dumpContent());
                LOG.trace(newAuthzPerms.dumpContent());
                return true;
            }
            finally {
                this.lock.writeLock().unlock();
            }
        }
        LOG.error("Received NULL updates from Sentry");
        return false;
    }

    private <K extends Updateable.Update, V extends Updateable<K>> V processUpdates(List<K> updates, V updateable) {
        Object newUpdateable = updateable;
        if (!updates.isEmpty()) {
            if (((Updateable.Update)updates.get(0)).hasFullImage()) {
                String logMessage = String.format("Processing Update : FULL IMAGE", new Object[0]) + newUpdateable.getSequenceInfo();
                LOG.info(logMessage);
                newUpdateable = newUpdateable.updateFull((Updateable.Update)((Updateable.Update)updates.remove(0)));
                LOG.info(String.format("Processing Update : Finished processing FULL IMAGE update..", new Object[0]));
            }
            if (!updates.isEmpty()) {
                LOG.debug(String.format("Processing Update : Delta updates.. [%s] Count:[%d]", newUpdateable.getSequenceInfo(), updates.size()));
                newUpdateable.updatePartial(updates, this.lock);
            }
            LOG.debug(String.format("Processing Update : Finished updates..", new Object[0]));
        }
        return newUpdateable;
    }

    @Override
    public void run() {
        boolean success = false;
        try {
            long currTime = System.currentTimeMillis();
            if (this.waitUntil > currTime) {
                Thread.sleep(this.waitUntil - currTime);
            }
            success = this.update();
        }
        catch (Exception ex) {
            success = false;
            LOG.warn("Failed to update, will retry in [{}]ms, error: ", new Object[]{this.retryWaitMillisec, ex.getMessage(), ex});
        }
        catch (Throwable t) {
            LOG.error("Received a throwable while refreshing the cache", t);
            System.out.println("Received a throwable  while refreshing the cache " + t);
            throw t;
        }
        this.waitUntil = success ? (this.lastUpdate = System.currentTimeMillis()) : System.currentTimeMillis() + (long)this.retryWaitMillisec;
    }

    public void start() {
        if (this.authzPaths != null || this.authzPermissions != null) {
            boolean success = false;
            try {
                success = this.update();
            }
            catch (Exception ex) {
                success = false;
                LOG.warn("Failed to do initial update, will retry in [{}]ms, error: ", new Object[]{this.retryWaitMillisec, ex.getMessage(), ex});
            }
            if (!success) {
                this.waitUntil = System.currentTimeMillis() + (long)this.retryWaitMillisec;
            }
            ThreadFactory sentryAuthInfoRefresherThreadFactory = new ThreadFactoryBuilder().setNameFormat(SENTRY_AUTHORIZATION_INFO_THREAD_NAME).setDaemon(true).build();
            this.executor = Executors.newSingleThreadScheduledExecutor(sentryAuthInfoRefresherThreadFactory);
            this.executor.scheduleWithFixedDelay(this, this.refreshIntervalMillisec, this.refreshIntervalMillisec, TimeUnit.MILLISECONDS);
        }
    }

    public void stop() {
        if (this.authzPaths != null) {
            LOG.info(this.getClass().getSimpleName() + ": Stopping");
            this.executor.shutdownNow();
        }
    }

    public boolean isStale() {
        boolean stale;
        long now = System.currentTimeMillis();
        boolean bl = stale = now - this.lastUpdate > (long)this.staleThresholdMillisec;
        if (stale && now - this.lastStaleReport > (long)this.retryWaitMillisec) {
            LOG.warn("Authorization information has been stale for [{}]s", (Object)((now - this.lastUpdate) / 1000L));
            this.lastStaleReport = now;
        }
        return stale;
    }

    public boolean isUnderPrefix(String[] pathElements) {
        this.lock.readLock().lock();
        try {
            boolean bl = this.authzPaths.isUnderPrefix(pathElements);
            return bl;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    @Deprecated
    public boolean isManaged(String[] pathElements) {
        return this.isUnderPrefix(pathElements);
    }

    public boolean doesBelongToAuthzObject(String[] pathElements) {
        this.lock.readLock().lock();
        try {
            boolean bl = this.authzPaths.findAuthzObject(pathElements) != null;
            return bl;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    public boolean isSentryManaged(String[] pathElements) {
        this.lock.readLock().lock();
        try {
            boolean bl = this.authzPaths.isUnderPrefix(pathElements) && this.authzPaths.findAuthzObject(pathElements) != null;
            return bl;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<AclEntry> getAclEntries(String[] pathElements) {
        this.lock.readLock().lock();
        try {
            Set<String> authzObjs = this.authzPaths.findAuthzObject(pathElements);
            AclEntry noGroup = AclEntry.parseAclEntry((String)"group::---", (boolean)true);
            HashSet<Object> retSet = new HashSet<Object>();
            retSet.add(noGroup);
            if (authzObjs == null) {
                retSet.addAll(Collections.emptyList());
                ArrayList<AclEntry> arrayList = new ArrayList<AclEntry>(retSet);
                return arrayList;
            }
            for (String authzObj : authzObjs) {
                retSet.addAll(this.authzPermissions.getAcls(authzObj));
            }
            ArrayList arrayList = new ArrayList(retSet);
            return arrayList;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }
}

