/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.protobuf.InvalidProtocolBufferException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.TreeMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.crypto.CipherSuite;
import org.apache.hadoop.crypto.CryptoProtocolVersion;
import org.apache.hadoop.crypto.key.KeyProviderCryptoExtension;
import org.apache.hadoop.fs.BatchedRemoteIterator;
import org.apache.hadoop.fs.ParentNotDirectoryException;
import org.apache.hadoop.fs.UnresolvedLinkException;
import org.apache.hadoop.fs.XAttr;
import org.apache.hadoop.fs.XAttrSetFlag;
import org.apache.hadoop.hdfs.XAttrHelper;
import org.apache.hadoop.hdfs.protocol.EncryptionZone;
import org.apache.hadoop.hdfs.protocol.ReencryptionStatus;
import org.apache.hadoop.hdfs.protocol.SnapshotAccessControlException;
import org.apache.hadoop.hdfs.protocol.ZoneReencryptionStatus;
import org.apache.hadoop.hdfs.protocol.proto.HdfsProtos;
import org.apache.hadoop.hdfs.protocolPB.PBHelperClient;
import org.apache.hadoop.hdfs.server.namenode.FSDirEncryptionZoneOp;
import org.apache.hadoop.hdfs.server.namenode.FSDirXAttrOp;
import org.apache.hadoop.hdfs.server.namenode.FSDirectory;
import org.apache.hadoop.hdfs.server.namenode.FSPermissionChecker;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.hdfs.server.namenode.INodesInPath;
import org.apache.hadoop.hdfs.server.namenode.ReencryptionHandler;
import org.apache.hadoop.security.AccessControlException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EncryptionZoneManager {
    public static final Logger LOG = LoggerFactory.getLogger(EncryptionZoneManager.class);
    private TreeMap<Long, EncryptionZoneInt> encryptionZones = null;
    private final FSDirectory dir;
    private final int maxListEncryptionZonesResponses;
    private final int maxListRecncryptionStatusResponses;
    private ThreadFactory reencryptionThreadFactory;
    private ExecutorService reencryptHandlerExecutor;
    private ReencryptionHandler reencryptionHandler;
    private final ReencryptionStatus reencryptionStatus;
    public static final BatchedRemoteIterator.BatchedListEntries<ZoneReencryptionStatus> EMPTY_LIST = new BatchedRemoteIterator.BatchedListEntries(new ArrayList(), false);

    @VisibleForTesting
    public void pauseReencryptForTesting() {
        this.reencryptionHandler.pauseForTesting();
    }

    @VisibleForTesting
    public void resumeReencryptForTesting() {
        this.reencryptionHandler.resumeForTesting();
    }

    @VisibleForTesting
    public void pauseForTestingAfterNthSubmission(int count) {
        this.reencryptionHandler.pauseForTestingAfterNthSubmission(count);
    }

    @VisibleForTesting
    public void pauseReencryptUpdaterForTesting() {
        this.reencryptionHandler.pauseUpdaterForTesting();
    }

    @VisibleForTesting
    public void resumeReencryptUpdaterForTesting() {
        this.reencryptionHandler.resumeUpdaterForTesting();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    public void pauseForTestingAfterNthCheckpoint(String zone, int count) throws IOException {
        INodesInPath iip;
        this.dir.readLock();
        try {
            iip = this.dir.resolvePath(this.dir.getPermissionChecker(), zone, FSDirectory.DirOp.READ);
        }
        finally {
            this.dir.readUnlock();
        }
        this.reencryptionHandler.pauseForTestingAfterNthCheckpoint(iip.getLastINode().getId(), count);
    }

    @VisibleForTesting
    public void resetMetricsForTesting() {
        this.reencryptionStatus.resetMetrics();
    }

    @VisibleForTesting
    public ReencryptionStatus getReencryptionStatus() {
        return this.reencryptionStatus;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    public ZoneReencryptionStatus getZoneStatus(String zone) throws IOException {
        FSPermissionChecker pc = this.dir.getPermissionChecker();
        this.dir.getFSNamesystem().readLock();
        this.dir.readLock();
        try {
            INodesInPath iip = this.dir.resolvePath(pc, zone, FSDirectory.DirOp.READ);
            INode inode = iip.getLastINode();
            if (inode == null) {
                ZoneReencryptionStatus zoneReencryptionStatus = null;
                return zoneReencryptionStatus;
            }
            ZoneReencryptionStatus zoneReencryptionStatus = this.getReencryptionStatus().getZoneStatus(Long.valueOf(inode.getId()));
            return zoneReencryptionStatus;
        }
        finally {
            this.dir.readUnlock();
            this.dir.getFSNamesystem().readUnlock();
        }
    }

    FSDirectory getFSDirectory() {
        return this.dir;
    }

    public EncryptionZoneManager(FSDirectory dir, Configuration conf) {
        this.dir = dir;
        this.maxListEncryptionZonesResponses = conf.getInt("dfs.namenode.list.encryption.zones.num.responses", 100);
        Preconditions.checkArgument((this.maxListEncryptionZonesResponses >= 0 ? 1 : 0) != 0, (Object)"dfs.namenode.list.encryption.zones.num.responses must be a positive integer.");
        if (this.getProvider() != null) {
            this.reencryptionHandler = new ReencryptionHandler(this, conf);
            this.reencryptionThreadFactory = new ThreadFactoryBuilder().setDaemon(true).setNameFormat("reencryptionHandlerThread #%d").build();
        }
        this.maxListRecncryptionStatusResponses = conf.getInt("dfs.namenode.list.reencryption.status.num.responses", 100);
        Preconditions.checkArgument((this.maxListRecncryptionStatusResponses >= 0 ? 1 : 0) != 0, (Object)"dfs.namenode.list.reencryption.status.num.responses must be a positive integer.");
        this.reencryptionStatus = new ReencryptionStatus();
    }

    KeyProviderCryptoExtension getProvider() {
        return this.dir.getProvider();
    }

    void startReencryptThreads() {
        if (this.getProvider() == null) {
            return;
        }
        Preconditions.checkNotNull((Object)this.reencryptionHandler);
        this.reencryptHandlerExecutor = Executors.newSingleThreadExecutor(this.reencryptionThreadFactory);
        this.reencryptHandlerExecutor.execute(this.reencryptionHandler);
        this.reencryptionHandler.startUpdaterThread();
    }

    void stopReencryptThread() {
        if (this.getProvider() == null || this.reencryptionHandler == null) {
            return;
        }
        this.dir.writeLock();
        try {
            this.reencryptionHandler.stopThreads();
        }
        finally {
            this.dir.writeUnlock();
        }
        if (this.reencryptHandlerExecutor != null) {
            this.reencryptHandlerExecutor.shutdownNow();
            this.reencryptHandlerExecutor = null;
        }
    }

    void addEncryptionZone(Long inodeId, CipherSuite suite, CryptoProtocolVersion version, String keyName) {
        assert (this.dir.hasWriteLock());
        this.unprotectedAddEncryptionZone(inodeId, suite, version, keyName);
    }

    void unprotectedAddEncryptionZone(Long inodeId, CipherSuite suite, CryptoProtocolVersion version, String keyName) {
        EncryptionZoneInt ez = new EncryptionZoneInt(inodeId, suite, version, keyName);
        if (this.encryptionZones == null) {
            this.encryptionZones = new TreeMap();
        }
        this.encryptionZones.put(inodeId, ez);
    }

    void removeEncryptionZone(Long inodeId) {
        assert (this.dir.hasWriteLock());
        if (this.hasCreatedEncryptionZone()) {
            if (this.encryptionZones.remove(inodeId) == null || !this.getReencryptionStatus().hasRunningZone(inodeId)) {
                return;
            }
            if (this.reencryptionHandler != null) {
                this.reencryptionHandler.removeZone(inodeId);
            }
        }
    }

    boolean isInAnEZ(INodesInPath iip) throws UnresolvedLinkException, SnapshotAccessControlException, IOException {
        assert (this.dir.hasReadLock());
        return this.getEncryptionZoneForPath(iip) != null;
    }

    String getFullPathName(Long nodeId) {
        assert (this.dir.hasReadLock());
        INode inode = this.dir.getInode(nodeId);
        if (inode == null) {
            return null;
        }
        return inode.getFullPathName();
    }

    String getKeyName(INodesInPath iip) throws IOException {
        assert (this.dir.hasReadLock());
        EncryptionZoneInt ezi = this.getEncryptionZoneForPath(iip);
        if (ezi == null) {
            return null;
        }
        return ezi.getKeyName();
    }

    private EncryptionZoneInt getEncryptionZoneForPath(INodesInPath iip) throws IOException {
        assert (this.dir.hasReadLock());
        Preconditions.checkNotNull((Object)iip);
        if (!this.hasCreatedEncryptionZone()) {
            return null;
        }
        int snapshotID = iip.getPathSnapshotId();
        for (int i = iip.length() - 1; i >= 0; --i) {
            INode inode = iip.getINode(i);
            if (inode == null || !inode.isDirectory()) continue;
            if (snapshotID == 0x7FFFFFFE) {
                EncryptionZoneInt ezi = this.encryptionZones.get(inode.getId());
                if (ezi == null) continue;
                return ezi;
            }
            XAttr xAttr = FSDirXAttrOp.unprotectedGetXAttrByPrefixedName(inode, snapshotID, "raw.hdfs.crypto.encryption.zone");
            if (xAttr == null) continue;
            try {
                HdfsProtos.ZoneEncryptionInfoProto ezProto = HdfsProtos.ZoneEncryptionInfoProto.parseFrom((byte[])xAttr.getValue());
                return new EncryptionZoneInt(inode.getId(), PBHelperClient.convert((HdfsProtos.CipherSuiteProto)ezProto.getSuite()), PBHelperClient.convert((HdfsProtos.CryptoProtocolVersionProto)ezProto.getCryptoProtocolVersion()), ezProto.getKeyName());
            }
            catch (InvalidProtocolBufferException e) {
                throw new IOException("Could not parse encryption zone for inode " + iip.getPath(), e);
            }
        }
        return null;
    }

    private EncryptionZoneInt getParentEncryptionZoneForPath(INodesInPath iip) throws IOException {
        assert (this.dir.hasReadLock());
        Preconditions.checkNotNull((Object)iip);
        INodesInPath parentIIP = iip.getParentINodesInPath();
        return parentIIP == null ? null : this.getEncryptionZoneForPath(parentIIP);
    }

    EncryptionZone getEZINodeForPath(INodesInPath iip) throws IOException {
        EncryptionZoneInt ezi = this.getEncryptionZoneForPath(iip);
        if (ezi == null) {
            return null;
        }
        return new EncryptionZone(ezi.getINodeId(), this.getFullPathName(ezi.getINodeId()), ezi.getSuite(), ezi.getVersion(), ezi.getKeyName());
    }

    void checkMoveValidity(INodesInPath srcIIP, INodesInPath dstIIP) throws IOException {
        boolean dstInEZ;
        assert (this.dir.hasReadLock());
        if (!this.hasCreatedEncryptionZone()) {
            return;
        }
        EncryptionZoneInt srcParentEZI = this.getParentEncryptionZoneForPath(srcIIP);
        EncryptionZoneInt dstParentEZI = this.getParentEncryptionZoneForPath(dstIIP);
        boolean srcInEZ = srcParentEZI != null;
        boolean bl = dstInEZ = dstParentEZI != null;
        if (srcInEZ && !dstInEZ) {
            throw new IOException(srcIIP.getPath() + " can't be moved from an encryption zone.");
        }
        if (dstInEZ && !srcInEZ) {
            throw new IOException(srcIIP.getPath() + " can't be moved into an encryption zone.");
        }
        if (srcInEZ) {
            if (!srcParentEZI.equals(dstParentEZI)) {
                String srcEZPath = this.getFullPathName(srcParentEZI.getINodeId());
                String dstEZPath = this.getFullPathName(dstParentEZI.getINodeId());
                StringBuilder sb = new StringBuilder(srcIIP.getPath());
                sb.append(" can't be moved from encryption zone ");
                sb.append(srcEZPath);
                sb.append(" to encryption zone ");
                sb.append(dstEZPath);
                sb.append(".");
                throw new IOException(sb.toString());
            }
            this.checkMoveValidityForReencryption(srcIIP.getPath(), srcParentEZI.getINodeId());
        } else if (dstInEZ) {
            this.checkMoveValidityForReencryption(dstIIP.getPath(), dstParentEZI.getINodeId());
        }
    }

    private void checkMoveValidityForReencryption(String pathName, long zoneId) throws IOException {
        assert (this.dir.hasReadLock());
        ZoneReencryptionStatus zs = this.reencryptionStatus.getZoneStatus(Long.valueOf(zoneId));
        if (zs != null && zs.getState() != ZoneReencryptionStatus.State.Completed) {
            StringBuilder sb = new StringBuilder(pathName);
            sb.append(" can't be moved because encryption zone ");
            sb.append(this.getFullPathName(zoneId));
            sb.append(" is currently under re-encryption");
            throw new IOException(sb.toString());
        }
    }

    XAttr createEncryptionZone(INodesInPath srcIIP, CipherSuite suite, CryptoProtocolVersion version, String keyName) throws IOException {
        assert (this.dir.hasWriteLock());
        if (srcIIP.getLastINode() == null) {
            throw new FileNotFoundException("cannot find " + srcIIP.getPath());
        }
        if (this.dir.isNonEmptyDirectory(srcIIP)) {
            throw new IOException("Attempt to create an encryption zone for a non-empty directory.");
        }
        INode srcINode = srcIIP.getLastINode();
        if (!srcINode.isDirectory()) {
            throw new IOException("Attempt to create an encryption zone for a file.");
        }
        if (this.hasCreatedEncryptionZone() && this.encryptionZones.get(srcINode.getId()) != null) {
            throw new IOException("Directory " + srcIIP.getPath() + " is already an encryption zone.");
        }
        HdfsProtos.ZoneEncryptionInfoProto proto = PBHelperClient.convert((CipherSuite)suite, (CryptoProtocolVersion)version, (String)keyName);
        XAttr ezXAttr = XAttrHelper.buildXAttr((String)"raw.hdfs.crypto.encryption.zone", (byte[])proto.toByteArray());
        ArrayList xattrs = Lists.newArrayListWithCapacity((int)1);
        xattrs.add(ezXAttr);
        FSDirXAttrOp.unprotectedSetXAttrs(this.dir, srcIIP, xattrs, EnumSet.of(XAttrSetFlag.CREATE));
        return ezXAttr;
    }

    BatchedRemoteIterator.BatchedListEntries<EncryptionZone> listEncryptionZones(long prevId) throws IOException {
        assert (this.dir.hasReadLock());
        if (!this.hasCreatedEncryptionZone()) {
            return new BatchedRemoteIterator.BatchedListEntries((List)Lists.newArrayList(), false);
        }
        NavigableMap<Long, EncryptionZoneInt> tailMap = this.encryptionZones.tailMap(prevId, false);
        int numResponses = Math.min(this.maxListEncryptionZonesResponses, tailMap.size());
        ArrayList zones = Lists.newArrayListWithExpectedSize((int)numResponses);
        int count = 0;
        for (EncryptionZoneInt ezi : tailMap.values()) {
            String pathName = this.getFullPathName(ezi.getINodeId());
            if (!this.pathResolvesToId(ezi.getINodeId(), pathName)) continue;
            zones.add(new EncryptionZone(ezi.getINodeId(), pathName, ezi.getSuite(), ezi.getVersion(), ezi.getKeyName()));
            if (++count < numResponses) continue;
            break;
        }
        boolean hasMore = numResponses < tailMap.size();
        return new BatchedRemoteIterator.BatchedListEntries((List)zones, hasMore);
    }

    private boolean pathResolvesToId(long zoneId, String zonePath) throws UnresolvedLinkException, AccessControlException, ParentNotDirectoryException {
        assert (this.dir.hasReadLock());
        INode inode = this.dir.getInode(zoneId);
        if (inode == null) {
            return false;
        }
        INode lastINode = null;
        if (INode.isValidAbsolutePath(zonePath)) {
            INodesInPath iip = this.dir.getINodesInPath(zonePath, FSDirectory.DirOp.READ_LINK);
            lastINode = iip.getLastINode();
        }
        return lastINode != null && lastINode.getId() == zoneId;
    }

    List<XAttr> reencryptEncryptionZone(INodesInPath zoneIIP, String keyVersionName) throws IOException {
        assert (this.dir.hasWriteLock());
        if (this.reencryptionHandler == null) {
            throw new IOException("No key provider configured, re-encryption operation is rejected");
        }
        ArrayList xAttrs = Lists.newArrayListWithCapacity((int)1);
        INode inode = zoneIIP.getLastINode();
        String zoneName = zoneIIP.getPath();
        this.checkEncryptionZoneRoot(inode, zoneName);
        if (this.getReencryptionStatus().hasRunningZone(Long.valueOf(inode.getId()))) {
            throw new IOException("Zone " + zoneName + " is already submitted for re-encryption.");
        }
        LOG.info("Zone {}({}) is submitted for re-encryption.", (Object)zoneName, (Object)inode.getId());
        XAttr xattr = FSDirEncryptionZoneOp.updateReencryptionSubmitted(this.dir, zoneIIP, keyVersionName);
        xAttrs.add(xattr);
        this.reencryptionHandler.notifyNewSubmission();
        return xAttrs;
    }

    List<XAttr> cancelReencryptEncryptionZone(INodesInPath zoneIIP) throws IOException {
        assert (this.dir.hasWriteLock());
        if (this.reencryptionHandler == null) {
            throw new IOException("No key provider configured, re-encryption operation is rejected");
        }
        long zoneId = zoneIIP.getLastINode().getId();
        String zoneName = zoneIIP.getPath();
        this.checkEncryptionZoneRoot(zoneIIP.getLastINode(), zoneName);
        this.reencryptionHandler.cancelZone(zoneId, zoneName);
        LOG.info("Cancelled zone {}({}) for re-encryption.", (Object)zoneName, (Object)zoneId);
        return FSDirEncryptionZoneOp.updateReencryptionFinish(this.dir, zoneIIP, this.reencryptionStatus.getZoneStatus(Long.valueOf(zoneId)));
    }

    BatchedRemoteIterator.BatchedListEntries<ZoneReencryptionStatus> listReencryptionStatus(long prevId) throws IOException {
        assert (this.dir.hasReadLock());
        if (!this.hasCreatedEncryptionZone()) {
            return ReencryptionStatus.EMPTY_LIST;
        }
        NavigableMap stats = this.reencryptionStatus.getZoneStatuses();
        if (stats.isEmpty()) {
            return EMPTY_LIST;
        }
        NavigableMap tailMap = stats.tailMap(prevId, false);
        int numResp = Math.min(this.maxListRecncryptionStatusResponses, tailMap.size());
        ArrayList ret = Lists.newArrayListWithExpectedSize((int)numResp);
        int count = 0;
        for (ZoneReencryptionStatus zs : tailMap.values()) {
            String name = this.getFullPathName(zs.getId());
            if (name == null || !this.pathResolvesToId(zs.getId(), name)) continue;
            zs.setZoneName(name);
            ret.add(zs);
            if (++count < numResp) continue;
            break;
        }
        boolean hasMore = numResp < tailMap.size();
        return new BatchedRemoteIterator.BatchedListEntries((List)ret, hasMore);
    }

    boolean isEncryptionZoneRoot(INode inode, String name) throws FileNotFoundException {
        assert (this.dir.hasReadLock());
        if (inode == null) {
            throw new FileNotFoundException("INode does not exist for " + name);
        }
        if (!inode.isDirectory()) {
            return false;
        }
        return this.hasCreatedEncryptionZone() && this.encryptionZones.containsKey(inode.getId());
    }

    void checkEncryptionZoneRoot(INode inode, String name) throws IOException {
        if (!this.isEncryptionZoneRoot(inode, name)) {
            throw new IOException("Path " + name + " is not the root of an encryption zone.");
        }
    }

    public int getNumEncryptionZones() {
        return this.hasCreatedEncryptionZone() ? this.encryptionZones.size() : 0;
    }

    public boolean hasCreatedEncryptionZone() {
        return this.encryptionZones != null;
    }

    String[] getKeyNames() {
        assert (this.dir.hasReadLock());
        if (!this.hasCreatedEncryptionZone()) {
            return new String[0];
        }
        String[] ret = new String[this.encryptionZones.size()];
        int index = 0;
        for (Map.Entry<Long, EncryptionZoneInt> entry : this.encryptionZones.entrySet()) {
            ret[index++] = entry.getValue().getKeyName();
        }
        return ret;
    }

    private static class EncryptionZoneInt {
        private final long inodeId;
        private final CipherSuite suite;
        private final CryptoProtocolVersion version;
        private final String keyName;

        EncryptionZoneInt(long inodeId, CipherSuite suite, CryptoProtocolVersion version, String keyName) {
            Preconditions.checkArgument((suite != CipherSuite.UNKNOWN ? 1 : 0) != 0);
            Preconditions.checkArgument((version != CryptoProtocolVersion.UNKNOWN ? 1 : 0) != 0);
            this.inodeId = inodeId;
            this.suite = suite;
            this.version = version;
            this.keyName = keyName;
        }

        long getINodeId() {
            return this.inodeId;
        }

        CipherSuite getSuite() {
            return this.suite;
        }

        CryptoProtocolVersion getVersion() {
            return this.version;
        }

        String getKeyName() {
            return this.keyName;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof EncryptionZoneInt)) {
                return false;
            }
            EncryptionZoneInt b = (EncryptionZoneInt)o;
            return new EqualsBuilder().append(this.inodeId, b.getINodeId()).append((Object)this.suite, (Object)b.getSuite()).append((Object)this.version, (Object)b.getVersion()).append((Object)this.keyName, (Object)b.getKeyName()).isEquals();
        }

        public int hashCode() {
            return new HashCodeBuilder().append(this.inodeId).append((Object)this.suite).append((Object)this.version).append((Object)this.keyName).toHashCode();
        }
    }
}

