/*
 * Decompiled with CFR 0.152.
 */
package org.apache.sentry.core.common.transport;

import com.google.common.base.Preconditions;
import com.google.common.net.HostAndPort;
import java.util.ArrayList;
import java.util.Collections;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.annotation.concurrent.ThreadSafe;
import org.apache.commons.pool2.BaseKeyedPooledObjectFactory;
import org.apache.commons.pool2.KeyedObjectPool;
import org.apache.commons.pool2.KeyedPooledObjectFactory;
import org.apache.commons.pool2.PooledObject;
import org.apache.commons.pool2.impl.DefaultPooledObject;
import org.apache.commons.pool2.impl.GenericKeyedObjectPool;
import org.apache.commons.pool2.impl.GenericKeyedObjectPoolConfig;
import org.apache.hadoop.conf.Configuration;
import org.apache.sentry.core.common.transport.SentryClientTransportConfigInterface;
import org.apache.sentry.core.common.transport.TTransportWrapper;
import org.apache.sentry.core.common.transport.TransportFactory;
import org.apache.sentry.core.common.utils.ThriftUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public final class SentryTransportPool
implements AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(SentryTransportPool.class);
    private static int poolId = 0;
    private final int id;
    private final boolean isPoolEnabled;
    private final boolean doLoadBalancing;
    private final ArrayList<HostAndPort> endpoints;
    private final KeyedObjectPool<HostAndPort, TTransportWrapper> pool;
    private final TransportFactory transportFactory;
    private final AtomicBoolean closed = new AtomicBoolean();

    public SentryTransportPool(Configuration conf, SentryClientTransportConfigInterface transportConfig, TransportFactory transportFactory) {
        this.id = ++poolId;
        this.transportFactory = transportFactory;
        this.doLoadBalancing = transportConfig.isLoadBalancingEnabled(conf);
        this.isPoolEnabled = transportConfig.isTransportPoolEnabled(conf);
        String hostsAndPortsStr = transportConfig.getSentryServerRpcAddress(conf);
        int serverPort = transportConfig.getServerRpcPort(conf);
        LOGGER.info("Creating pool for {} with default port {}", (Object)hostsAndPortsStr, (Object)serverPort);
        String[] hostsAndPortsStrArr = hostsAndPortsStr.split(",");
        Preconditions.checkArgument((hostsAndPortsStrArr.length > 0 ? 1 : 0) != 0, (Object)"At least one server should be specified");
        this.endpoints = new ArrayList(hostsAndPortsStrArr.length);
        for (String addr : hostsAndPortsStrArr) {
            HostAndPort endpoint = ThriftUtil.parseAddress(addr, serverPort);
            LOGGER.info("Adding endpoint {}", (Object)endpoint);
            this.endpoints.add(endpoint);
        }
        if (!this.isPoolEnabled) {
            this.pool = null;
            LOGGER.info("Connection pooling is disabled");
            return;
        }
        LOGGER.info("Connection pooling is enabled");
        GenericKeyedObjectPoolConfig poolConfig = new GenericKeyedObjectPoolConfig();
        poolConfig.setMaxTotal(-1);
        poolConfig.setMinIdlePerKey(transportConfig.getPoolMinIdle(conf));
        poolConfig.setMaxIdlePerKey(transportConfig.getPoolMaxIdle(conf));
        poolConfig.setBlockWhenExhausted(false);
        poolConfig.setTestOnReturn(true);
        poolConfig.setMaxTotalPerKey(transportConfig.getPoolMaxTotal(conf));
        poolConfig.setMinEvictableIdleTimeMillis(transportConfig.getMinEvictableTimeSec(conf));
        poolConfig.setTimeBetweenEvictionRunsMillis(transportConfig.getTimeBetweenEvictionRunsSec(conf));
        this.pool = new GenericKeyedObjectPool((KeyedPooledObjectFactory)new PoolFactory(this.transportFactory, this.id), poolConfig);
    }

    public TTransportWrapper getTransport() throws Exception {
        ArrayList<HostAndPort> servers;
        if (this.doLoadBalancing && this.endpoints.size() > 1) {
            servers = new ArrayList<HostAndPort>(this.endpoints);
            Collections.shuffle(servers);
        } else {
            servers = this.endpoints;
        }
        Exception failure = null;
        boolean ignoreEmptyPool = true;
        for (int attempt = 0; attempt < 2; ++attempt) {
            for (HostAndPort addr : servers) {
                if (this.isPoolEnabled && ignoreEmptyPool && this.pool.getNumIdle((Object)addr) == 0) {
                    LOGGER.debug("Ignoring empty pool {}", (Object)addr);
                    ignoreEmptyPool = false;
                    continue;
                }
                try {
                    TTransportWrapper transport = this.isPoolEnabled ? (TTransportWrapper)this.pool.borrowObject((Object)addr) : this.transportFactory.getTransport(addr);
                    LOGGER.debug("[{}] obtained transport {}", (Object)this.id, (Object)transport);
                    if (LOGGER.isDebugEnabled() && this.isPoolEnabled) {
                        LOGGER.debug("Currently {} active connections, {} idle connections", (Object)this.pool.getNumActive(), (Object)this.pool.getNumIdle());
                    }
                    return transport;
                }
                catch (IllegalStateException e) {
                    LOGGER.error("Unexpected error from pool {}", (Object)this.id, (Object)e);
                    failure = e;
                }
                catch (Exception e) {
                    LOGGER.error("Failed to obtain transport for {}: {}", (Object)addr, (Object)e.getMessage());
                    failure = e;
                }
            }
            ignoreEmptyPool = false;
        }
        assert (failure != null);
        throw failure;
    }

    public void returnTransport(TTransportWrapper transport) {
        if (this.closed.get()) {
            LOGGER.debug("Returned {} to closed pool", (Object)transport);
            transport.close();
            return;
        }
        try {
            if (this.isPoolEnabled) {
                LOGGER.debug("[{}] returning {}", (Object)this.id, (Object)transport);
                this.pool.returnObject((Object)transport.getAddress(), (Object)transport);
            } else {
                LOGGER.debug("Closing {}", (Object)transport);
                transport.close();
            }
        }
        catch (Exception e) {
            LOGGER.error("Failed to return {}", (Object)transport, (Object)e);
        }
    }

    public void invalidateTransport(TTransportWrapper transport) {
        if (this.closed.get()) {
            LOGGER.debug("invalidated {} for closed pool", (Object)transport);
            transport.close();
            return;
        }
        try {
            LOGGER.debug("[{}] Invalidating address {}", (Object)this.id, (Object)transport);
            if (!this.isPoolEnabled) {
                transport.close();
            } else {
                this.pool.invalidateObject((Object)transport.getAddress(), (Object)transport);
                this.pool.clear((Object)transport.getAddress());
            }
        }
        catch (Exception e) {
            LOGGER.error("Failed to invalidate {}", (Object)transport, (Object)e);
        }
    }

    @Override
    public void close() throws Exception {
        if (this.closed.get()) {
            return;
        }
        LOGGER.debug("[{}] closing", (Object)this.id);
        if (this.pool != null) {
            LOGGER.debug("Closing pool of {}/{} endpoints", (Object)this.pool.getNumIdle(), (Object)this.pool.getNumActive());
            this.pool.close();
        }
    }

    private static final class PoolFactory
    extends BaseKeyedPooledObjectFactory<HostAndPort, TTransportWrapper> {
        private final TransportFactory transportFactory;
        private final int id;

        private PoolFactory(TransportFactory transportFactory, int id) {
            this.transportFactory = transportFactory;
            this.id = id;
        }

        public boolean validateObject(HostAndPort key, PooledObject<TTransportWrapper> p) {
            TTransportWrapper transport = (TTransportWrapper)p.getObject();
            if (transport == null) {
                LOGGER.error("No transport to validate");
                return false;
            }
            if (transport.getAddress() != key) {
                LOGGER.error("Invalid endpoint {}: does not match {}", (Object)transport, (Object)key);
                return false;
            }
            return true;
        }

        public TTransportWrapper create(HostAndPort key) throws Exception {
            TTransportWrapper transportWrapper = this.transportFactory.getTransport(key);
            LOGGER.debug("[{}] created {}", (Object)this.id, (Object)transportWrapper);
            return transportWrapper;
        }

        public void destroyObject(HostAndPort key, PooledObject<TTransportWrapper> p) throws Exception {
            TTransportWrapper transport = (TTransportWrapper)p.getObject();
            if (transport != null) {
                LOGGER.debug("[{}] Destroying endpoint {}", (Object)this.id, (Object)transport);
                try {
                    transport.close();
                }
                catch (RuntimeException e) {
                    LOGGER.error("fail to destroy endpoint {}", (Object)transport, (Object)e);
                }
            }
        }

        public PooledObject<TTransportWrapper> wrap(TTransportWrapper value) {
            return new DefaultPooledObject((Object)value);
        }
    }
}

