/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.aliyun.oss;

import com.aliyun.oss.ClientConfiguration;
import com.aliyun.oss.ClientException;
import com.aliyun.oss.OSSClient;
import com.aliyun.oss.OSSException;
import com.aliyun.oss.common.auth.CredentialsProvider;
import com.aliyun.oss.common.comm.Protocol;
import com.aliyun.oss.model.AbortMultipartUploadRequest;
import com.aliyun.oss.model.CannedAccessControlList;
import com.aliyun.oss.model.CompleteMultipartUploadRequest;
import com.aliyun.oss.model.CompleteMultipartUploadResult;
import com.aliyun.oss.model.CopyObjectResult;
import com.aliyun.oss.model.DeleteObjectsRequest;
import com.aliyun.oss.model.DeleteObjectsResult;
import com.aliyun.oss.model.GetObjectRequest;
import com.aliyun.oss.model.InitiateMultipartUploadRequest;
import com.aliyun.oss.model.InitiateMultipartUploadResult;
import com.aliyun.oss.model.ListObjectsRequest;
import com.aliyun.oss.model.OSSObjectSummary;
import com.aliyun.oss.model.ObjectListing;
import com.aliyun.oss.model.ObjectMetadata;
import com.aliyun.oss.model.PartETag;
import com.aliyun.oss.model.PutObjectResult;
import com.aliyun.oss.model.UploadPartCopyRequest;
import com.aliyun.oss.model.UploadPartCopyResult;
import com.aliyun.oss.model.UploadPartRequest;
import com.aliyun.oss.model.UploadPartResult;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;
import java.util.NoSuchElementException;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BlockLocation;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.aliyun.oss.AliyunOSSUtils;
import org.apache.hadoop.fs.aliyun.oss.Constants;
import org.apache.hadoop.fs.aliyun.oss.FileStatusAcceptor;
import org.apache.hadoop.util.VersionInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AliyunOSSFileSystemStore {
    public static final Logger LOG = LoggerFactory.getLogger(AliyunOSSFileSystemStore.class);
    private FileSystem.Statistics statistics;
    private OSSClient ossClient;
    private String bucketName;
    private long uploadPartSize;
    private long multipartThreshold;
    private long partSize;
    private int maxKeys;
    private String serverSideEncryptionAlgorithm;

    public void initialize(URI uri, Configuration conf, FileSystem.Statistics stat) throws IOException {
        String cannedACLName;
        this.statistics = stat;
        ClientConfiguration clientConf = new ClientConfiguration();
        clientConf.setMaxConnections(conf.getInt("fs.oss.connection.maximum", 32));
        boolean secureConnections = conf.getBoolean("fs.oss.connection.secure.enabled", true);
        clientConf.setProtocol(secureConnections ? Protocol.HTTPS : Protocol.HTTP);
        clientConf.setMaxErrorRetry(conf.getInt("fs.oss.attempts.maximum", 10));
        clientConf.setConnectionTimeout(conf.getInt("fs.oss.connection.establish.timeout", 50000));
        clientConf.setSocketTimeout(conf.getInt("fs.oss.connection.timeout", 200000));
        clientConf.setUserAgent(conf.get("fs.oss.user.agent.prefix", Constants.USER_AGENT_PREFIX_DEFAULT) + ", Hadoop/" + VersionInfo.getVersion());
        String proxyHost = conf.getTrimmed("fs.oss.proxy.host", "");
        int proxyPort = conf.getInt("fs.oss.proxy.port", -1);
        if (StringUtils.isNotEmpty((String)proxyHost)) {
            clientConf.setProxyHost(proxyHost);
            if (proxyPort >= 0) {
                clientConf.setProxyPort(proxyPort);
            } else if (secureConnections) {
                LOG.warn("Proxy host set without port. Using HTTPS default 443");
                clientConf.setProxyPort(443);
            } else {
                LOG.warn("Proxy host set without port. Using HTTP default 80");
                clientConf.setProxyPort(80);
            }
            String proxyUsername = conf.getTrimmed("fs.oss.proxy.username");
            String proxyPassword = conf.getTrimmed("fs.oss.proxy.password");
            if (proxyUsername == null != (proxyPassword == null)) {
                String msg = "Proxy error: fs.oss.proxy.username or fs.oss.proxy.password set without the other.";
                LOG.error(msg);
                throw new IllegalArgumentException(msg);
            }
            clientConf.setProxyUsername(proxyUsername);
            clientConf.setProxyPassword(proxyPassword);
            clientConf.setProxyDomain(conf.getTrimmed("fs.oss.proxy.domain"));
            clientConf.setProxyWorkstation(conf.getTrimmed("fs.oss.proxy.workstation"));
        } else if (proxyPort >= 0) {
            String msg = "Proxy error: fs.oss.proxy.port set without fs.oss.proxy.host";
            LOG.error(msg);
            throw new IllegalArgumentException(msg);
        }
        String endPoint = conf.getTrimmed("fs.oss.endpoint", "");
        if (StringUtils.isEmpty((String)endPoint)) {
            throw new IllegalArgumentException("Aliyun OSS endpoint should not be null or empty. Please set proper endpoint with 'fs.oss.endpoint'.");
        }
        CredentialsProvider provider = AliyunOSSUtils.getCredentialsProvider(conf);
        this.ossClient = new OSSClient(endPoint, provider, clientConf);
        this.uploadPartSize = conf.getLong("fs.oss.multipart.upload.size", 0xA00000L);
        this.multipartThreshold = conf.getLong("fs.oss.multipart.upload.threshold", 0x1400000L);
        this.partSize = conf.getLong("fs.oss.multipart.upload.size", 0xA00000L);
        if (this.partSize < 102400L) {
            this.partSize = 102400L;
        }
        this.serverSideEncryptionAlgorithm = conf.get("fs.oss.server-side-encryption-algorithm", "");
        if (this.uploadPartSize < 0x500000L) {
            LOG.warn("fs.oss.multipart.upload.size must be at least 5 MB");
            this.uploadPartSize = 0x500000L;
        }
        if (this.multipartThreshold < 0x500000L) {
            LOG.warn("fs.oss.multipart.upload.threshold must be at least 5 MB");
            this.multipartThreshold = 0x500000L;
        }
        if (this.multipartThreshold > 0x40000000L) {
            LOG.warn("fs.oss.multipart.upload.threshold must be less than 1 GB");
            this.multipartThreshold = 0x40000000L;
        }
        if (StringUtils.isNotEmpty((String)(cannedACLName = conf.get("fs.oss.acl.default", "")))) {
            CannedAccessControlList cannedACL = CannedAccessControlList.valueOf((String)cannedACLName);
            this.ossClient.setBucketAcl(this.bucketName, cannedACL);
        }
        this.maxKeys = conf.getInt("fs.oss.paging.maximum", 1000);
        this.bucketName = uri.getHost();
    }

    public void deleteObject(String key) {
        this.ossClient.deleteObject(this.bucketName, key);
        this.statistics.incrementWriteOps(1);
    }

    public void deleteObjects(List<String> keysToDelete) throws IOException {
        if (CollectionUtils.isEmpty(keysToDelete)) {
            LOG.warn("Keys to delete is empty.");
            return;
        }
        int retry = 10;
        int tries = 0;
        List deleteFailed = keysToDelete;
        while (CollectionUtils.isNotEmpty(deleteFailed)) {
            DeleteObjectsRequest deleteRequest = new DeleteObjectsRequest(this.bucketName);
            deleteRequest.setKeys(deleteFailed);
            deleteRequest.setQuiet(true);
            DeleteObjectsResult result = this.ossClient.deleteObjects(deleteRequest);
            deleteFailed = result.getDeletedObjects();
            if (++tries != retry) continue;
            break;
        }
        if (tries == retry && CollectionUtils.isNotEmpty(deleteFailed)) {
            throw new IOException("Failed to delete Aliyun OSS objects for " + tries + " times.");
        }
    }

    public void deleteDirs(String key) throws IOException {
        key = AliyunOSSUtils.maybeAddTrailingSlash(key);
        ListObjectsRequest listRequest = new ListObjectsRequest(this.bucketName);
        listRequest.setPrefix(key);
        listRequest.setDelimiter(null);
        listRequest.setMaxKeys(Integer.valueOf(this.maxKeys));
        while (true) {
            ObjectListing objects = this.ossClient.listObjects(listRequest);
            this.statistics.incrementReadOps(1);
            ArrayList<String> keysToDelete = new ArrayList<String>();
            for (OSSObjectSummary objectSummary : objects.getObjectSummaries()) {
                keysToDelete.add(objectSummary.getKey());
            }
            this.deleteObjects(keysToDelete);
            if (!objects.isTruncated()) break;
            listRequest.setMarker(objects.getNextMarker());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ObjectMetadata getObjectMetadata(String key) {
        try {
            ObjectMetadata objectMetadata = this.ossClient.getObjectMetadata(this.bucketName, key);
            return objectMetadata;
        }
        catch (OSSException osse) {
            ObjectMetadata objectMetadata = null;
            return objectMetadata;
        }
        finally {
            this.statistics.incrementReadOps(1);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void storeEmptyFile(String key) throws IOException {
        ObjectMetadata dirMeta = new ObjectMetadata();
        byte[] buffer = new byte[]{};
        dirMeta.setContentLength(0L);
        try (ByteArrayInputStream in = new ByteArrayInputStream(buffer);){
            this.ossClient.putObject(this.bucketName, key, (InputStream)in, dirMeta);
        }
    }

    public boolean copyFile(String srcKey, String dstKey) {
        ObjectMetadata objectMeta = this.ossClient.getObjectMetadata(this.bucketName, srcKey);
        long contentLength = objectMeta.getContentLength();
        if (contentLength <= this.multipartThreshold) {
            return this.singleCopy(srcKey, dstKey);
        }
        return this.multipartCopy(srcKey, contentLength, dstKey);
    }

    private boolean singleCopy(String srcKey, String dstKey) {
        CopyObjectResult copyResult = this.ossClient.copyObject(this.bucketName, srcKey, this.bucketName, dstKey);
        LOG.debug(copyResult.getETag());
        return true;
    }

    private boolean multipartCopy(String srcKey, long contentLength, String dstKey) {
        long realPartSize = AliyunOSSUtils.calculatePartSize(contentLength, this.uploadPartSize);
        int partNum = (int)(contentLength / realPartSize);
        if (contentLength % realPartSize != 0L) {
            ++partNum;
        }
        InitiateMultipartUploadRequest initiateMultipartUploadRequest = new InitiateMultipartUploadRequest(this.bucketName, dstKey);
        ObjectMetadata meta = new ObjectMetadata();
        if (StringUtils.isNotEmpty((String)this.serverSideEncryptionAlgorithm)) {
            meta.setServerSideEncryption(this.serverSideEncryptionAlgorithm);
        }
        initiateMultipartUploadRequest.setObjectMetadata(meta);
        InitiateMultipartUploadResult initiateMultipartUploadResult = this.ossClient.initiateMultipartUpload(initiateMultipartUploadRequest);
        String uploadId = initiateMultipartUploadResult.getUploadId();
        ArrayList<PartETag> partETags = new ArrayList<PartETag>();
        try {
            for (int i = 0; i < partNum; ++i) {
                long skipBytes = realPartSize * (long)i;
                long size = realPartSize < contentLength - skipBytes ? realPartSize : contentLength - skipBytes;
                UploadPartCopyRequest partCopyRequest = new UploadPartCopyRequest();
                partCopyRequest.setSourceBucketName(this.bucketName);
                partCopyRequest.setSourceKey(srcKey);
                partCopyRequest.setBucketName(this.bucketName);
                partCopyRequest.setKey(dstKey);
                partCopyRequest.setUploadId(uploadId);
                partCopyRequest.setPartSize(Long.valueOf(size));
                partCopyRequest.setBeginIndex(Long.valueOf(skipBytes));
                partCopyRequest.setPartNumber(i + 1);
                UploadPartCopyResult partCopyResult = this.ossClient.uploadPartCopy(partCopyRequest);
                this.statistics.incrementWriteOps(1);
                partETags.add(partCopyResult.getPartETag());
            }
            CompleteMultipartUploadRequest completeMultipartUploadRequest = new CompleteMultipartUploadRequest(this.bucketName, dstKey, uploadId, partETags);
            CompleteMultipartUploadResult completeMultipartUploadResult = this.ossClient.completeMultipartUpload(completeMultipartUploadRequest);
            LOG.debug(completeMultipartUploadResult.getETag());
            return true;
        }
        catch (ClientException | OSSException e) {
            AbortMultipartUploadRequest abortMultipartUploadRequest = new AbortMultipartUploadRequest(this.bucketName, dstKey, uploadId);
            this.ossClient.abortMultipartUpload(abortMultipartUploadRequest);
            return false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void uploadObject(String key, File file) throws IOException {
        File object = file.getAbsoluteFile();
        FileInputStream fis = new FileInputStream(object);
        ObjectMetadata meta = new ObjectMetadata();
        meta.setContentLength(object.length());
        if (StringUtils.isNotEmpty((String)this.serverSideEncryptionAlgorithm)) {
            meta.setServerSideEncryption(this.serverSideEncryptionAlgorithm);
        }
        try {
            PutObjectResult result = this.ossClient.putObject(this.bucketName, key, (InputStream)fis, meta);
            LOG.debug(result.getETag());
            this.statistics.incrementWriteOps(1);
        }
        finally {
            fis.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void multipartUploadObject(String key, File file) throws IOException {
        File object = file.getAbsoluteFile();
        long dataLen = object.length();
        long realPartSize = AliyunOSSUtils.calculatePartSize(dataLen, this.partSize);
        int partNum = (int)(dataLen / realPartSize);
        if (dataLen % realPartSize != 0L) {
            ++partNum;
        }
        InitiateMultipartUploadRequest initiateMultipartUploadRequest = new InitiateMultipartUploadRequest(this.bucketName, key);
        ObjectMetadata meta = new ObjectMetadata();
        if (StringUtils.isNotEmpty((String)this.serverSideEncryptionAlgorithm)) {
            meta.setServerSideEncryption(this.serverSideEncryptionAlgorithm);
        }
        initiateMultipartUploadRequest.setObjectMetadata(meta);
        InitiateMultipartUploadResult initiateMultipartUploadResult = this.ossClient.initiateMultipartUpload(initiateMultipartUploadRequest);
        ArrayList<PartETag> partETags = new ArrayList<PartETag>();
        String uploadId = initiateMultipartUploadResult.getUploadId();
        try {
            for (int i = 0; i < partNum; ++i) {
                try (FileInputStream fis = new FileInputStream(object);){
                    long skipBytes = realPartSize * (long)i;
                    AliyunOSSUtils.skipFully(fis, skipBytes);
                    long size = realPartSize < dataLen - skipBytes ? realPartSize : dataLen - skipBytes;
                    UploadPartRequest uploadPartRequest = new UploadPartRequest();
                    uploadPartRequest.setBucketName(this.bucketName);
                    uploadPartRequest.setKey(key);
                    uploadPartRequest.setUploadId(uploadId);
                    uploadPartRequest.setInputStream((InputStream)fis);
                    uploadPartRequest.setPartSize(size);
                    uploadPartRequest.setPartNumber(i + 1);
                    UploadPartResult uploadPartResult = this.ossClient.uploadPart(uploadPartRequest);
                    this.statistics.incrementWriteOps(1);
                    partETags.add(uploadPartResult.getPartETag());
                    continue;
                }
            }
            CompleteMultipartUploadRequest completeMultipartUploadRequest = new CompleteMultipartUploadRequest(this.bucketName, key, uploadId, partETags);
            CompleteMultipartUploadResult completeMultipartUploadResult = this.ossClient.completeMultipartUpload(completeMultipartUploadRequest);
            LOG.debug(completeMultipartUploadResult.getETag());
        }
        catch (ClientException | OSSException e) {
            AbortMultipartUploadRequest abortMultipartUploadRequest = new AbortMultipartUploadRequest(this.bucketName, key, uploadId);
            this.ossClient.abortMultipartUpload(abortMultipartUploadRequest);
        }
    }

    public ObjectListing listObjects(String prefix, int maxListingLength, String marker, boolean recursive) {
        String delimiter = recursive ? null : "/";
        prefix = AliyunOSSUtils.maybeAddTrailingSlash(prefix);
        ListObjectsRequest listRequest = new ListObjectsRequest(this.bucketName);
        listRequest.setPrefix(prefix);
        listRequest.setDelimiter(delimiter);
        listRequest.setMaxKeys(Integer.valueOf(maxListingLength));
        listRequest.setMarker(marker);
        ObjectListing listing = this.ossClient.listObjects(listRequest);
        this.statistics.incrementReadOps(1);
        return listing;
    }

    public InputStream retrieve(String key, long byteStart, long byteEnd) {
        try {
            GetObjectRequest request = new GetObjectRequest(this.bucketName, key);
            request.setRange(byteStart, byteEnd);
            return this.ossClient.getObject(request).getObjectContent();
        }
        catch (ClientException | OSSException e) {
            return null;
        }
    }

    public void close() {
        if (this.ossClient != null) {
            this.ossClient.shutdown();
            this.ossClient = null;
        }
    }

    public void purge(String prefix) throws IOException {
        try {
            ObjectListing objects = this.listObjects(prefix, this.maxKeys, null, true);
            for (OSSObjectSummary object : objects.getObjectSummaries()) {
                String key = object.getKey();
                this.ossClient.deleteObject(this.bucketName, key);
            }
            for (String dir : objects.getCommonPrefixes()) {
                this.deleteDirs(dir);
            }
        }
        catch (ClientException | OSSException e) {
            LOG.error("Failed to purge " + prefix);
        }
    }

    public RemoteIterator<LocatedFileStatus> singleStatusRemoteIterator(final FileStatus fileStatus, final BlockLocation[] locations) {
        return new RemoteIterator<LocatedFileStatus>(){
            private boolean hasNext = true;

            public boolean hasNext() throws IOException {
                return fileStatus != null && this.hasNext;
            }

            public LocatedFileStatus next() throws IOException {
                if (this.hasNext()) {
                    LocatedFileStatus s = new LocatedFileStatus(fileStatus, fileStatus.isFile() ? locations : null);
                    this.hasNext = false;
                    return s;
                }
                throw new NoSuchElementException();
            }
        };
    }

    public RemoteIterator<LocatedFileStatus> createLocatedFileStatusIterator(final String prefix, final int maxListingLength, final FileSystem fs, final PathFilter filter, final FileStatusAcceptor acceptor, final String delimiter) {
        return new RemoteIterator<LocatedFileStatus>(){
            private String nextMarker = null;
            private boolean firstListing = true;
            private boolean meetEnd = false;
            private ListIterator<FileStatus> batchIterator;

            public boolean hasNext() throws IOException {
                if (this.firstListing) {
                    this.requestNextBatch();
                    this.firstListing = false;
                }
                return this.batchIterator.hasNext() || this.requestNextBatch();
            }

            public LocatedFileStatus next() throws IOException {
                if (this.hasNext()) {
                    FileStatus status = this.batchIterator.next();
                    BlockLocation[] locations = fs.getFileBlockLocations(status, 0L, status.getLen());
                    return new LocatedFileStatus(status, (BlockLocation[])(status.isFile() ? locations : null));
                }
                throw new NoSuchElementException();
            }

            private boolean requestNextBatch() {
                if (this.meetEnd) {
                    return false;
                }
                ListObjectsRequest listRequest = new ListObjectsRequest(AliyunOSSFileSystemStore.this.bucketName);
                listRequest.setPrefix(AliyunOSSUtils.maybeAddTrailingSlash(prefix));
                listRequest.setMaxKeys(Integer.valueOf(maxListingLength));
                listRequest.setMarker(this.nextMarker);
                listRequest.setDelimiter(delimiter);
                ObjectListing listing = AliyunOSSFileSystemStore.this.ossClient.listObjects(listRequest);
                ArrayList<FileStatus> stats = new ArrayList<FileStatus>(listing.getObjectSummaries().size() + listing.getCommonPrefixes().size());
                for (OSSObjectSummary summary : listing.getObjectSummaries()) {
                    String key = summary.getKey();
                    Path path = fs.makeQualified(new Path("/" + key));
                    if (!filter.accept(path) || !acceptor.accept(path, summary)) continue;
                    FileStatus status = new FileStatus(summary.getSize(), key.endsWith("/"), 1, fs.getDefaultBlockSize(path), summary.getLastModified().getTime(), path);
                    stats.add(status);
                }
                for (String commonPrefix : listing.getCommonPrefixes()) {
                    Path path = fs.makeQualified(new Path("/" + commonPrefix));
                    if (!filter.accept(path) || !acceptor.accept(path, commonPrefix)) continue;
                    FileStatus status = new FileStatus(0L, true, 1, 0L, 0L, path);
                    stats.add(status);
                }
                this.batchIterator = stats.listIterator();
                if (listing.isTruncated()) {
                    this.nextMarker = listing.getNextMarker();
                } else {
                    this.meetEnd = true;
                }
                AliyunOSSFileSystemStore.this.statistics.incrementReadOps(1);
                return this.batchIterator.hasNext();
            }
        };
    }
}

