/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.shaded.org.apache.avro;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import org.apache.hadoop.shaded.org.apache.avro.AvroRuntimeException;
import org.apache.hadoop.shaded.org.apache.avro.Schema;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SchemaCompatibility {
    private static final Logger LOG = LoggerFactory.getLogger(SchemaCompatibility.class);
    public static final String READER_WRITER_COMPATIBLE_MESSAGE = "Reader schema can always successfully decode data written using the writer schema.";

    private SchemaCompatibility() {
    }

    public static SchemaPairCompatibility checkReaderWriterCompatibility(Schema reader, Schema writer) {
        String message;
        SchemaCompatibilityType compatibility = new ReaderWriterCompatiblityChecker().getCompatibility(reader, writer);
        switch (compatibility) {
            case INCOMPATIBLE: {
                message = String.format("Data encoded using writer schema:%n%s%nwill or may fail to decode using reader schema:%n%s%n", writer.toString(true), reader.toString(true));
                break;
            }
            case COMPATIBLE: {
                message = READER_WRITER_COMPATIBLE_MESSAGE;
                break;
            }
            default: {
                throw new AvroRuntimeException("Unknown compatibility: " + (Object)((Object)compatibility));
            }
        }
        return new SchemaPairCompatibility(compatibility, reader, writer, message);
    }

    public static boolean schemaNameEquals(Schema reader, Schema writer) {
        String writerFullName = writer.getFullName();
        if (SchemaCompatibility.objectsEqual(reader.getFullName(), writerFullName)) {
            return true;
        }
        return reader.getAliases().contains(writerFullName);
    }

    public static Schema.Field lookupWriterField(Schema writerSchema, Schema.Field readerField) {
        assert (writerSchema.getType() == Schema.Type.RECORD);
        ArrayList<Schema.Field> writerFields = new ArrayList<Schema.Field>();
        Schema.Field direct = writerSchema.getField(readerField.name());
        if (direct != null) {
            writerFields.add(direct);
        }
        for (String readerFieldAliasName : readerField.aliases()) {
            Schema.Field writerField = writerSchema.getField(readerFieldAliasName);
            if (writerField == null) continue;
            writerFields.add(writerField);
        }
        switch (writerFields.size()) {
            case 0: {
                return null;
            }
            case 1: {
                return (Schema.Field)writerFields.get(0);
            }
        }
        throw new AvroRuntimeException(String.format("Reader record field %s matches multiple fields in writer record schema %s", readerField, writerSchema));
    }

    private static boolean objectsEqual(Object obj1, Object obj2) {
        return obj1 == obj2 || obj1 != null && obj1.equals(obj2);
    }

    public static final class SchemaPairCompatibility {
        private final SchemaCompatibilityType mType;
        private final Schema mReader;
        private final Schema mWriter;
        private final String mDescription;

        public SchemaPairCompatibility(SchemaCompatibilityType type, Schema reader, Schema writer, String description) {
            this.mType = type;
            this.mReader = reader;
            this.mWriter = writer;
            this.mDescription = description;
        }

        public SchemaCompatibilityType getType() {
            return this.mType;
        }

        public Schema getReader() {
            return this.mReader;
        }

        public Schema getWriter() {
            return this.mWriter;
        }

        public String getDescription() {
            return this.mDescription;
        }

        public String toString() {
            return String.format("SchemaPairCompatibility{type:%s, readerSchema:%s, writerSchema:%s, description:%s}", new Object[]{this.mType, this.mReader, this.mWriter, this.mDescription});
        }

        public boolean equals(Object other) {
            if (null != other && other instanceof SchemaPairCompatibility) {
                SchemaPairCompatibility result = (SchemaPairCompatibility)other;
                return SchemaCompatibility.objectsEqual((Object)result.mType, (Object)this.mType) && SchemaCompatibility.objectsEqual(result.mReader, this.mReader) && SchemaCompatibility.objectsEqual(result.mWriter, this.mWriter) && SchemaCompatibility.objectsEqual(result.mDescription, this.mDescription);
            }
            return false;
        }

        public int hashCode() {
            return Arrays.hashCode(new Object[]{this.mType, this.mReader, this.mWriter, this.mDescription});
        }
    }

    public static enum SchemaCompatibilityType {
        COMPATIBLE,
        INCOMPATIBLE,
        RECURSION_IN_PROGRESS;

    }

    private static final class ReaderWriterCompatiblityChecker {
        private final Map<ReaderWriter, SchemaCompatibilityType> mMemoizeMap = new HashMap<ReaderWriter, SchemaCompatibilityType>();

        private ReaderWriterCompatiblityChecker() {
        }

        public SchemaCompatibilityType getCompatibility(Schema reader, Schema writer) {
            LOG.debug("Checking compatibility of reader {} with writer {}", (Object)reader, (Object)writer);
            ReaderWriter pair = new ReaderWriter(reader, writer);
            SchemaCompatibilityType existing = this.mMemoizeMap.get(pair);
            if (existing != null) {
                if (existing == SchemaCompatibilityType.RECURSION_IN_PROGRESS) {
                    return SchemaCompatibilityType.COMPATIBLE;
                }
                return existing;
            }
            this.mMemoizeMap.put(pair, SchemaCompatibilityType.RECURSION_IN_PROGRESS);
            SchemaCompatibilityType calculated = this.calculateCompatibility(reader, writer);
            this.mMemoizeMap.put(pair, calculated);
            return calculated;
        }

        private SchemaCompatibilityType calculateCompatibility(Schema reader, Schema writer) {
            assert (reader != null);
            assert (writer != null);
            if (reader.getType() == writer.getType()) {
                switch (reader.getType()) {
                    case NULL: 
                    case BOOLEAN: 
                    case INT: 
                    case LONG: 
                    case FLOAT: 
                    case DOUBLE: 
                    case BYTES: 
                    case STRING: {
                        return SchemaCompatibilityType.COMPATIBLE;
                    }
                    case ARRAY: {
                        return this.getCompatibility(reader.getElementType(), writer.getElementType());
                    }
                    case MAP: {
                        return this.getCompatibility(reader.getValueType(), writer.getValueType());
                    }
                    case FIXED: {
                        if (!SchemaCompatibility.schemaNameEquals(reader, writer)) {
                            return SchemaCompatibilityType.INCOMPATIBLE;
                        }
                        if (reader.getFixedSize() != writer.getFixedSize()) {
                            return SchemaCompatibilityType.INCOMPATIBLE;
                        }
                        return SchemaCompatibilityType.COMPATIBLE;
                    }
                    case ENUM: {
                        if (!SchemaCompatibility.schemaNameEquals(reader, writer)) {
                            return SchemaCompatibilityType.INCOMPATIBLE;
                        }
                        HashSet<String> symbols = new HashSet<String>(writer.getEnumSymbols());
                        symbols.removeAll(reader.getEnumSymbols());
                        return symbols.isEmpty() ? SchemaCompatibilityType.COMPATIBLE : SchemaCompatibilityType.INCOMPATIBLE;
                    }
                    case RECORD: {
                        if (!SchemaCompatibility.schemaNameEquals(reader, writer)) {
                            return SchemaCompatibilityType.INCOMPATIBLE;
                        }
                        for (Schema.Field readerField : reader.getFields()) {
                            Schema.Field writerField = SchemaCompatibility.lookupWriterField(writer, readerField);
                            if (!(writerField == null ? readerField.defaultValue() == null : this.getCompatibility(readerField.schema(), writerField.schema()) == SchemaCompatibilityType.INCOMPATIBLE)) continue;
                            return SchemaCompatibilityType.INCOMPATIBLE;
                        }
                        return SchemaCompatibilityType.COMPATIBLE;
                    }
                    case UNION: {
                        for (Schema writerBranch : writer.getTypes()) {
                            if (this.getCompatibility(reader, writerBranch) != SchemaCompatibilityType.INCOMPATIBLE) continue;
                            return SchemaCompatibilityType.INCOMPATIBLE;
                        }
                        return SchemaCompatibilityType.COMPATIBLE;
                    }
                }
                throw new AvroRuntimeException("Unknown schema type: " + (Object)((Object)reader.getType()));
            }
            if (writer.getType() == Schema.Type.UNION) {
                for (Schema s : writer.getTypes()) {
                    SchemaCompatibilityType compatibility = this.getCompatibility(reader, s);
                    if (compatibility != SchemaCompatibilityType.INCOMPATIBLE) continue;
                    return SchemaCompatibilityType.INCOMPATIBLE;
                }
                return SchemaCompatibilityType.COMPATIBLE;
            }
            switch (reader.getType()) {
                case NULL: {
                    return SchemaCompatibilityType.INCOMPATIBLE;
                }
                case BOOLEAN: {
                    return SchemaCompatibilityType.INCOMPATIBLE;
                }
                case INT: {
                    return SchemaCompatibilityType.INCOMPATIBLE;
                }
                case LONG: {
                    return writer.getType() == Schema.Type.INT ? SchemaCompatibilityType.COMPATIBLE : SchemaCompatibilityType.INCOMPATIBLE;
                }
                case FLOAT: {
                    return writer.getType() == Schema.Type.INT || writer.getType() == Schema.Type.LONG ? SchemaCompatibilityType.COMPATIBLE : SchemaCompatibilityType.INCOMPATIBLE;
                }
                case DOUBLE: {
                    return writer.getType() == Schema.Type.INT || writer.getType() == Schema.Type.LONG || writer.getType() == Schema.Type.FLOAT ? SchemaCompatibilityType.COMPATIBLE : SchemaCompatibilityType.INCOMPATIBLE;
                }
                case BYTES: {
                    return writer.getType() == Schema.Type.STRING ? SchemaCompatibilityType.COMPATIBLE : SchemaCompatibilityType.INCOMPATIBLE;
                }
                case STRING: {
                    return writer.getType() == Schema.Type.BYTES ? SchemaCompatibilityType.COMPATIBLE : SchemaCompatibilityType.INCOMPATIBLE;
                }
                case ARRAY: {
                    return SchemaCompatibilityType.INCOMPATIBLE;
                }
                case MAP: {
                    return SchemaCompatibilityType.INCOMPATIBLE;
                }
                case FIXED: {
                    return SchemaCompatibilityType.INCOMPATIBLE;
                }
                case ENUM: {
                    return SchemaCompatibilityType.INCOMPATIBLE;
                }
                case RECORD: {
                    return SchemaCompatibilityType.INCOMPATIBLE;
                }
                case UNION: {
                    for (Schema readerBranch : reader.getTypes()) {
                        if (this.getCompatibility(readerBranch, writer) != SchemaCompatibilityType.COMPATIBLE) continue;
                        return SchemaCompatibilityType.COMPATIBLE;
                    }
                    return SchemaCompatibilityType.INCOMPATIBLE;
                }
            }
            throw new AvroRuntimeException("Unknown schema type: " + (Object)((Object)reader.getType()));
        }
    }

    private static final class ReaderWriter {
        private final Schema mReader;
        private final Schema mWriter;

        public ReaderWriter(Schema reader, Schema writer) {
            this.mReader = reader;
            this.mWriter = writer;
        }

        public Schema getReader() {
            return this.mReader;
        }

        public Schema getWriter() {
            return this.mWriter;
        }

        public int hashCode() {
            return System.identityHashCode(this.mReader) ^ System.identityHashCode(this.mWriter);
        }

        public boolean equals(Object obj) {
            if (!(obj instanceof ReaderWriter)) {
                return false;
            }
            ReaderWriter that = (ReaderWriter)obj;
            return this.mReader == that.mReader && this.mWriter == that.mWriter;
        }

        public String toString() {
            return String.format("ReaderWriter{reader:%s, writer:%s}", this.mReader, this.mWriter);
        }
    }
}

