/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.federation.router;

import java.io.IOException;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.server.federation.router.ConnectionContext;
import org.apache.hadoop.hdfs.server.federation.router.ConnectionPool;
import org.apache.hadoop.hdfs.server.federation.router.ConnectionPoolId;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.shaded.org.eclipse.jetty.util.ajax.JSON;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConnectionManager {
    private static final Logger LOG = LoggerFactory.getLogger(ConnectionManager.class);
    protected static final int MAX_NEW_CONNECTIONS = 100;
    protected static final float MIN_ACTIVE_RATIO = 0.5f;
    private final Configuration conf;
    private final int minSize = 1;
    private final int maxSize;
    private final long poolCleanupPeriodMs;
    private final long connectionCleanupPeriodMs;
    private final Map<ConnectionPoolId, ConnectionPool> pools;
    private final ReadWriteLock readWriteLock = new ReentrantReadWriteLock();
    private final Lock readLock = this.readWriteLock.readLock();
    private final Lock writeLock = this.readWriteLock.writeLock();
    private final BlockingQueue<ConnectionPool> creatorQueue = new ArrayBlockingQueue<ConnectionPool>(100);
    private final ConnectionCreator creator;
    private final ScheduledThreadPoolExecutor cleaner = new ScheduledThreadPoolExecutor(1);
    private boolean running = false;

    public ConnectionManager(Configuration config) {
        this.conf = config;
        this.maxSize = this.conf.getInt("dfs.federation.router.connection.pool-size", 64);
        this.pools = new HashMap<ConnectionPoolId, ConnectionPool>();
        this.creator = new ConnectionCreator(this.creatorQueue);
        this.creator.setDaemon(true);
        this.poolCleanupPeriodMs = this.conf.getLong("dfs.federation.router.connection.pool.clean.ms", DFSConfigKeys.DFS_ROUTER_NAMENODE_CONNECTION_POOL_CLEAN_DEFAULT);
        LOG.info("Cleaning connection pools every {} seconds", (Object)TimeUnit.MILLISECONDS.toSeconds(this.poolCleanupPeriodMs));
        this.connectionCleanupPeriodMs = this.conf.getLong("dfs.federation.router.connection.clean.ms", DFSConfigKeys.DFS_ROUTER_NAMENODE_CONNECTION_CLEAN_MS_DEFAULT);
        LOG.info("Cleaning connections every {} seconds", (Object)TimeUnit.MILLISECONDS.toSeconds(this.connectionCleanupPeriodMs));
    }

    public void start() {
        this.creator.start();
        long recyleTimeMs = Math.min(this.poolCleanupPeriodMs, this.connectionCleanupPeriodMs);
        LOG.info("Cleaning every {} seconds", (Object)TimeUnit.MILLISECONDS.toSeconds(recyleTimeMs));
        this.cleaner.scheduleAtFixedRate(new CleanupTask(), 0L, recyleTimeMs, TimeUnit.MILLISECONDS);
        this.running = true;
    }

    public void close() {
        this.creator.shutdown();
        this.cleaner.shutdown();
        this.running = false;
        this.writeLock.lock();
        try {
            for (ConnectionPool pool : this.pools.values()) {
                pool.close();
            }
            this.pools.clear();
        }
        finally {
            this.writeLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ConnectionContext getConnection(UserGroupInformation ugi, String nnAddress) throws IOException {
        ConnectionContext conn;
        if (!this.running) {
            LOG.error("Cannot get a connection to {} because the manager isn't running", (Object)nnAddress);
            return null;
        }
        ConnectionPoolId connectionId = new ConnectionPoolId(ugi, nnAddress);
        ConnectionPool pool = null;
        this.readLock.lock();
        try {
            pool = this.pools.get(connectionId);
        }
        finally {
            this.readLock.unlock();
        }
        if (pool == null) {
            this.writeLock.lock();
            try {
                pool = this.pools.get(connectionId);
                if (pool == null) {
                    pool = new ConnectionPool(this.conf, nnAddress, ugi, this.minSize, this.maxSize);
                    this.pools.put(connectionId, pool);
                }
            }
            finally {
                this.writeLock.unlock();
            }
        }
        if (!((conn = pool.getConnection()) != null && conn.isUsable() || this.creatorQueue.offer(pool))) {
            LOG.error("Cannot add more than {} connections at the same time", (Object)100);
        }
        if (conn != null && conn.isClosed()) {
            LOG.error("We got a closed connection from {}", (Object)pool);
            conn = null;
        }
        return conn;
    }

    public int getNumConnectionPools() {
        this.readLock.lock();
        try {
            int n = this.pools.size();
            return n;
        }
        finally {
            this.readLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int getNumConnections() {
        int total = 0;
        this.readLock.lock();
        try {
            for (ConnectionPool pool : this.pools.values()) {
                total += pool.getNumConnections();
            }
        }
        finally {
            this.readLock.unlock();
        }
        return total;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int getNumActiveConnections() {
        int total = 0;
        this.readLock.lock();
        try {
            for (ConnectionPool pool : this.pools.values()) {
                total += pool.getNumActiveConnections();
            }
        }
        finally {
            this.readLock.unlock();
        }
        return total;
    }

    public int getNumCreatingConnections() {
        return this.creatorQueue.size();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String getJSON() {
        TreeMap<String, String> info = new TreeMap<String, String>();
        this.readLock.lock();
        try {
            for (Map.Entry<ConnectionPoolId, ConnectionPool> entry : this.pools.entrySet()) {
                ConnectionPoolId connectionPoolId = entry.getKey();
                ConnectionPool pool = entry.getValue();
                info.put(connectionPoolId.toString(), pool.getJSON());
            }
        }
        finally {
            this.readLock.unlock();
        }
        return JSON.toString(info);
    }

    private static class ConnectionCreator
    extends Thread {
        private boolean running = true;
        private BlockingQueue<ConnectionPool> queue;

        ConnectionCreator(BlockingQueue<ConnectionPool> blockingQueue) {
            super("Connection creator");
            this.queue = blockingQueue;
        }

        @Override
        public void run() {
            while (this.running) {
                try {
                    ConnectionPool pool = this.queue.take();
                    try {
                        int total = pool.getNumConnections();
                        int active = pool.getNumActiveConnections();
                        if (pool.getNumConnections() < pool.getMaxSize() && (float)active >= 0.5f * (float)total) {
                            ConnectionContext conn = pool.newConnection();
                            pool.addConnection(conn);
                            continue;
                        }
                        LOG.debug("Cannot add more than {} connections to {}", (Object)pool.getMaxSize(), (Object)pool);
                    }
                    catch (IOException e) {
                        LOG.error("Cannot create a new connection", (Throwable)e);
                    }
                }
                catch (InterruptedException e) {
                    LOG.error("The connection creator was interrupted");
                    this.running = false;
                }
            }
        }

        public void shutdown() {
            this.running = false;
            this.interrupt();
        }
    }

    private class CleanupTask
    implements Runnable {
        private CleanupTask() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            long currentTime = Time.now();
            LinkedList<ConnectionPoolId> toRemove = new LinkedList<ConnectionPoolId>();
            ConnectionManager.this.readLock.lock();
            try {
                for (Map.Entry entry : ConnectionManager.this.pools.entrySet()) {
                    boolean isStale;
                    ConnectionPool pool = (ConnectionPool)entry.getValue();
                    long lastTimeActive = pool.getLastActiveTime();
                    boolean bl = isStale = currentTime > lastTimeActive + ConnectionManager.this.poolCleanupPeriodMs;
                    if (lastTimeActive > 0L && isStale) {
                        LOG.debug("Closing and removing stale pool {}", (Object)pool);
                        pool.close();
                        ConnectionPoolId poolId = (ConnectionPoolId)entry.getKey();
                        toRemove.add(poolId);
                        continue;
                    }
                    LOG.debug("Cleaning up {}", (Object)pool);
                    this.cleanup(pool);
                }
            }
            finally {
                ConnectionManager.this.readLock.unlock();
            }
            if (!toRemove.isEmpty()) {
                ConnectionManager.this.writeLock.lock();
                try {
                    for (ConnectionPoolId poolId : toRemove) {
                        ConnectionManager.this.pools.remove(poolId);
                    }
                }
                finally {
                    ConnectionManager.this.writeLock.unlock();
                }
            }
        }

        private void cleanup(ConnectionPool pool) {
            if (pool.getNumConnections() > pool.getMinSize()) {
                long timeSinceLastActive = Time.now() - pool.getLastActiveTime();
                int total = pool.getNumConnections();
                int active = ConnectionManager.this.getNumActiveConnections();
                if (timeSinceLastActive > ConnectionManager.this.connectionCleanupPeriodMs || (float)active < 0.5f * (float)total) {
                    List<ConnectionContext> conns = pool.removeConnections(1);
                    for (ConnectionContext conn : conns) {
                        conn.close();
                    }
                    LOG.debug("Removed connection {} used {} seconds ago. Pool has {}/{} connections", new Object[]{pool.getConnectionPoolId(), TimeUnit.MILLISECONDS.toSeconds(timeSinceLastActive), pool.getNumConnections(), pool.getMaxSize()});
                }
            }
        }
    }
}

