/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.datanode;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.DatanodeID;
import org.apache.hadoop.hdfs.protocol.DatanodeInfo;
import org.apache.hadoop.hdfs.protocol.ErasureCodingPolicy;
import org.apache.hadoop.hdfs.protocol.ExtendedBlock;
import org.apache.hadoop.hdfs.protocol.RecoveryInProgressException;
import org.apache.hadoop.hdfs.protocolPB.DatanodeProtocolClientSideTranslatorPB;
import org.apache.hadoop.hdfs.server.common.HdfsServerConstants;
import org.apache.hadoop.hdfs.server.datanode.BPOfferService;
import org.apache.hadoop.hdfs.server.datanode.DNConf;
import org.apache.hadoop.hdfs.server.datanode.DataNode;
import org.apache.hadoop.hdfs.server.protocol.BlockRecoveryCommand;
import org.apache.hadoop.hdfs.server.protocol.InterDatanodeProtocol;
import org.apache.hadoop.hdfs.server.protocol.ReplicaRecoveryInfo;
import org.apache.hadoop.hdfs.util.StripedBlockUtil;
import org.apache.hadoop.ipc.RemoteException;
import org.apache.hadoop.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.shaded.com.google.common.base.Joiner;
import org.apache.hadoop.shaded.com.google.common.base.Preconditions;
import org.apache.hadoop.util.Daemon;
import org.slf4j.Logger;

@InterfaceAudience.Private
public class BlockRecoveryWorker {
    public static final Logger LOG = DataNode.LOG;
    private final DataNode datanode;
    private final Configuration conf;
    private final DNConf dnConf;

    BlockRecoveryWorker(DataNode datanode) {
        this.datanode = datanode;
        this.conf = datanode.getConf();
        this.dnConf = datanode.getDnConf();
    }

    private DatanodeID getDatanodeID(String bpid) throws IOException {
        BPOfferService bpos = this.datanode.getBPOfferService(bpid);
        if (bpos == null) {
            throw new IOException("No block pool offer service for bpid=" + bpid);
        }
        return new DatanodeID((DatanodeID)bpos.bpRegistration);
    }

    private static void logRecoverBlock(String who, BlockRecoveryCommand.RecoveringBlock rb) {
        ExtendedBlock block = rb.getBlock();
        Object[] targets = rb.getLocations();
        LOG.info("BlockRecoveryWorker: " + who + " calls recoverBlock(" + block + ", targets=[" + Joiner.on((String)", ").join(targets) + "], newGenerationStamp=" + rb.getNewGenerationStamp() + ", newBlock=" + rb.getNewBlock() + ", isStriped=" + rb.isStriped() + ")");
    }

    private static ReplicaRecoveryInfo callInitReplicaRecovery(InterDatanodeProtocol datanode, BlockRecoveryCommand.RecoveringBlock rBlock) throws IOException {
        try {
            return datanode.initReplicaRecovery(rBlock);
        }
        catch (RemoteException re) {
            throw re.unwrapRemoteException();
        }
    }

    DatanodeProtocolClientSideTranslatorPB getActiveNamenodeForBP(String bpid) throws IOException {
        BPOfferService bpos = this.datanode.getBPOfferService(bpid);
        if (bpos == null) {
            throw new IOException("No block pool offer service for bpid=" + bpid);
        }
        DatanodeProtocolClientSideTranslatorPB activeNN = bpos.getActiveNN();
        if (activeNN == null) {
            throw new IOException("Block pool " + bpid + " has not recognized an active NN");
        }
        return activeNN;
    }

    public Daemon recoverBlocks(final String who, final Collection<BlockRecoveryCommand.RecoveringBlock> blocks) {
        Daemon d = new Daemon(this.datanode.threadGroup, new Runnable(){

            @Override
            public void run() {
                for (BlockRecoveryCommand.RecoveringBlock b : blocks) {
                    try {
                        BlockRecoveryWorker.logRecoverBlock(who, b);
                        if (b.isStriped()) {
                            new RecoveryTaskStriped((BlockRecoveryCommand.RecoveringStripedBlock)b).recover();
                            continue;
                        }
                        new RecoveryTaskContiguous(b).recover();
                    }
                    catch (IOException e) {
                        LOG.warn("recoverBlocks FAILED: " + (Object)((Object)b), (Throwable)e);
                    }
                }
            }
        });
        d.start();
        return d;
    }

    public class RecoveryTaskStriped {
        private final BlockRecoveryCommand.RecoveringBlock rBlock;
        private final ExtendedBlock block;
        private final String bpid;
        private final DatanodeInfo[] locs;
        private final long recoveryId;
        private final byte[] blockIndices;
        private final ErasureCodingPolicy ecPolicy;

        RecoveryTaskStriped(BlockRecoveryCommand.RecoveringStripedBlock rBlock) {
            this.rBlock = rBlock;
            Preconditions.checkArgument((rBlock.getNewBlock() == null ? 1 : 0) != 0);
            this.block = rBlock.getBlock();
            this.bpid = this.block.getBlockPoolId();
            this.locs = rBlock.getLocations();
            this.recoveryId = rBlock.getNewGenerationStamp();
            this.blockIndices = rBlock.getBlockIndices();
            this.ecPolicy = rBlock.getErasureCodingPolicy();
        }

        protected void recover() throws IOException {
            this.checkLocations(this.locs.length);
            HashMap<Long, BlockRecord> syncBlocks = new HashMap<Long, BlockRecord>(this.locs.length);
            int dataBlkNum = this.ecPolicy.getNumDataUnits();
            int totalBlkNum = dataBlkNum + this.ecPolicy.getNumParityUnits();
            for (int i = 0; i < this.locs.length; ++i) {
                DatanodeInfo id = this.locs[i];
                try {
                    BlockRecord existing;
                    DatanodeID bpReg = BlockRecoveryWorker.this.getDatanodeID(this.bpid);
                    InterDatanodeProtocol proxyDN = bpReg.equals((Object)id) ? BlockRecoveryWorker.this.datanode : DataNode.createInterDataNodeProtocolProxy((DatanodeID)id, BlockRecoveryWorker.this.conf, ((BlockRecoveryWorker)BlockRecoveryWorker.this).dnConf.socketTimeout, ((BlockRecoveryWorker)BlockRecoveryWorker.this).dnConf.connectToDnViaHostname);
                    ExtendedBlock internalBlk = new ExtendedBlock(this.block);
                    long blockId = this.block.getBlockId() + (long)this.blockIndices[i];
                    internalBlk.setBlockId(blockId);
                    ReplicaRecoveryInfo info = BlockRecoveryWorker.callInitReplicaRecovery(proxyDN, new BlockRecoveryCommand.RecoveringBlock(internalBlk, null, this.recoveryId));
                    if (info == null || info.getGenerationStamp() < this.block.getGenerationStamp() || info.getNumBytes() <= 0L || (existing = (BlockRecord)syncBlocks.get(blockId)) != null && info.getNumBytes() <= existing.rInfo.getNumBytes()) continue;
                    syncBlocks.put(blockId, new BlockRecord((DatanodeID)id, proxyDN, info));
                    continue;
                }
                catch (RecoveryInProgressException ripE) {
                    InterDatanodeProtocol.LOG.warn((Object)("Recovery for replica " + this.block + " on data-node " + id + " is already in progress. Recovery id = " + this.rBlock.getNewGenerationStamp() + " is aborted."), (Throwable)ripE);
                    return;
                }
                catch (IOException e) {
                    InterDatanodeProtocol.LOG.warn((Object)("Failed to recover block (block=" + this.block + ", datanode=" + id + ")"), (Throwable)e);
                }
            }
            this.checkLocations(syncBlocks.size());
            long safeLength = this.getSafeLength(syncBlocks);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Recovering block " + this.block + ", length=" + this.block.getNumBytes() + ", safeLength=" + safeLength + ", syncList=" + syncBlocks);
            }
            ArrayList<BlockRecord> rurList = new ArrayList<BlockRecord>(this.locs.length);
            for (BlockRecord r : syncBlocks.values()) {
                int blockIndex = (int)(r.rInfo.getBlockId() & 0xFL);
                long newSize = StripedBlockUtil.getInternalBlockLength((long)safeLength, (int)this.ecPolicy.getCellSize(), (int)dataBlkNum, (int)blockIndex);
                if (r.rInfo.getNumBytes() < newSize) continue;
                rurList.add(r);
            }
            assert (rurList.size() >= dataBlkNum) : "incorrect safe length";
            this.truncatePartialBlock(rurList, safeLength);
            DatanodeID[] newLocs = new DatanodeID[totalBlkNum];
            String[] newStorages = new String[totalBlkNum];
            for (int i = 0; i < totalBlkNum; ++i) {
                newLocs[this.blockIndices[i]] = DatanodeID.EMPTY_DATANODE_ID;
                newStorages[this.blockIndices[i]] = "";
            }
            for (BlockRecord r : rurList) {
                int index = (int)(r.rInfo.getBlockId() & 0xFL);
                newLocs[index] = r.id;
                newStorages[index] = r.storageID;
            }
            ExtendedBlock newBlock = new ExtendedBlock(this.bpid, this.block.getBlockId(), safeLength, this.recoveryId);
            DatanodeProtocolClientSideTranslatorPB nn = BlockRecoveryWorker.this.getActiveNamenodeForBP(this.bpid);
            nn.commitBlockSynchronization(this.block, newBlock.getGenerationStamp(), newBlock.getNumBytes(), true, false, newLocs, newStorages);
        }

        private void truncatePartialBlock(List<BlockRecord> rurList, long safeLength) throws IOException {
            int cellSize = this.ecPolicy.getCellSize();
            int dataBlkNum = this.ecPolicy.getNumDataUnits();
            ArrayList<DatanodeID> failedList = new ArrayList<DatanodeID>();
            for (BlockRecord r : rurList) {
                int blockIndex = (int)(r.rInfo.getBlockId() & 0xFL);
                long newSize = StripedBlockUtil.getInternalBlockLength((long)safeLength, (int)cellSize, (int)dataBlkNum, (int)blockIndex);
                try {
                    r.updateReplicaUnderRecovery(this.bpid, this.recoveryId, r.rInfo.getBlockId(), newSize);
                }
                catch (IOException e) {
                    InterDatanodeProtocol.LOG.warn((Object)("Failed to updateBlock (newblock=, datanode=" + r.id + ")"), (Throwable)e);
                    failedList.add(r.id);
                }
            }
            if (!failedList.isEmpty()) {
                throw new IOException("Cannot recover " + this.block + ", the following datanodes failed: " + failedList);
            }
        }

        @VisibleForTesting
        long getSafeLength(Map<Long, BlockRecord> syncBlocks) {
            int dataBlkNum = this.ecPolicy.getNumDataUnits();
            Preconditions.checkArgument((syncBlocks.size() >= dataBlkNum ? 1 : 0) != 0);
            long[] blockLengths = new long[syncBlocks.size()];
            int i = 0;
            for (BlockRecord r : syncBlocks.values()) {
                ReplicaRecoveryInfo rInfo = r.getReplicaRecoveryInfo();
                blockLengths[i++] = rInfo.getNumBytes();
            }
            return StripedBlockUtil.getSafeLength((ErasureCodingPolicy)this.ecPolicy, (long[])blockLengths);
        }

        private void checkLocations(int locationCount) throws IOException {
            if (locationCount < this.ecPolicy.getNumDataUnits()) {
                throw new IOException(this.block + " has no enough internal blocks, unable to start recovery. Locations=" + Arrays.asList(this.locs));
            }
        }
    }

    class RecoveryTaskContiguous {
        private final BlockRecoveryCommand.RecoveringBlock rBlock;
        private final ExtendedBlock block;
        private final String bpid;
        private final DatanodeInfo[] locs;
        private final long recoveryId;

        RecoveryTaskContiguous(BlockRecoveryCommand.RecoveringBlock rBlock) {
            this.rBlock = rBlock;
            this.block = rBlock.getBlock();
            this.bpid = this.block.getBlockPoolId();
            this.locs = rBlock.getLocations();
            this.recoveryId = rBlock.getNewGenerationStamp();
        }

        protected void recover() throws IOException {
            ArrayList<BlockRecord> syncList = new ArrayList<BlockRecord>(this.locs.length);
            int errorCount = 0;
            int candidateReplicaCnt = 0;
            for (DatanodeInfo id : this.locs) {
                try {
                    DatanodeID bpReg = BlockRecoveryWorker.this.getDatanodeID(this.bpid);
                    DataNode proxyDN = bpReg.equals((Object)id) ? BlockRecoveryWorker.this.datanode : DataNode.createInterDataNodeProtocolProxy((DatanodeID)id, BlockRecoveryWorker.this.conf, ((BlockRecoveryWorker)BlockRecoveryWorker.this).dnConf.socketTimeout, ((BlockRecoveryWorker)BlockRecoveryWorker.this).dnConf.connectToDnViaHostname);
                    ReplicaRecoveryInfo info = BlockRecoveryWorker.callInitReplicaRecovery(proxyDN, this.rBlock);
                    if (info != null && info.getGenerationStamp() >= this.block.getGenerationStamp() && info.getNumBytes() > 0L) {
                        ++candidateReplicaCnt;
                        if (info.getOriginalReplicaState().getValue() <= HdfsServerConstants.ReplicaState.RWR.getValue()) {
                            syncList.add(new BlockRecord((DatanodeID)id, proxyDN, info));
                            continue;
                        }
                        if (!LOG.isDebugEnabled()) continue;
                        LOG.debug("Block recovery: Ignored replica with invalid original state: " + (Object)((Object)info) + " from DataNode: " + id);
                        continue;
                    }
                    if (!LOG.isDebugEnabled()) continue;
                    if (info == null) {
                        LOG.debug("Block recovery: DataNode: " + id + " does not have replica for block: " + this.block);
                        continue;
                    }
                    LOG.debug("Block recovery: Ignored replica with invalid generation stamp or length: " + (Object)((Object)info) + " from DataNode: " + id);
                }
                catch (RecoveryInProgressException ripE) {
                    InterDatanodeProtocol.LOG.warn((Object)("Recovery for replica " + this.block + " on data-node " + id + " is already in progress. Recovery id = " + this.rBlock.getNewGenerationStamp() + " is aborted."), (Throwable)ripE);
                    return;
                }
                catch (IOException e) {
                    ++errorCount;
                    InterDatanodeProtocol.LOG.warn((Object)("Failed to recover block (block=" + this.block + ", datanode=" + id + ")"), (Throwable)e);
                }
            }
            if (errorCount == this.locs.length) {
                throw new IOException("All datanodes failed: block=" + this.block + ", datanodeids=" + Arrays.asList(this.locs));
            }
            if (candidateReplicaCnt > 0 && syncList.isEmpty()) {
                throw new IOException("Found " + candidateReplicaCnt + " replica(s) for block " + this.block + " but none is in " + HdfsServerConstants.ReplicaState.RWR.name() + " or better state. datanodeids=" + Arrays.asList(this.locs));
            }
            this.syncBlock(syncList);
        }

        void syncBlock(List<BlockRecord> syncList) throws IOException {
            DatanodeProtocolClientSideTranslatorPB nn = BlockRecoveryWorker.this.getActiveNamenodeForBP(this.block.getBlockPoolId());
            boolean isTruncateRecovery = this.rBlock.getNewBlock() != null;
            long blockId = isTruncateRecovery ? this.rBlock.getNewBlock().getBlockId() : this.block.getBlockId();
            LOG.info("BlockRecoveryWorker: block={} (length={}), isTruncateRecovery={}, syncList={}", new Object[]{this.block, this.block.getNumBytes(), isTruncateRecovery, syncList});
            if (syncList.isEmpty()) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("syncBlock for block " + this.block + ", all datanodes don't have the block or their replicas have 0 length. The block can be deleted.");
                }
                nn.commitBlockSynchronization(this.block, this.recoveryId, 0L, true, true, DatanodeID.EMPTY_ARRAY, null);
                return;
            }
            HdfsServerConstants.ReplicaState bestState = HdfsServerConstants.ReplicaState.RWR;
            long finalizedLength = -1L;
            for (BlockRecord r : syncList) {
                assert (r.rInfo.getNumBytes() > 0L) : "zero length replica";
                Object rState = r.rInfo.getOriginalReplicaState();
                if (rState.getValue() < bestState.getValue()) {
                    bestState = rState;
                }
                if (rState != HdfsServerConstants.ReplicaState.FINALIZED) continue;
                if (finalizedLength > 0L && finalizedLength != r.rInfo.getNumBytes()) {
                    throw new IOException("Inconsistent size of finalized replicas. Replica " + (Object)((Object)r.rInfo) + " expected size: " + finalizedLength);
                }
                finalizedLength = r.rInfo.getNumBytes();
            }
            ArrayList<BlockRecord> participatingList = new ArrayList<BlockRecord>();
            ExtendedBlock newBlock = new ExtendedBlock(this.bpid, blockId, -1L, this.recoveryId);
            switch (bestState) {
                case FINALIZED: {
                    assert (finalizedLength > 0L) : "finalizedLength is not positive";
                    for (BlockRecord r : syncList) {
                        Object rState = r.rInfo.getOriginalReplicaState();
                        if (rState == HdfsServerConstants.ReplicaState.FINALIZED || rState == HdfsServerConstants.ReplicaState.RBW && r.rInfo.getNumBytes() == finalizedLength) {
                            participatingList.add(r);
                        }
                        if (!LOG.isDebugEnabled()) continue;
                        LOG.debug("syncBlock replicaInfo: block=" + this.block + ", from datanode " + r.id + ", receivedState=" + ((Enum)rState).name() + ", receivedLength=" + r.rInfo.getNumBytes() + ", bestState=FINALIZED, finalizedLength=" + finalizedLength);
                    }
                    newBlock.setNumBytes(finalizedLength);
                    break;
                }
                case RBW: 
                case RWR: {
                    long minLength = Long.MAX_VALUE;
                    for (BlockRecord r : syncList) {
                        HdfsServerConstants.ReplicaState rState = r.rInfo.getOriginalReplicaState();
                        if (rState == bestState) {
                            minLength = Math.min(minLength, r.rInfo.getNumBytes());
                            participatingList.add(r);
                        }
                        if (!LOG.isDebugEnabled()) continue;
                        LOG.debug("syncBlock replicaInfo: block=" + this.block + ", from datanode " + r.id + ", receivedState=" + rState.name() + ", receivedLength=" + r.rInfo.getNumBytes() + ", bestState=" + bestState.name());
                    }
                    if (minLength == Long.MAX_VALUE) {
                        throw new IOException("Incorrect block size");
                    }
                    newBlock.setNumBytes(minLength);
                    break;
                }
                case RUR: 
                case TEMPORARY: {
                    assert (false) : "bad replica state: " + (Object)((Object)bestState);
                    break;
                }
            }
            if (isTruncateRecovery) {
                newBlock.setNumBytes(this.rBlock.getNewBlock().getNumBytes());
            }
            LOG.info("BlockRecoveryWorker: block={} (length={}), bestState={}, newBlock={} (length={}), participatingList={}", new Object[]{this.block, this.block.getNumBytes(), bestState.name(), newBlock, newBlock.getNumBytes(), participatingList});
            ArrayList<DatanodeID> failedList = new ArrayList<DatanodeID>();
            ArrayList<BlockRecord> successList = new ArrayList<BlockRecord>();
            for (BlockRecord r : participatingList) {
                try {
                    r.updateReplicaUnderRecovery(this.bpid, this.recoveryId, blockId, newBlock.getNumBytes());
                    successList.add(r);
                }
                catch (IOException e) {
                    InterDatanodeProtocol.LOG.warn((Object)("Failed to updateBlock (newblock=" + newBlock + ", datanode=" + r.id + ")"), (Throwable)e);
                    failedList.add(r.id);
                }
            }
            if (successList.isEmpty()) {
                throw new IOException("Cannot recover " + this.block + ", the following datanodes failed: " + failedList);
            }
            DatanodeID[] datanodes = new DatanodeID[successList.size()];
            String[] storages = new String[datanodes.length];
            for (int i = 0; i < datanodes.length; ++i) {
                BlockRecord r = (BlockRecord)successList.get(i);
                datanodes[i] = r.id;
                storages[i] = r.storageID;
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("Datanode triggering commitBlockSynchronization, block=" + this.block + ", newGs=" + newBlock.getGenerationStamp() + ", newLength=" + newBlock.getNumBytes());
            }
            nn.commitBlockSynchronization(this.block, newBlock.getGenerationStamp(), newBlock.getNumBytes(), true, false, datanodes, storages);
        }
    }

    static class BlockRecord {
        private final DatanodeID id;
        private final InterDatanodeProtocol datanode;
        private final ReplicaRecoveryInfo rInfo;
        private String storageID;

        BlockRecord(DatanodeID id, InterDatanodeProtocol datanode, ReplicaRecoveryInfo rInfo) {
            this.id = id;
            this.datanode = datanode;
            this.rInfo = rInfo;
        }

        private void updateReplicaUnderRecovery(String bpid, long recoveryId, long newBlockId, long newLength) throws IOException {
            ExtendedBlock b = new ExtendedBlock(bpid, (Block)this.rInfo);
            this.storageID = this.datanode.updateReplicaUnderRecovery(b, recoveryId, newBlockId, newLength);
        }

        public ReplicaRecoveryInfo getReplicaRecoveryInfo() {
            return this.rInfo;
        }

        public String toString() {
            return "block:" + (Object)((Object)this.rInfo) + " node:" + this.id;
        }
    }
}

