/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.nfs.nfs3;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;
import java.io.IOException;
import java.net.URI;
import java.nio.file.FileSystemException;
import java.security.PrivilegedExceptionAction;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.hdfs.DFSClient;
import org.apache.hadoop.hdfs.DFSInputStream;
import org.apache.hadoop.hdfs.nfs.conf.NfsConfiguration;
import org.apache.hadoop.hdfs.nfs.nfs3.Nfs3Utils;
import org.apache.hadoop.io.MultipleIOException;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.util.ShutdownHookManager;

class DFSClientCache {
    private static final Log LOG = LogFactory.getLog(DFSClientCache.class);
    private final LoadingCache<DfsClientKey, DFSClient> clientCache;
    static final int DEFAULT_DFS_CLIENT_CACHE_SIZE = 256;
    private final LoadingCache<DFSInputStreamCacheKey, FSDataInputStream> inputstreamCache;
    static final int DEFAULT_DFS_INPUTSTREAM_CACHE_SIZE = 1024;
    static final int DEFAULT_DFS_INPUTSTREAM_CACHE_TTL = 600;
    private final NfsConfiguration config;
    private final HashMap<Integer, URI> namenodeUriMap;
    public static final int SHUTDOWN_HOOK_PRIORITY = 10;

    DFSClientCache(NfsConfiguration config) throws IOException {
        this(config, 256);
    }

    DFSClientCache(NfsConfiguration config, int clientCache) throws IOException {
        this.config = config;
        this.namenodeUriMap = new HashMap();
        this.prepareAddressMap();
        this.clientCache = CacheBuilder.newBuilder().maximumSize((long)clientCache).removalListener(this.clientRemovalListener()).build(this.clientLoader());
        this.inputstreamCache = CacheBuilder.newBuilder().maximumSize(1024L).expireAfterAccess(600L, TimeUnit.SECONDS).removalListener(this.inputStreamRemovalListener()).build(this.inputStreamLoader());
        ShutdownHookManager.get().addShutdownHook((Runnable)new CacheFinalizer(), 10);
    }

    private void prepareAddressMap() throws IOException {
        String[] exportsPath;
        FileSystem fs = FileSystem.get((Configuration)this.config);
        for (String exportPath : exportsPath = this.config.getStrings("nfs.export.point", new String[]{"/"})) {
            URI exportURI = Nfs3Utils.getResolvedURI(fs, exportPath);
            int namenodeId = Nfs3Utils.getNamenodeId((Configuration)this.config, exportURI);
            URI value = this.namenodeUriMap.get(namenodeId);
            if (value != null) {
                String msg = String.format("FS:%s, Namenode ID collision for path:%s nnid:%s uri being added:%s existing uri:%s", fs.getScheme(), exportPath, namenodeId, exportURI, value);
                LOG.error((Object)msg);
                throw new FileSystemException(msg);
            }
            LOG.info((Object)("Added export:" + exportPath + " FileSystem URI:" + exportURI + " with namenodeId:" + namenodeId));
            this.namenodeUriMap.put(namenodeId, exportURI);
        }
    }

    @VisibleForTesting
    public LoadingCache<DfsClientKey, DFSClient> getClientCache() {
        return this.clientCache;
    }

    synchronized void closeAll(boolean onlyAutomatic) throws IOException {
        ArrayList<IOException> exceptions = new ArrayList<IOException>();
        ConcurrentMap map = this.clientCache.asMap();
        for (Map.Entry item : map.entrySet()) {
            DFSClient client = (DFSClient)item.getValue();
            if (client == null) continue;
            try {
                client.close();
            }
            catch (IOException ioe) {
                exceptions.add(ioe);
            }
        }
        if (!exceptions.isEmpty()) {
            throw MultipleIOException.createIOException(exceptions);
        }
    }

    private CacheLoader<DfsClientKey, DFSClient> clientLoader() {
        return new CacheLoader<DfsClientKey, DFSClient>(){

            public DFSClient load(final DfsClientKey key) throws Exception {
                UserGroupInformation ugi = DFSClientCache.this.getUserGroupInformation(key.userName, UserGroupInformation.getCurrentUser());
                return (DFSClient)ugi.doAs((PrivilegedExceptionAction)new PrivilegedExceptionAction<DFSClient>(){

                    @Override
                    public DFSClient run() throws IOException {
                        URI namenodeURI = (URI)DFSClientCache.this.namenodeUriMap.get(key.namenodeId);
                        if (namenodeURI == null) {
                            throw new IOException("No namenode URI found for user:" + key.userName + " namenodeId:" + key.namenodeId);
                        }
                        return new DFSClient(namenodeURI, (Configuration)DFSClientCache.this.config);
                    }
                });
            }
        };
    }

    UserGroupInformation getUserGroupInformation(String effectiveUser, UserGroupInformation realUser) throws IOException {
        Preconditions.checkNotNull((Object)effectiveUser);
        Preconditions.checkNotNull((Object)realUser);
        realUser.checkTGTAndReloginFromKeytab();
        UserGroupInformation ugi = UserGroupInformation.createProxyUser((String)effectiveUser, (UserGroupInformation)realUser);
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)String.format("Created ugi: %s for username: %s", ugi, effectiveUser));
        }
        return ugi;
    }

    private RemovalListener<DfsClientKey, DFSClient> clientRemovalListener() {
        return new RemovalListener<DfsClientKey, DFSClient>(){

            public void onRemoval(RemovalNotification<DfsClientKey, DFSClient> notification) {
                DFSClient client = (DFSClient)notification.getValue();
                try {
                    client.close();
                }
                catch (IOException e) {
                    LOG.warn((Object)String.format("IOException when closing the DFSClient(%s), cause: %s", client, e));
                }
            }
        };
    }

    private RemovalListener<DFSInputStreamCacheKey, FSDataInputStream> inputStreamRemovalListener() {
        return new RemovalListener<DFSInputStreamCacheKey, FSDataInputStream>(){

            public void onRemoval(RemovalNotification<DFSInputStreamCacheKey, FSDataInputStream> notification) {
                try {
                    ((FSDataInputStream)notification.getValue()).close();
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            }
        };
    }

    private CacheLoader<DFSInputStreamCacheKey, FSDataInputStream> inputStreamLoader() {
        return new CacheLoader<DFSInputStreamCacheKey, FSDataInputStream>(){

            public FSDataInputStream load(DFSInputStreamCacheKey key) throws Exception {
                DFSClient client = DFSClientCache.this.getDfsClient(key.userId, key.namenodeId);
                DFSInputStream dis = client.open(key.inodePath);
                return client.createWrappedInputStream(dis);
            }
        };
    }

    DFSClient getDfsClient(String userName, int namenodeId) {
        DFSClient client = null;
        try {
            client = (DFSClient)this.clientCache.get((Object)new DfsClientKey(userName, namenodeId));
        }
        catch (ExecutionException e) {
            LOG.error((Object)("Failed to create DFSClient for user:" + userName + " Cause:" + e));
        }
        return client;
    }

    FSDataInputStream getDfsInputStream(String userName, String inodePath, int namenodeId) {
        DFSInputStreamCacheKey k = new DFSInputStreamCacheKey(userName, inodePath, namenodeId);
        FSDataInputStream s = null;
        try {
            s = (FSDataInputStream)this.inputstreamCache.get((Object)k);
        }
        catch (ExecutionException e) {
            LOG.warn((Object)("Failed to create DFSInputStream for user:" + userName + " Cause:" + e));
        }
        return s;
    }

    public void invalidateDfsInputStream(String userName, String inodePath, int namenodeId) {
        DFSInputStreamCacheKey k = new DFSInputStreamCacheKey(userName, inodePath, namenodeId);
        this.inputstreamCache.invalidate((Object)k);
    }

    private class CacheFinalizer
    implements Runnable {
        private CacheFinalizer() {
        }

        @Override
        public synchronized void run() {
            try {
                DFSClientCache.this.closeAll(true);
            }
            catch (IOException e) {
                LOG.info((Object)"DFSClientCache.closeAll() threw an exception:\n", (Throwable)e);
            }
        }
    }

    private static final class DfsClientKey {
        private final String userName;
        private final int namenodeId;

        private DfsClientKey(String userName, int namenodeId) {
            this.userName = userName;
            this.namenodeId = namenodeId;
        }

        public boolean equals(Object obj) {
            if (obj instanceof DfsClientKey) {
                DfsClientKey k = (DfsClientKey)obj;
                return this.userName.equals(k.userName) && this.namenodeId == k.namenodeId;
            }
            return false;
        }

        public int hashCode() {
            return Objects.hashCode((Object[])new Object[]{this.userName, this.namenodeId});
        }
    }

    private static final class DFSInputStreamCacheKey {
        private final String userId;
        private final String inodePath;
        private final int namenodeId;

        private DFSInputStreamCacheKey(String userId, String inodePath, int namenodeId) {
            this.userId = userId;
            this.inodePath = inodePath;
            this.namenodeId = namenodeId;
        }

        public boolean equals(Object obj) {
            if (obj instanceof DFSInputStreamCacheKey) {
                DFSInputStreamCacheKey k = (DFSInputStreamCacheKey)obj;
                return this.userId.equals(k.userId) && this.inodePath.equals(k.inodePath) && this.namenodeId == k.namenodeId;
            }
            return false;
        }

        public int hashCode() {
            return Objects.hashCode((Object[])new Object[]{this.userId, this.inodePath, this.namenodeId});
        }
    }
}

