/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.client;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import java.text.ParseException;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.hadoop.hdds.annotation.InterfaceAudience;
import org.apache.hadoop.hdds.annotation.InterfaceStability;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.ratis.conf.RatisClientConfig;
import org.apache.hadoop.hdds.scm.container.common.helpers.StorageContainerException;
import org.apache.hadoop.io.retry.RetryPolicies;
import org.apache.hadoop.io.retry.RetryPolicy;
import org.apache.hadoop.ozone.OzoneConsts;
import org.apache.ratis.protocol.AlreadyClosedException;
import org.apache.ratis.protocol.GroupMismatchException;
import org.apache.ratis.protocol.NotReplicatedException;
import org.apache.ratis.protocol.RaftRetryFailureException;

@InterfaceAudience.Public
@InterfaceStability.Unstable
public final class HddsClientUtils {
    private static final List<Class<? extends Exception>> EXCEPTION_LIST = ImmutableList.builder().add(TimeoutException.class).add(StorageContainerException.class).add(RaftRetryFailureException.class).add(AlreadyClosedException.class).add(GroupMismatchException.class).add(NotReplicatedException.class).build();
    private static final ThreadLocal<DateTimeFormatter> DATE_FORMAT = ThreadLocal.withInitial(() -> {
        DateTimeFormatter format = DateTimeFormatter.ofPattern("EEE, dd MMM yyyy HH:mm:ss zzz");
        return format.withZone(ZoneId.of("GMT"));
    });

    private HddsClientUtils() {
    }

    public static String formatDateTime(long millis) {
        ZonedDateTime dateTime = ZonedDateTime.ofInstant(Instant.ofEpochMilli(millis), DATE_FORMAT.get().getZone());
        return DATE_FORMAT.get().format(dateTime);
    }

    public static long formatDateTime(String date) throws ParseException {
        Preconditions.checkNotNull((Object)date, (Object)"Date string should not be null.");
        return ZonedDateTime.parse(date, DATE_FORMAT.get()).toInstant().toEpochMilli();
    }

    private static void doNameChecks(String resName) {
        if (resName == null) {
            throw new IllegalArgumentException("Bucket or Volume name is null");
        }
        if (resName.length() < 3 || resName.length() > 63) {
            throw new IllegalArgumentException("Bucket or Volume length is illegal, valid length is 3-63 characters");
        }
        if (resName.charAt(0) == '.' || resName.charAt(0) == '-') {
            throw new IllegalArgumentException("Bucket or Volume name cannot start with a period or dash");
        }
        if (resName.charAt(resName.length() - 1) == '.' || resName.charAt(resName.length() - 1) == '-') {
            throw new IllegalArgumentException("Bucket or Volume name cannot end with a period or dash");
        }
    }

    private static boolean isSupportedCharacter(char c) {
        return c == '.' || c == '-' || Character.isLowerCase(c) || Character.isDigit(c);
    }

    private static void doCharacterChecks(char currChar, char prev) {
        if (Character.isUpperCase(currChar)) {
            throw new IllegalArgumentException("Bucket or Volume name does not support uppercase characters");
        }
        if (!HddsClientUtils.isSupportedCharacter(currChar)) {
            throw new IllegalArgumentException("Bucket or Volume name has an unsupported character : " + currChar);
        }
        if (prev == '.' && currChar == '.') {
            throw new IllegalArgumentException("Bucket or Volume name should not have two contiguous periods");
        }
        if (prev == '-' && currChar == '.') {
            throw new IllegalArgumentException("Bucket or Volume name should not have period after dash");
        }
        if (prev == '.' && currChar == '-') {
            throw new IllegalArgumentException("Bucket or Volume name should not have dash after period");
        }
    }

    public static void verifyResourceName(String resName) {
        HddsClientUtils.doNameChecks(resName);
        boolean isIPv4 = true;
        char prev = '\u0000';
        for (int index = 0; index < resName.length(); ++index) {
            char currChar = resName.charAt(index);
            if (currChar != '.') {
                isIPv4 = currChar >= '0' && currChar <= '9' && isIPv4;
            }
            HddsClientUtils.doCharacterChecks(currChar, prev);
            prev = currChar;
        }
        if (isIPv4) {
            throw new IllegalArgumentException("Bucket or Volume name cannot be an IPv4 address or all numeric");
        }
    }

    public static void verifyResourceName(String ... resourceNames) {
        for (String resourceName : resourceNames) {
            HddsClientUtils.verifyResourceName(resourceName);
        }
    }

    public static void verifyKeyName(String keyName) {
        if (keyName == null) {
            throw new IllegalArgumentException("Key name is null");
        }
        if (!OzoneConsts.KEYNAME_ILLEGAL_CHARACTER_CHECK_REGEX.matcher(keyName).matches()) {
            throw new IllegalArgumentException("Invalid key name: " + keyName);
        }
    }

    public static <T> void checkNotNull(T ... references) {
        for (T ref : references) {
            Preconditions.checkNotNull(ref);
        }
    }

    public static int getListCacheSize(ConfigurationSource conf) {
        return conf.getInt("ozone.client.list.cache", 1000);
    }

    public static String getS3VolumeName(ConfigurationSource conf) {
        return conf.get("ozone.s3g.volume.name", "s3v");
    }

    public static int getMaxOutstandingRequests(ConfigurationSource config) {
        return ((RatisClientConfig.RaftConfig)config.getObject(RatisClientConfig.RaftConfig.class)).getMaxOutstandingRequests();
    }

    public static Throwable checkForException(Exception e) {
        Throwable t;
        for (t = e; t != null && t.getCause() != null; t = t.getCause()) {
            for (Class<? extends Exception> cls : HddsClientUtils.getExceptionList()) {
                if (!cls.isInstance(t)) continue;
                return t;
            }
        }
        return t;
    }

    public static RetryPolicy createRetryPolicy(int maxRetryCount, long retryInterval) {
        return RetryPolicies.retryUpToMaximumCountWithFixedSleep((int)maxRetryCount, (long)retryInterval, (TimeUnit)TimeUnit.MILLISECONDS);
    }

    public static Map<Class<? extends Throwable>, RetryPolicy> getRetryPolicyByException(int maxRetryCount, long retryInterval) {
        HashMap<Class<? extends Throwable>, RetryPolicy> policyMap = new HashMap<Class<? extends Throwable>, RetryPolicy>();
        for (Class<? extends Exception> ex : EXCEPTION_LIST) {
            if (ex == TimeoutException.class || ex == RaftRetryFailureException.class) {
                policyMap.put(ex, HddsClientUtils.createRetryPolicy(maxRetryCount, 0L));
                continue;
            }
            policyMap.put(ex, HddsClientUtils.createRetryPolicy(maxRetryCount, retryInterval));
        }
        policyMap.put(Exception.class, HddsClientUtils.createRetryPolicy(maxRetryCount, retryInterval));
        return policyMap;
    }

    public static List<Class<? extends Exception>> getExceptionList() {
        return EXCEPTION_LIST;
    }
}

