/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.replication.regionserver;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.LongAccumulator;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.Cell;
import org.apache.hadoop.hbase.CellUtil;
import org.apache.hadoop.hbase.replication.ReplicationEndpoint;
import org.apache.hadoop.hbase.replication.ReplicationUtils;
import org.apache.hadoop.hbase.replication.regionserver.ReplicationRuntimeException;
import org.apache.hadoop.hbase.replication.regionserver.ReplicationSource;
import org.apache.hadoop.hbase.replication.regionserver.ReplicationSourceLogQueue;
import org.apache.hadoop.hbase.replication.regionserver.ReplicationSourceWALReader;
import org.apache.hadoop.hbase.replication.regionserver.WALEntryBatch;
import org.apache.hadoop.hbase.shaded.protobuf.generated.WALProtos;
import org.apache.hadoop.hbase.util.Threads;
import org.apache.hadoop.hbase.wal.WAL;
import org.apache.hadoop.hbase.wal.WALEdit;
import org.apache.hbase.thirdparty.com.google.protobuf.ProtocolStringList;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class ReplicationSourceShipper
extends Thread {
    private static final Logger LOG = LoggerFactory.getLogger(ReplicationSourceShipper.class);
    private final Configuration conf;
    protected final String walGroupId;
    protected final ReplicationSourceLogQueue logQueue;
    private final ReplicationSource source;
    private volatile long currentPosition = -1L;
    private Path currentPath;
    private volatile WorkerState state;
    protected ReplicationSourceWALReader entryReader;
    protected final long sleepForRetries;
    protected final int maxRetriesMultiplier;
    private final int DEFAULT_TIMEOUT = 20000;
    private final int getEntriesTimeout;
    private final int shipEditsTimeout;

    public ReplicationSourceShipper(Configuration conf, String walGroupId, ReplicationSourceLogQueue logQueue, ReplicationSource source) {
        this.conf = conf;
        this.walGroupId = walGroupId;
        this.logQueue = logQueue;
        this.source = source;
        this.sleepForRetries = this.conf.getLong("replication.source.sleepforretries", 1000L);
        this.maxRetriesMultiplier = this.conf.getInt("replication.source.maxretriesmultiplier", 300);
        this.getEntriesTimeout = this.conf.getInt("replication.source.getEntries.timeout", 20000);
        this.shipEditsTimeout = this.conf.getInt("replication.source.shipedits.timeout", 60000);
    }

    @Override
    public final void run() {
        this.setWorkerState(WorkerState.RUNNING);
        LOG.info("Running ReplicationSourceShipper Thread for wal group: {}", (Object)this.walGroupId);
        while (this.isActive()) {
            if (!this.source.isPeerEnabled()) {
                this.sleepForRetries("Replication is disabled", 1);
                continue;
            }
            try {
                WALEntryBatch entryBatch = this.entryReader.poll(this.getEntriesTimeout);
                LOG.debug("Shipper from source {} got entry batch from reader: {}", (Object)this.source.getQueueId(), (Object)entryBatch);
                if (entryBatch == null) continue;
                if (entryBatch == WALEntryBatch.NO_MORE_DATA) {
                    this.noMoreData();
                    continue;
                }
                this.shipEdits(entryBatch);
            }
            catch (InterruptedException | ReplicationRuntimeException e) {
                LOG.warn("Interrupted while waiting for next replication entry batch", (Throwable)e);
                Thread.currentThread().interrupt();
            }
        }
        if (!this.isFinished()) {
            this.setWorkerState(WorkerState.STOPPED);
        } else {
            this.source.workerThreads.remove(this.walGroupId);
            this.postFinish();
        }
    }

    protected void noMoreData() {
    }

    protected void postFinish() {
    }

    private int getBatchEntrySizeExcludeBulkLoad(WALEntryBatch entryBatch) {
        int totalSize = 0;
        for (WAL.Entry entry : entryBatch.getWalEntries()) {
            totalSize = (int)((long)totalSize + ReplicationSourceWALReader.getEntrySizeExcludeBulkLoad(entry));
        }
        return totalSize;
    }

    private void shipEdits(WALEntryBatch entryBatch) {
        List<WAL.Entry> entries = entryBatch.getWalEntries();
        int sleepMultiplier = 0;
        if (entries.isEmpty()) {
            this.updateLogPosition(entryBatch);
            return;
        }
        int currentSize = (int)entryBatch.getHeapSize();
        int sizeExcludeBulkLoad = this.getBatchEntrySizeExcludeBulkLoad(entryBatch);
        this.source.getSourceMetrics().setTimeStampNextToReplicate(entries.get(entries.size() - 1).getKey().getWriteTime());
        while (this.isActive()) {
            try {
                try {
                    this.source.tryThrottle(currentSize);
                }
                catch (InterruptedException e) {
                    LOG.debug("Interrupted while sleeping for throttling control");
                    Thread.currentThread().interrupt();
                    continue;
                }
                ReplicationEndpoint.ReplicateContext replicateContext = new ReplicationEndpoint.ReplicateContext();
                replicateContext.setEntries(entries).setSize(currentSize);
                replicateContext.setWalGroupId(this.walGroupId);
                replicateContext.setTimeout(ReplicationUtils.getAdaptiveTimeout((int)this.shipEditsTimeout, (int)sleepMultiplier));
                long startTimeNs = System.nanoTime();
                boolean replicated = this.source.getReplicationEndpoint().replicate(replicateContext);
                long endTimeNs = System.nanoTime();
                if (!replicated) continue;
                sleepMultiplier = Math.max(sleepMultiplier - 1, 0);
                for (WAL.Entry entry : entries) {
                    this.cleanUpHFileRefs(entry.getEdit());
                    LOG.trace("shipped entry {}: ", (Object)entry);
                }
                this.updateLogPosition(entryBatch);
                this.source.postShipEdits(entries, sizeExcludeBulkLoad);
                this.source.getSourceMetrics().shipBatch(entryBatch.getNbOperations(), currentSize, entryBatch.getNbHFiles());
                this.source.getSourceMetrics().setAgeOfLastShippedOp(entries.get(entries.size() - 1).getKey().getWriteTime(), this.walGroupId);
                this.source.getSourceMetrics().updateTableLevelMetrics(entryBatch.getWalEntriesWithSize());
                if (!LOG.isTraceEnabled()) break;
                LOG.debug("Replicated {} entries or {} operations in {} ms", new Object[]{entries.size(), entryBatch.getNbOperations(), (endTimeNs - startTimeNs) / 1000000L});
                break;
            }
            catch (Exception ex) {
                LOG.warn("{} threw unknown exception:", (Object)this.source.getReplicationEndpoint().getClass().getName(), (Object)ex);
                if (!this.sleepForRetries("ReplicationEndpoint threw exception", sleepMultiplier)) continue;
                ++sleepMultiplier;
            }
        }
    }

    private void cleanUpHFileRefs(WALEdit edit) throws IOException {
        String peerId = this.source.getPeerId();
        if (peerId.contains("-")) {
            peerId = peerId.split("-")[0];
        }
        ArrayList<Cell> cells = edit.getCells();
        int totalCells = cells.size();
        for (int i = 0; i < totalCells; ++i) {
            Cell cell = (Cell)cells.get(i);
            if (!CellUtil.matchingQualifier((Cell)cell, (byte[])WALEdit.BULK_LOAD)) continue;
            WALProtos.BulkLoadDescriptor bld = WALEdit.getBulkLoadDescriptor(cell);
            List stores = bld.getStoresList();
            int totalStores = stores.size();
            for (int j = 0; j < totalStores; ++j) {
                ProtocolStringList storeFileList = ((WALProtos.StoreDescriptor)stores.get(j)).getStoreFileList();
                this.source.getSourceManager().cleanUpHFileRefs(peerId, (List<String>)storeFileList);
                this.source.getSourceMetrics().decrSizeOfHFileRefsQueue(storeFileList.size());
            }
        }
    }

    private boolean updateLogPosition(WALEntryBatch batch) {
        boolean updated = false;
        if (batch.isEndOfFile() || !batch.getLastWalPath().equals((Object)this.currentPath) || batch.getLastWalPosition() != this.currentPosition) {
            this.source.logPositionAndCleanOldLogs(batch);
            updated = true;
        }
        if (batch.isEndOfFile()) {
            this.currentPath = this.entryReader.getCurrentPath();
            this.currentPosition = 0L;
        } else {
            this.currentPath = batch.getLastWalPath();
            this.currentPosition = batch.getLastWalPosition();
        }
        return updated;
    }

    public void startup(Thread.UncaughtExceptionHandler handler) {
        String name = Thread.currentThread().getName();
        Threads.setDaemonThreadRunning((Thread)this, (String)(name + ".replicationSource.shipper" + this.walGroupId + "," + this.source.getQueueId()), handler::uncaughtException);
    }

    Path getCurrentPath() {
        return this.entryReader.getCurrentPath();
    }

    long getCurrentPosition() {
        return this.currentPosition;
    }

    void setWALReader(ReplicationSourceWALReader entryReader) {
        this.entryReader = entryReader;
    }

    long getStartPosition() {
        return 0L;
    }

    protected boolean isActive() {
        return this.source.isSourceActive() && this.state == WorkerState.RUNNING && !this.isInterrupted();
    }

    protected final void setWorkerState(WorkerState state) {
        this.state = state;
    }

    void stopWorker() {
        this.setWorkerState(WorkerState.STOPPED);
    }

    public boolean isFinished() {
        return this.state == WorkerState.FINISHED;
    }

    public boolean sleepForRetries(String msg, int sleepMultiplier) {
        try {
            LOG.trace("{}, sleeping {} times {}", new Object[]{msg, this.sleepForRetries, sleepMultiplier});
            Thread.sleep(this.sleepForRetries * (long)sleepMultiplier);
        }
        catch (InterruptedException e) {
            LOG.debug("Interrupted while sleeping between retries");
            Thread.currentThread().interrupt();
        }
        return sleepMultiplier < this.maxRetriesMultiplier;
    }

    void clearWALEntryBatch() {
        long timeout = System.currentTimeMillis() + (long)this.shipEditsTimeout;
        while (this.isAlive() || this.entryReader.isAlive()) {
            try {
                if (System.currentTimeMillis() >= timeout) {
                    LOG.warn("Shipper clearWALEntryBatch method timed out whilst waiting reader/shipper thread to stop. Not cleaning buffer usage. Shipper alive: {}; Reader alive: {}", new Object[]{this.source.getPeerId(), this.isAlive(), this.entryReader.isAlive()});
                    return;
                }
                Thread.sleep(this.sleepForRetries);
            }
            catch (InterruptedException e) {
                LOG.warn("{} Interrupted while waiting {} to stop on clearWALEntryBatch. Not cleaning buffer usage: {}", new Object[]{this.source.getPeerId(), this.getName(), e});
                return;
            }
        }
        LongAccumulator totalToDecrement = new LongAccumulator((a, b) -> a + b, 0L);
        this.entryReader.entryBatchQueue.forEach(w -> {
            this.entryReader.entryBatchQueue.remove(w);
            w.getWalEntries().forEach(e -> {
                long entrySizeExcludeBulkLoad = ReplicationSourceWALReader.getEntrySizeExcludeBulkLoad(e);
                totalToDecrement.accumulate(entrySizeExcludeBulkLoad);
            });
        });
        if (LOG.isTraceEnabled()) {
            LOG.trace("Decrementing totalBufferUsed by {}B while stopping Replication WAL Readers.", (Object)totalToDecrement.longValue());
        }
        long newBufferUsed = this.source.getSourceManager().getTotalBufferUsed().addAndGet(-totalToDecrement.longValue());
        this.source.getSourceManager().getGlobalMetrics().setWALReaderEditsBufferBytes(newBufferUsed);
    }

    public static enum WorkerState {
        RUNNING,
        STOPPED,
        FINISHED;

    }
}

