/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.zookeeper;

import java.io.File;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.BindException;
import java.net.ConnectException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.net.Address;
import org.apache.hadoop.hbase.shaded.org.apache.zookeeper.client.FourLetterWordMain;
import org.apache.hadoop.hbase.shaded.org.apache.zookeeper.common.X509Exception;
import org.apache.hadoop.hbase.shaded.org.apache.zookeeper.server.NIOServerCnxnFactory;
import org.apache.hadoop.hbase.shaded.org.apache.zookeeper.server.ZooKeeperServer;
import org.apache.hadoop.hbase.shaded.org.apache.zookeeper.server.persistence.FileTxnLog;
import org.apache.hadoop.hbase.util.Threads;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Public
public class MiniZooKeeperCluster {
    private static final Logger LOG = LoggerFactory.getLogger(MiniZooKeeperCluster.class);
    private static final int TICK_TIME = 2000;
    private static final int TIMEOUT = 1000;
    private static final int DEFAULT_CONNECTION_TIMEOUT = 30000;
    private int connectionTimeout;
    public static final String LOOPBACK_HOST;
    public static final String HOST;
    private boolean started = false;
    private int defaultClientPort = 0;
    private final List<NIOServerCnxnFactory> standaloneServerFactoryList;
    private final List<ZooKeeperServer> zooKeeperServers;
    private final List<Integer> clientPortList;
    private int activeZKServerIndex;
    private int tickTime = 0;
    private final Configuration configuration;

    public MiniZooKeeperCluster() {
        this(new Configuration());
    }

    public MiniZooKeeperCluster(Configuration configuration) {
        this.configuration = configuration;
        this.activeZKServerIndex = -1;
        this.zooKeeperServers = new ArrayList<ZooKeeperServer>();
        this.clientPortList = new ArrayList<Integer>();
        this.standaloneServerFactoryList = new ArrayList<NIOServerCnxnFactory>();
        this.connectionTimeout = configuration.getInt("zookeeper.session.timeout.localHBaseCluster", 30000);
    }

    public void addClientPort(int clientPort) {
        this.clientPortList.add(clientPort);
    }

    @InterfaceAudience.Private
    public List<Integer> getClientPortList() {
        return this.clientPortList;
    }

    private boolean hasValidClientPortInList(int index) {
        return this.clientPortList.size() > index && this.clientPortList.get(index) > 0;
    }

    public void setDefaultClientPort(int clientPort) {
        if (clientPort <= 0) {
            throw new IllegalArgumentException("Invalid default ZK client port: " + clientPort);
        }
        this.defaultClientPort = clientPort;
    }

    private int selectClientPort(int seedPort) {
        int i;
        int returnClientPort = seedPort + 1;
        if (returnClientPort == 0) {
            returnClientPort = this.defaultClientPort > 0 ? this.defaultClientPort : 49152 + ThreadLocalRandom.current().nextInt(16128);
        }
        block0: do {
            for (i = 0; i < this.clientPortList.size(); ++i) {
                if (returnClientPort != this.clientPortList.get(i)) continue;
                ++returnClientPort;
                continue block0;
            }
        } while (i != this.clientPortList.size());
        return returnClientPort;
    }

    public void setTickTime(int tickTime) {
        this.tickTime = tickTime;
    }

    public int getBackupZooKeeperServerNum() {
        return this.zooKeeperServers.size() - 1;
    }

    public int getZooKeeperServerNum() {
        return this.zooKeeperServers.size();
    }

    private static void setupTestEnv() {
        System.setProperty("zookeeper.preAllocSize", "100");
        FileTxnLog.setPreallocSize(102400L);
        System.setProperty("zookeeper.4lw.commands.whitelist", "*");
    }

    public int startup(File baseDir) throws IOException, InterruptedException {
        int numZooKeeperServers = this.clientPortList.size();
        if (numZooKeeperServers == 0) {
            numZooKeeperServers = 1;
        }
        return this.startup(baseDir, numZooKeeperServers);
    }

    public int startup(File baseDir, int numZooKeeperServers) throws IOException, InterruptedException {
        if (numZooKeeperServers <= 0) {
            return -1;
        }
        MiniZooKeeperCluster.setupTestEnv();
        this.shutdown();
        int tentativePort = -1;
        for (int i = 0; i < numZooKeeperServers; ++i) {
            NIOServerCnxnFactory standaloneServerFactory;
            File dir = new File(baseDir, "zookeeper_" + i).getAbsoluteFile();
            this.createDir(dir);
            int tickTimeToUse = this.tickTime > 0 ? this.tickTime : 2000;
            int currentClientPort = this.hasValidClientPortInList(i) ? this.clientPortList.get(i) : (tentativePort = this.selectClientPort(tentativePort));
            ZooKeeperServer server = new ZooKeeperServer(dir, dir, tickTimeToUse);
            server.setMinSessionTimeout(this.configuration.getInt("hbase.zookeeper.property.minSessionTimeout", -1));
            server.setMaxSessionTimeout(this.configuration.getInt("hbase.zookeeper.property.maxSessionTimeout", -1));
            while (true) {
                try {
                    standaloneServerFactory = new NIOServerCnxnFactory();
                    String bindAddr = this.configuration.get("hbase.zookeeper.property.clientPortAddress", LOOPBACK_HOST);
                    standaloneServerFactory.configure(new InetSocketAddress(bindAddr, currentClientPort), this.configuration.getInt("hbase.zookeeper.property.maxClientCnxns", 300));
                }
                catch (BindException e) {
                    LOG.debug("Failed binding ZK Server to client port: " + currentClientPort, (Throwable)e);
                    if (this.hasValidClientPortInList(i)) {
                        return -1;
                    }
                    currentClientPort = tentativePort = this.selectClientPort(tentativePort);
                    continue;
                }
                break;
            }
            standaloneServerFactory.startup(server);
            LOG.info("Started connectionTimeout={}, dir={}, {}", new Object[]{this.connectionTimeout, dir, this.getServerConfigurationOnOneLine(server)});
            if (!MiniZooKeeperCluster.waitForServerUp(currentClientPort, this.connectionTimeout)) {
                Threads.printThreadInfo(System.out, "Why is zk standalone server not coming up?");
                throw new IOException("Waiting for startup of standalone server; server isRunning=" + server.isRunning());
            }
            if (this.clientPortList.size() <= i) {
                this.clientPortList.add(currentClientPort);
            } else if (this.clientPortList.get(i) <= 0) {
                this.clientPortList.remove(i);
                this.clientPortList.add(i, currentClientPort);
            }
            this.standaloneServerFactoryList.add(standaloneServerFactory);
            this.zooKeeperServers.add(server);
        }
        this.activeZKServerIndex = 0;
        this.started = true;
        int clientPort = this.clientPortList.get(this.activeZKServerIndex);
        LOG.info("Started MiniZooKeeperCluster and ran 'stat' on client port={}", (Object)clientPort);
        return clientPort;
    }

    private String getServerConfigurationOnOneLine(ZooKeeperServer server) {
        StringWriter sw = new StringWriter();
        try (PrintWriter pw = new PrintWriter(sw){

            @Override
            public void println(int x) {
                super.print(x);
                super.print(", ");
            }

            @Override
            public void println(String x) {
                super.print(x);
                super.print(", ");
            }
        };){
            server.dumpConf(pw);
        }
        return sw.toString();
    }

    private void createDir(File dir) throws IOException {
        try {
            if (!dir.exists()) {
                dir.mkdirs();
            }
        }
        catch (SecurityException e) {
            throw new IOException("creating dir: " + dir, e);
        }
    }

    public void shutdown() throws IOException {
        for (int i = 0; i < this.standaloneServerFactoryList.size(); ++i) {
            NIOServerCnxnFactory standaloneServerFactory = this.standaloneServerFactoryList.get(i);
            int clientPort = this.clientPortList.get(i);
            standaloneServerFactory.shutdown();
            if (MiniZooKeeperCluster.waitForServerDown(clientPort, this.connectionTimeout)) continue;
            throw new IOException("Waiting for shutdown of standalone server at port=" + clientPort + ", timeout=" + this.connectionTimeout);
        }
        this.standaloneServerFactoryList.clear();
        for (ZooKeeperServer zkServer : this.zooKeeperServers) {
            zkServer.getZKDatabase().close();
        }
        this.zooKeeperServers.clear();
        if (this.started) {
            this.started = false;
            this.activeZKServerIndex = 0;
            this.clientPortList.clear();
            LOG.info("Shutdown MiniZK cluster with all ZK servers");
        }
    }

    public int killCurrentActiveZooKeeperServer() throws IOException, InterruptedException {
        if (!this.started || this.activeZKServerIndex < 0) {
            return -1;
        }
        NIOServerCnxnFactory standaloneServerFactory = this.standaloneServerFactoryList.get(this.activeZKServerIndex);
        int clientPort = this.clientPortList.get(this.activeZKServerIndex);
        standaloneServerFactory.shutdown();
        if (!MiniZooKeeperCluster.waitForServerDown(clientPort, this.connectionTimeout)) {
            throw new IOException("Waiting for shutdown of standalone server");
        }
        this.zooKeeperServers.get(this.activeZKServerIndex).getZKDatabase().close();
        this.standaloneServerFactoryList.remove(this.activeZKServerIndex);
        this.clientPortList.remove(this.activeZKServerIndex);
        this.zooKeeperServers.remove(this.activeZKServerIndex);
        LOG.info("Kill the current active ZK servers in the cluster on client port: {}", (Object)clientPort);
        if (this.standaloneServerFactoryList.isEmpty()) {
            return -1;
        }
        clientPort = this.clientPortList.get(this.activeZKServerIndex);
        LOG.info("Activate a backup zk server in the cluster on client port: {}", (Object)clientPort);
        return clientPort;
    }

    public void killOneBackupZooKeeperServer() throws IOException, InterruptedException {
        if (!this.started || this.activeZKServerIndex < 0 || this.standaloneServerFactoryList.size() <= 1) {
            return;
        }
        int backupZKServerIndex = this.activeZKServerIndex + 1;
        NIOServerCnxnFactory standaloneServerFactory = this.standaloneServerFactoryList.get(backupZKServerIndex);
        int clientPort = this.clientPortList.get(backupZKServerIndex);
        standaloneServerFactory.shutdown();
        if (!MiniZooKeeperCluster.waitForServerDown(clientPort, this.connectionTimeout)) {
            throw new IOException("Waiting for shutdown of standalone server");
        }
        this.zooKeeperServers.get(backupZKServerIndex).getZKDatabase().close();
        this.standaloneServerFactoryList.remove(backupZKServerIndex);
        this.clientPortList.remove(backupZKServerIndex);
        this.zooKeeperServers.remove(backupZKServerIndex);
        LOG.info("Kill one backup ZK servers in the cluster on client port: {}", (Object)clientPort);
    }

    private static boolean waitForServerDown(int port, long timeout2) throws IOException {
        long start = System.currentTimeMillis();
        while (true) {
            try {
                FourLetterWordMain.send4LetterWord(HOST, port, "stat", false, (int)timeout2);
            }
            catch (IOException | X509Exception.SSLContextException e) {
                return true;
            }
            if (System.currentTimeMillis() > start + timeout2) break;
            try {
                Thread.sleep(1000L);
            }
            catch (InterruptedException e) {
                throw (InterruptedIOException)new InterruptedIOException().initCause(e);
            }
        }
        return false;
    }

    private static boolean waitForServerUp(int port, long timeout2) throws IOException {
        long start = System.currentTimeMillis();
        while (true) {
            try {
                String result = FourLetterWordMain.send4LetterWord(HOST, port, "stat", false, (int)timeout2);
                if (result.startsWith("Zookeeper version:") && !result.contains("READ-ONLY")) {
                    return true;
                }
                LOG.debug("Read {}", (Object)result);
            }
            catch (ConnectException e) {
                LOG.info("{}:{} not up: {}", new Object[]{HOST, port, e.toString()});
            }
            catch (IOException | X509Exception.SSLContextException e) {
                LOG.info("{}:{} not up", new Object[]{HOST, port, e});
            }
            if (System.currentTimeMillis() > start + timeout2) break;
            try {
                Thread.sleep(1000L);
            }
            catch (InterruptedException e) {
                throw (InterruptedIOException)new InterruptedIOException().initCause(e);
            }
        }
        return false;
    }

    public int getClientPort() {
        return this.activeZKServerIndex < 0 || this.activeZKServerIndex >= this.clientPortList.size() ? -1 : this.clientPortList.get(this.activeZKServerIndex);
    }

    public Address getAddress() {
        return Address.fromParts(HOST, this.getClientPort());
    }

    List<ZooKeeperServer> getZooKeeperServers() {
        return this.zooKeeperServers;
    }

    static {
        HOST = LOOPBACK_HOST = InetAddress.getLoopbackAddress().getHostName();
    }
}

