/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.hfile;

import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.hbase.Cell;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.fs.HFileSystem;
import org.apache.hadoop.hbase.io.ByteArrayOutputStream;
import org.apache.hadoop.hbase.io.ByteBuffAllocator;
import org.apache.hadoop.hbase.io.ByteBuffInputStream;
import org.apache.hadoop.hbase.io.ByteBufferWriterDataOutputStream;
import org.apache.hadoop.hbase.io.FSDataInputStreamWrapper;
import org.apache.hadoop.hbase.io.encoding.DataBlockEncoding;
import org.apache.hadoop.hbase.io.encoding.EncodingState;
import org.apache.hadoop.hbase.io.encoding.HFileBlockDecodingContext;
import org.apache.hadoop.hbase.io.encoding.HFileBlockDefaultDecodingContext;
import org.apache.hadoop.hbase.io.encoding.HFileBlockDefaultEncodingContext;
import org.apache.hadoop.hbase.io.encoding.HFileBlockEncodingContext;
import org.apache.hadoop.hbase.io.hfile.BlockType;
import org.apache.hadoop.hbase.io.hfile.CacheConfig;
import org.apache.hadoop.hbase.io.hfile.Cacheable;
import org.apache.hadoop.hbase.io.hfile.CacheableDeserializer;
import org.apache.hadoop.hbase.io.hfile.CacheableDeserializerIdManager;
import org.apache.hadoop.hbase.io.hfile.ChecksumUtil;
import org.apache.hadoop.hbase.io.hfile.HFile;
import org.apache.hadoop.hbase.io.hfile.HFileBlockBuilder;
import org.apache.hadoop.hbase.io.hfile.HFileContext;
import org.apache.hadoop.hbase.io.hfile.HFileContextBuilder;
import org.apache.hadoop.hbase.io.hfile.HFileDataBlockEncoder;
import org.apache.hadoop.hbase.io.hfile.NoOpDataBlockEncoder;
import org.apache.hadoop.hbase.io.hfile.ReaderContext;
import org.apache.hadoop.hbase.io.util.BlockIOUtils;
import org.apache.hadoop.hbase.nio.ByteBuff;
import org.apache.hadoop.hbase.nio.MultiByteBuff;
import org.apache.hadoop.hbase.nio.SingleByteBuff;
import org.apache.hadoop.hbase.regionserver.ShipperListener;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.ChecksumType;
import org.apache.hadoop.hbase.util.ClassSize;
import org.apache.hbase.thirdparty.com.google.common.base.Preconditions;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class HFileBlock
implements Cacheable {
    private static final Logger LOG = LoggerFactory.getLogger(HFileBlock.class);
    public static final long FIXED_OVERHEAD = ClassSize.estimateBase(HFileBlock.class, false);
    private BlockType blockType;
    private int onDiskSizeWithoutHeader;
    private int uncompressedSizeWithoutHeader;
    private long prevBlockOffset;
    private int onDiskDataSizeWithHeader;
    private ByteBuff buf;
    private HFileContext fileContext;
    private long offset = UNSET;
    private int nextBlockOnDiskSize = UNSET;
    private ByteBuffAllocator allocator;
    static final int CHECKSUM_VERIFICATION_NUM_IO_THRESHOLD = 3;
    private static int UNSET = -1;
    public static final boolean FILL_HEADER = true;
    public static final boolean DONT_FILL_HEADER = false;
    public static final int MULTI_BYTE_BUFFER_HEAP_SIZE = (int)ClassSize.estimateBase(MultiByteBuff.class, false);
    static final int BLOCK_METADATA_SPACE = 13;
    static final int CHECKSUM_SIZE = 4;
    static final byte[] DUMMY_HEADER_NO_CHECKSUM = new byte[24];
    public static final CacheableDeserializer<Cacheable> BLOCK_DESERIALIZER = new BlockDeserializer();
    private static final int DESERIALIZER_IDENTIFIER = CacheableDeserializerIdManager.registerDeserializer(BLOCK_DESERIALIZER);

    public HFileBlock(BlockType blockType, int onDiskSizeWithoutHeader, int uncompressedSizeWithoutHeader, long prevBlockOffset, ByteBuff buf, boolean fillHeader, long offset, int nextBlockOnDiskSize, int onDiskDataSizeWithHeader, HFileContext fileContext, ByteBuffAllocator allocator) {
        this.blockType = blockType;
        this.onDiskSizeWithoutHeader = onDiskSizeWithoutHeader;
        this.uncompressedSizeWithoutHeader = uncompressedSizeWithoutHeader;
        this.prevBlockOffset = prevBlockOffset;
        this.offset = offset;
        this.onDiskDataSizeWithHeader = onDiskDataSizeWithHeader;
        this.nextBlockOnDiskSize = nextBlockOnDiskSize;
        this.fileContext = fileContext;
        this.allocator = allocator;
        this.buf = buf;
        if (fillHeader) {
            this.overwriteHeader();
        }
        this.buf.rewind();
    }

    static HFileBlock createFromBuff(ByteBuff buf, boolean usesHBaseChecksum, long offset, int nextBlockOnDiskSize, HFileContext fileContext, ByteBuffAllocator allocator) throws IOException {
        int onDiskDataSizeWithHeader;
        buf.rewind();
        BlockType blockType = BlockType.read(buf);
        int onDiskSizeWithoutHeader = buf.getInt(Header.ON_DISK_SIZE_WITHOUT_HEADER_INDEX);
        int uncompressedSizeWithoutHeader = buf.getInt(Header.UNCOMPRESSED_SIZE_WITHOUT_HEADER_INDEX);
        long prevBlockOffset = buf.getLong(Header.PREV_BLOCK_OFFSET_INDEX);
        HFileContextBuilder fileContextBuilder = fileContext != null ? new HFileContextBuilder(fileContext) : new HFileContextBuilder();
        fileContextBuilder.withHBaseCheckSum(usesHBaseChecksum);
        if (usesHBaseChecksum) {
            byte checksumType = buf.get(Header.CHECKSUM_TYPE_INDEX);
            int bytesPerChecksum = buf.getInt(Header.BYTES_PER_CHECKSUM_INDEX);
            onDiskDataSizeWithHeader = buf.getInt(Header.ON_DISK_DATA_SIZE_WITH_HEADER_INDEX);
            fileContextBuilder.withChecksumType(ChecksumType.codeToType(checksumType));
            fileContextBuilder.withBytesPerCheckSum(bytesPerChecksum);
        } else {
            fileContextBuilder.withChecksumType(ChecksumType.NULL);
            fileContextBuilder.withBytesPerCheckSum(0);
            onDiskDataSizeWithHeader = onDiskSizeWithoutHeader + HFileBlock.headerSize(usesHBaseChecksum);
        }
        fileContext = fileContextBuilder.build();
        assert (usesHBaseChecksum == fileContext.isUseHBaseChecksum());
        return new HFileBlockBuilder().withBlockType(blockType).withOnDiskSizeWithoutHeader(onDiskSizeWithoutHeader).withUncompressedSizeWithoutHeader(uncompressedSizeWithoutHeader).withPrevBlockOffset(prevBlockOffset).withOffset(offset).withOnDiskDataSizeWithHeader(onDiskDataSizeWithHeader).withNextBlockOnDiskSize(nextBlockOnDiskSize).withHFileContext(fileContext).withByteBuffAllocator(allocator).withByteBuff(buf.rewind()).withShared(!buf.hasArray()).build();
    }

    private static int getOnDiskSizeWithHeader(ByteBuff headerBuf, boolean verifyChecksum) {
        return headerBuf.getInt(Header.ON_DISK_SIZE_WITHOUT_HEADER_INDEX) + HFileBlock.headerSize(verifyChecksum);
    }

    int getNextBlockOnDiskSize() {
        return this.nextBlockOnDiskSize;
    }

    @Override
    public BlockType getBlockType() {
        return this.blockType;
    }

    @Override
    public int refCnt() {
        return this.buf.refCnt();
    }

    @Override
    public HFileBlock retain() {
        this.buf.retain();
        return this;
    }

    @Override
    public boolean release() {
        return this.buf.release();
    }

    @Override
    public HFileBlock touch() {
        return this.touch(this);
    }

    @Override
    public HFileBlock touch(Object hint) {
        this.buf.touch(hint);
        return this;
    }

    short getDataBlockEncodingId() {
        if (this.blockType != BlockType.ENCODED_DATA) {
            throw new IllegalArgumentException("Querying encoder ID of a block of type other than " + (Object)((Object)BlockType.ENCODED_DATA) + ": " + (Object)((Object)this.blockType));
        }
        return this.buf.getShort(this.headerSize());
    }

    public int getOnDiskSizeWithHeader() {
        return this.onDiskSizeWithoutHeader + this.headerSize();
    }

    int getOnDiskSizeWithoutHeader() {
        return this.onDiskSizeWithoutHeader;
    }

    int getUncompressedSizeWithoutHeader() {
        return this.uncompressedSizeWithoutHeader;
    }

    long getPrevBlockOffset() {
        return this.prevBlockOffset;
    }

    private void overwriteHeader() {
        this.buf.rewind();
        this.blockType.write(this.buf);
        this.buf.putInt(this.onDiskSizeWithoutHeader);
        this.buf.putInt(this.uncompressedSizeWithoutHeader);
        this.buf.putLong(this.prevBlockOffset);
        if (this.fileContext.isUseHBaseChecksum()) {
            this.buf.put(this.fileContext.getChecksumType().getCode());
            this.buf.putInt(this.fileContext.getBytesPerChecksum());
            this.buf.putInt(this.onDiskDataSizeWithHeader);
        }
    }

    public ByteBuff getBufferWithoutHeader() {
        ByteBuff dup = this.getBufferReadOnly();
        return dup.position(this.headerSize()).slice();
    }

    public ByteBuff getBufferReadOnly() {
        ByteBuff dup = this.buf.duplicate();
        assert (dup.position() == 0);
        return dup;
    }

    public ByteBuffAllocator getByteBuffAllocator() {
        return this.allocator;
    }

    private void sanityCheckAssertion(long valueFromBuf, long valueFromField, String fieldName) throws IOException {
        if (valueFromBuf != valueFromField) {
            throw new AssertionError((Object)(fieldName + " in the buffer (" + valueFromBuf + ") is different from that in the field (" + valueFromField + ")"));
        }
    }

    private void sanityCheckAssertion(BlockType valueFromBuf, BlockType valueFromField) throws IOException {
        if (valueFromBuf != valueFromField) {
            throw new IOException("Block type stored in the buffer: " + (Object)((Object)valueFromBuf) + ", block type field: " + (Object)((Object)valueFromField));
        }
    }

    void sanityCheck() throws IOException {
        ByteBuff dup = this.buf.duplicate().rewind();
        this.sanityCheckAssertion(BlockType.read(dup), this.blockType);
        this.sanityCheckAssertion(dup.getInt(), this.onDiskSizeWithoutHeader, "onDiskSizeWithoutHeader");
        this.sanityCheckAssertion(dup.getInt(), this.uncompressedSizeWithoutHeader, "uncompressedSizeWithoutHeader");
        this.sanityCheckAssertion(dup.getLong(), this.prevBlockOffset, "prevBlockOffset");
        if (this.fileContext.isUseHBaseChecksum()) {
            this.sanityCheckAssertion(dup.get(), this.fileContext.getChecksumType().getCode(), "checksumType");
            this.sanityCheckAssertion(dup.getInt(), this.fileContext.getBytesPerChecksum(), "bytesPerChecksum");
            this.sanityCheckAssertion(dup.getInt(), this.onDiskDataSizeWithHeader, "onDiskDataSizeWithHeader");
        }
        if (dup.limit() != this.onDiskDataSizeWithHeader) {
            throw new AssertionError((Object)("Expected limit " + this.onDiskDataSizeWithHeader + ", got " + dup.limit()));
        }
        int hdrSize = this.headerSize();
        dup.rewind();
        if (dup.remaining() != this.onDiskDataSizeWithHeader && dup.remaining() != this.onDiskDataSizeWithHeader + hdrSize) {
            throw new AssertionError((Object)("Invalid buffer capacity: " + dup.remaining() + ", expected " + this.onDiskDataSizeWithHeader + " or " + (this.onDiskDataSizeWithHeader + hdrSize)));
        }
    }

    public String toString() {
        String dataBegin;
        StringBuilder sb = new StringBuilder().append("[").append("blockType=").append((Object)this.blockType).append(", fileOffset=").append(this.offset).append(", headerSize=").append(this.headerSize()).append(", onDiskSizeWithoutHeader=").append(this.onDiskSizeWithoutHeader).append(", uncompressedSizeWithoutHeader=").append(this.uncompressedSizeWithoutHeader).append(", prevBlockOffset=").append(this.prevBlockOffset).append(", isUseHBaseChecksum=").append(this.fileContext.isUseHBaseChecksum());
        if (this.fileContext.isUseHBaseChecksum()) {
            sb.append(", checksumType=").append((Object)ChecksumType.codeToType(this.buf.get(24))).append(", bytesPerChecksum=").append(this.buf.getInt(25)).append(", onDiskDataSizeWithHeader=").append(this.onDiskDataSizeWithHeader);
        } else {
            sb.append(", onDiskDataSizeWithHeader=").append(this.onDiskDataSizeWithHeader).append("(").append(this.onDiskSizeWithoutHeader).append("+").append(24).append(")");
        }
        if (this.buf.hasArray()) {
            dataBegin = Bytes.toStringBinary(this.buf.array(), this.buf.arrayOffset() + this.headerSize(), Math.min(32, this.buf.limit() - this.buf.arrayOffset() - this.headerSize()));
        } else {
            ByteBuff bufWithoutHeader = this.getBufferWithoutHeader();
            byte[] dataBeginBytes = new byte[Math.min(32, bufWithoutHeader.limit() - bufWithoutHeader.position())];
            bufWithoutHeader.get(dataBeginBytes);
            dataBegin = Bytes.toStringBinary(dataBeginBytes);
        }
        sb.append(", getOnDiskSizeWithHeader=").append(this.getOnDiskSizeWithHeader()).append(", totalChecksumBytes=").append(this.totalChecksumBytes()).append(", isUnpacked=").append(this.isUnpacked()).append(", buf=[").append(this.buf).append("]").append(", dataBeginsWith=").append(dataBegin).append(", fileContext=").append(this.fileContext).append(", nextBlockOnDiskSize=").append(this.nextBlockOnDiskSize).append("]");
        return sb.toString();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    HFileBlock unpack(HFileContext fileContext, FSReader reader) throws IOException {
        if (!fileContext.isCompressedOrEncrypted()) {
            return this;
        }
        ByteBuff newBuf = this.allocateBufferForUnpacking();
        HFileBlock unpacked = HFileBlock.shallowClone(this, newBuf);
        boolean succ = false;
        try {
            HFileBlockDecodingContext ctx = this.blockType == BlockType.ENCODED_DATA ? reader.getBlockDecodingContext() : reader.getDefaultBlockDecodingContext();
            int headerSize = this.headerSize();
            ByteBuff dup = this.buf.duplicate();
            dup.position(headerSize);
            dup = dup.slice();
            ctx.prepareDecoding(unpacked.getOnDiskDataSizeWithHeader() - headerSize, unpacked.getUncompressedSizeWithoutHeader(), unpacked.getBufferWithoutHeader(), dup);
            succ = true;
            HFileBlock hFileBlock = unpacked;
            return hFileBlock;
        }
        finally {
            if (!succ) {
                unpacked.release();
            }
        }
    }

    private ByteBuff allocateBufferForUnpacking() {
        int headerSize = this.headerSize();
        int capacityNeeded = headerSize + this.uncompressedSizeWithoutHeader;
        ByteBuff source2 = this.buf.duplicate();
        ByteBuff newBuf = this.allocator.allocate(capacityNeeded);
        source2.position(0);
        newBuf.put(0, source2, 0, headerSize);
        newBuf.limit(capacityNeeded);
        return newBuf;
    }

    public boolean isUnpacked() {
        int headerSize = this.headerSize();
        int expectedCapacity = headerSize + this.uncompressedSizeWithoutHeader;
        int bufCapacity = this.buf.remaining();
        return bufCapacity == expectedCapacity || bufCapacity == expectedCapacity + headerSize;
    }

    long getOffset() {
        if (this.offset < 0L) {
            throw new IllegalStateException("HFile block offset not initialized properly");
        }
        return this.offset;
    }

    DataInputStream getByteStream() {
        ByteBuff dup = this.buf.duplicate();
        dup.position(this.headerSize());
        return new DataInputStream(new ByteBuffInputStream(dup));
    }

    @Override
    public long heapSize() {
        long size = FIXED_OVERHEAD;
        size += this.fileContext.heapSize();
        if (this.buf != null) {
            size += (long)ClassSize.align(this.buf.capacity() + MULTI_BYTE_BUFFER_HEAP_SIZE);
        }
        return ClassSize.align(size);
    }

    public boolean isSharedMem() {
        return true;
    }

    void sanityCheckUncompressed() throws IOException {
        if (this.onDiskSizeWithoutHeader != this.uncompressedSizeWithoutHeader + this.totalChecksumBytes()) {
            throw new IOException("Using no compression but onDiskSizeWithoutHeader=" + this.onDiskSizeWithoutHeader + ", uncompressedSizeWithoutHeader=" + this.uncompressedSizeWithoutHeader + ", numChecksumbytes=" + this.totalChecksumBytes());
        }
    }

    @Override
    public int getSerializedLength() {
        if (this.buf != null) {
            return this.buf.limit() + 13;
        }
        return 0;
    }

    @Override
    public void serialize(ByteBuffer destination, boolean includeNextBlockMetadata) {
        this.buf.get(destination, 0, this.getSerializedLength() - 13);
        destination = this.addMetaData(destination, includeNextBlockMetadata);
        destination.flip();
    }

    public ByteBuffer getMetaData() {
        ByteBuffer bb = ByteBuffer.allocate(13);
        bb = this.addMetaData(bb, true);
        bb.flip();
        return bb;
    }

    private ByteBuffer addMetaData(ByteBuffer destination, boolean includeNextBlockMetadata) {
        destination.put(this.fileContext.isUseHBaseChecksum() ? (byte)1 : 0);
        destination.putLong(this.offset);
        if (includeNextBlockMetadata) {
            destination.putInt(this.nextBlockOnDiskSize);
        }
        return destination;
    }

    @Override
    public CacheableDeserializer<Cacheable> getDeserializer() {
        return BLOCK_DESERIALIZER;
    }

    public int hashCode() {
        int result = 1;
        result = result * 31 + this.blockType.hashCode();
        result = result * 31 + this.nextBlockOnDiskSize;
        result = result * 31 + (int)(this.offset ^ this.offset >>> 32);
        result = result * 31 + this.onDiskSizeWithoutHeader;
        result = result * 31 + (int)(this.prevBlockOffset ^ this.prevBlockOffset >>> 32);
        result = result * 31 + this.uncompressedSizeWithoutHeader;
        result = result * 31 + this.buf.hashCode();
        return result;
    }

    public boolean equals(Object comparison) {
        if (this == comparison) {
            return true;
        }
        if (comparison == null) {
            return false;
        }
        if (!(comparison instanceof HFileBlock)) {
            return false;
        }
        HFileBlock castedComparison = (HFileBlock)comparison;
        if (castedComparison.blockType != this.blockType) {
            return false;
        }
        if (castedComparison.nextBlockOnDiskSize != this.nextBlockOnDiskSize) {
            return false;
        }
        if (castedComparison.offset != this.offset) {
            return false;
        }
        if (castedComparison.onDiskSizeWithoutHeader != this.onDiskSizeWithoutHeader) {
            return false;
        }
        if (castedComparison.prevBlockOffset != this.prevBlockOffset) {
            return false;
        }
        if (castedComparison.uncompressedSizeWithoutHeader != this.uncompressedSizeWithoutHeader) {
            return false;
        }
        return ByteBuff.compareTo(this.buf, 0, this.buf.limit(), castedComparison.buf, 0, castedComparison.buf.limit()) == 0;
    }

    DataBlockEncoding getDataBlockEncoding() {
        if (this.blockType == BlockType.ENCODED_DATA) {
            return DataBlockEncoding.getEncodingById(this.getDataBlockEncodingId());
        }
        return DataBlockEncoding.NONE;
    }

    byte getChecksumType() {
        return this.fileContext.getChecksumType().getCode();
    }

    int getBytesPerChecksum() {
        return this.fileContext.getBytesPerChecksum();
    }

    int getOnDiskDataSizeWithHeader() {
        return this.onDiskDataSizeWithHeader;
    }

    int totalChecksumBytes() {
        if (!this.fileContext.isUseHBaseChecksum() || this.fileContext.getBytesPerChecksum() == 0) {
            return 0;
        }
        return (int)ChecksumUtil.numBytes(this.onDiskDataSizeWithHeader, this.fileContext.getBytesPerChecksum());
    }

    public int headerSize() {
        return HFileBlock.headerSize(this.fileContext.isUseHBaseChecksum());
    }

    public static int headerSize(boolean usesHBaseChecksum) {
        return usesHBaseChecksum ? 33 : 24;
    }

    byte[] getDummyHeaderForVersion() {
        return HFileBlock.getDummyHeaderForVersion(this.fileContext.isUseHBaseChecksum());
    }

    private static byte[] getDummyHeaderForVersion(boolean usesHBaseChecksum) {
        return usesHBaseChecksum ? HConstants.HFILEBLOCK_DUMMY_HEADER : DUMMY_HEADER_NO_CHECKSUM;
    }

    public HFileContext getHFileContext() {
        return this.fileContext;
    }

    static String toStringHeader(ByteBuff buf) throws IOException {
        byte[] magicBuf = new byte[Math.min(buf.limit() - buf.position(), 8)];
        buf.get(magicBuf);
        BlockType bt = BlockType.parse(magicBuf, 0, 8);
        int compressedBlockSizeNoHeader = buf.getInt();
        int uncompressedBlockSizeNoHeader = buf.getInt();
        long prevBlockOffset = buf.getLong();
        byte cksumtype = buf.get();
        long bytesPerChecksum = buf.getInt();
        long onDiskDataSizeWithHeader = buf.getInt();
        return " Header dump: magic: " + Bytes.toString(magicBuf) + " blockType " + (Object)((Object)bt) + " compressedBlockSizeNoHeader " + compressedBlockSizeNoHeader + " uncompressedBlockSizeNoHeader " + uncompressedBlockSizeNoHeader + " prevBlockOffset " + prevBlockOffset + " checksumType " + (Object)((Object)ChecksumType.codeToType(cksumtype)) + " bytesPerChecksum " + bytesPerChecksum + " onDiskDataSizeWithHeader " + onDiskDataSizeWithHeader;
    }

    private static HFileBlockBuilder createBuilder(HFileBlock blk, ByteBuff newBuff) {
        return new HFileBlockBuilder().withBlockType(blk.blockType).withOnDiskSizeWithoutHeader(blk.onDiskSizeWithoutHeader).withUncompressedSizeWithoutHeader(blk.uncompressedSizeWithoutHeader).withPrevBlockOffset(blk.prevBlockOffset).withByteBuff(newBuff).withOffset(blk.offset).withOnDiskDataSizeWithHeader(blk.onDiskDataSizeWithHeader).withNextBlockOnDiskSize(blk.nextBlockOnDiskSize).withHFileContext(blk.fileContext).withByteBuffAllocator(blk.allocator).withShared(!newBuff.hasArray());
    }

    private static HFileBlock shallowClone(HFileBlock blk, ByteBuff newBuf) {
        return HFileBlock.createBuilder(blk, newBuf).build();
    }

    static HFileBlock deepCloneOnHeap(HFileBlock blk) {
        ByteBuff deepCloned = ByteBuff.wrap(ByteBuffer.wrap(blk.buf.toBytes(0, blk.buf.limit())));
        return HFileBlock.createBuilder(blk, deepCloned).build();
    }

    static class FSReaderImpl
    implements FSReader {
        private FSDataInputStreamWrapper streamWrapper;
        private HFileBlockDecodingContext encodedBlockDecodingCtx;
        private final HFileBlockDefaultDecodingContext defaultDecodingCtx;
        private AtomicReference<PrefetchedHeader> prefetchedHeader = new AtomicReference<PrefetchedHeader>(new PrefetchedHeader());
        private long fileSize;
        protected final int hdrSize;
        private HFileSystem hfs;
        private HFileContext fileContext;
        private String pathName;
        private final ByteBuffAllocator allocator;
        private final Lock streamLock = new ReentrantLock();
        private final boolean isPreadAllBytes;

        FSReaderImpl(ReaderContext readerContext, HFileContext fileContext, ByteBuffAllocator allocator) throws IOException {
            this.fileSize = readerContext.getFileSize();
            this.hfs = readerContext.getFileSystem();
            if (readerContext.getFilePath() != null) {
                this.pathName = readerContext.getFilePath().toString();
            }
            this.fileContext = fileContext;
            this.hdrSize = HFileBlock.headerSize(fileContext.isUseHBaseChecksum());
            this.allocator = allocator;
            this.streamWrapper = readerContext.getInputStreamWrapper();
            this.streamWrapper.prepareForBlockReader(!fileContext.isUseHBaseChecksum());
            this.defaultDecodingCtx = new HFileBlockDefaultDecodingContext(fileContext);
            this.encodedBlockDecodingCtx = this.defaultDecodingCtx;
            this.isPreadAllBytes = readerContext.isPreadAllBytes();
        }

        @Override
        public BlockIterator blockRange(final long startOffset, final long endOffset) {
            final FSReaderImpl owner = this;
            return new BlockIterator(){
                private volatile boolean freed = false;
                private List<HFileBlock> blockTracker = new ArrayList<HFileBlock>();
                private long offset = startOffset;
                private long length = -1L;

                @Override
                public HFileBlock nextBlock() throws IOException {
                    if (this.offset >= endOffset) {
                        return null;
                    }
                    HFileBlock b = this.readBlockData(this.offset, this.length, false, false, true);
                    this.offset += (long)b.getOnDiskSizeWithHeader();
                    this.length = b.getNextBlockOnDiskSize();
                    HFileBlock uncompressed = b.unpack(fileContext, owner);
                    if (uncompressed != b) {
                        b.release();
                    }
                    this.blockTracker.add(uncompressed);
                    return uncompressed;
                }

                @Override
                public HFileBlock nextBlockWithBlockType(BlockType blockType) throws IOException {
                    HFileBlock blk = this.nextBlock();
                    if (blk.getBlockType() != blockType) {
                        throw new IOException("Expected block of type " + (Object)((Object)blockType) + " but found " + (Object)((Object)blk.getBlockType()));
                    }
                    return blk;
                }

                @Override
                public void freeBlocks() {
                    if (this.freed) {
                        return;
                    }
                    this.blockTracker.forEach(HFileBlock::release);
                    this.blockTracker = null;
                    this.freed = true;
                }
            };
        }

        protected boolean readAtOffset(FSDataInputStream istream, ByteBuff dest, int size, boolean peekIntoNextBlock, long fileOffset, boolean pread) throws IOException {
            if (!pread) {
                istream.seek(fileOffset);
                long realOffset = istream.getPos();
                if (realOffset != fileOffset) {
                    throw new IOException("Tried to seek to " + fileOffset + " to read " + size + " bytes, but pos=" + realOffset + " after seek");
                }
                if (!peekIntoNextBlock) {
                    BlockIOUtils.readFully(dest, istream, size);
                    return false;
                }
                if (!BlockIOUtils.readWithExtra(dest, istream, size, this.hdrSize)) {
                    return false;
                }
            } else {
                int extraSize;
                int n = extraSize = peekIntoNextBlock ? this.hdrSize : 0;
                if (!BlockIOUtils.preadWithExtra(dest, istream, fileOffset, size, extraSize, this.isPreadAllBytes)) {
                    return false;
                }
            }
            assert (peekIntoNextBlock);
            return true;
        }

        @Override
        public HFileBlock readBlockData(long offset, long onDiskSizeWithHeaderL, boolean pread, boolean updateMetrics, boolean intoHeap) throws IOException {
            boolean doVerificationThruHBaseChecksum = this.streamWrapper.shouldUseHBaseChecksum();
            FSDataInputStream is = this.streamWrapper.getStream(doVerificationThruHBaseChecksum);
            HFileBlock blk = this.readBlockDataInternal(is, offset, onDiskSizeWithHeaderL, pread, doVerificationThruHBaseChecksum, updateMetrics, intoHeap);
            if (blk == null) {
                HFile.LOG.warn("HBase checksum verification failed for file " + this.pathName + " at offset " + offset + " filesize " + this.fileSize + ". Retrying read with HDFS checksums turned on...");
                if (!doVerificationThruHBaseChecksum) {
                    String msg = "HBase checksum verification failed for file " + this.pathName + " at offset " + offset + " filesize " + this.fileSize + " but this cannot happen because doVerify is " + doVerificationThruHBaseChecksum;
                    HFile.LOG.warn(msg);
                    throw new IOException(msg);
                }
                HFile.CHECKSUM_FAILURES.increment();
                is = this.streamWrapper.fallbackToFsChecksum(3);
                doVerificationThruHBaseChecksum = false;
                blk = this.readBlockDataInternal(is, offset, onDiskSizeWithHeaderL, pread, doVerificationThruHBaseChecksum, updateMetrics, intoHeap);
                if (blk != null) {
                    HFile.LOG.warn("HDFS checksum verification succeeded for file " + this.pathName + " at offset " + offset + " filesize " + this.fileSize);
                }
            }
            if (blk == null && !doVerificationThruHBaseChecksum) {
                String msg = "readBlockData failed, possibly due to checksum verification failed for file " + this.pathName + " at offset " + offset + " filesize " + this.fileSize;
                HFile.LOG.warn(msg);
                throw new IOException(msg);
            }
            this.streamWrapper.checksumOk();
            return blk;
        }

        private static int checkAndGetSizeAsInt(long onDiskSizeWithHeaderL, int hdrSize) throws IOException {
            if (onDiskSizeWithHeaderL < (long)hdrSize && onDiskSizeWithHeaderL != -1L || onDiskSizeWithHeaderL >= Integer.MAX_VALUE) {
                throw new IOException("Invalid onDisksize=" + onDiskSizeWithHeaderL + ": expected to be at least " + hdrSize + " and at most " + Integer.MAX_VALUE + ", or -1");
            }
            return (int)onDiskSizeWithHeaderL;
        }

        private void verifyOnDiskSizeMatchesHeader(int passedIn, ByteBuff headerBuf, long offset, boolean verifyChecksum) throws IOException {
            int fromHeader = HFileBlock.getOnDiskSizeWithHeader(headerBuf, verifyChecksum);
            if (passedIn != fromHeader) {
                throw new IOException("Passed in onDiskSizeWithHeader=" + passedIn + " != " + fromHeader + ", offset=" + offset + ", fileContext=" + this.fileContext);
            }
        }

        private ByteBuff getCachedHeader(long offset) {
            PrefetchedHeader ph = this.prefetchedHeader.get();
            return ph != null && ph.offset == offset ? ph.buf : null;
        }

        private void cacheNextBlockHeader(long offset, ByteBuff onDiskBlock, int onDiskSizeWithHeader, int headerLength) {
            PrefetchedHeader ph = new PrefetchedHeader();
            ph.offset = offset;
            onDiskBlock.get(onDiskSizeWithHeader, ph.header, 0, headerLength);
            this.prefetchedHeader.set(ph);
        }

        private int getNextBlockOnDiskSize(boolean readNextHeader, ByteBuff onDiskBlock, int onDiskSizeWithHeader) {
            int nextBlockOnDiskSize = -1;
            if (readNextHeader) {
                nextBlockOnDiskSize = onDiskBlock.getIntAfterPosition(onDiskSizeWithHeader + 8) + this.hdrSize;
            }
            return nextBlockOnDiskSize;
        }

        private ByteBuff allocate(int size, boolean intoHeap) {
            return intoHeap ? ByteBuffAllocator.HEAP.allocate(size) : this.allocator.allocate(size);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        protected HFileBlock readBlockDataInternal(FSDataInputStream is, long offset, long onDiskSizeWithHeaderL, boolean pread, boolean verifyChecksum, boolean updateMetrics, boolean intoHeap) throws IOException {
            if (offset < 0L) {
                throw new IOException("Invalid offset=" + offset + " trying to read block (onDiskSize=" + onDiskSizeWithHeaderL + ")");
            }
            int onDiskSizeWithHeader = FSReaderImpl.checkAndGetSizeAsInt(onDiskSizeWithHeaderL, this.hdrSize);
            ByteBuff headerBuf = this.getCachedHeader(offset);
            LOG.trace("Reading {} at offset={}, pread={}, verifyChecksum={}, cachedHeader={}, onDiskSizeWithHeader={}", new Object[]{this.fileContext.getHFileName(), offset, pread, verifyChecksum, headerBuf, onDiskSizeWithHeader});
            boolean checksumSupport = this.fileContext.isUseHBaseChecksum();
            long startTime = System.currentTimeMillis();
            if (onDiskSizeWithHeader <= 0) {
                if (headerBuf == null) {
                    if (LOG.isTraceEnabled()) {
                        LOG.trace("Extra see to get block size!", (Throwable)new RuntimeException());
                    }
                    headerBuf = ByteBuffAllocator.HEAP.allocate(this.hdrSize);
                    this.readAtOffset(is, headerBuf, this.hdrSize, false, offset, pread);
                    headerBuf.rewind();
                }
                onDiskSizeWithHeader = HFileBlock.getOnDiskSizeWithHeader(headerBuf, checksumSupport);
            }
            int preReadHeaderSize = headerBuf == null ? 0 : this.hdrSize;
            ByteBuff onDiskBlock = this.allocate(onDiskSizeWithHeader + this.hdrSize, intoHeap);
            boolean initHFileBlockSuccess = false;
            try {
                if (headerBuf != null) {
                    onDiskBlock.put(0, headerBuf, 0, this.hdrSize).position(this.hdrSize);
                }
                boolean readNextHeader = this.readAtOffset(is, onDiskBlock, onDiskSizeWithHeader - preReadHeaderSize, true, offset + (long)preReadHeaderSize, pread);
                onDiskBlock.rewind();
                int nextBlockOnDiskSize = this.getNextBlockOnDiskSize(readNextHeader, onDiskBlock, onDiskSizeWithHeader);
                if (headerBuf == null) {
                    headerBuf = onDiskBlock.duplicate().position(0).limit(this.hdrSize);
                }
                assert (onDiskSizeWithHeader > this.hdrSize);
                this.verifyOnDiskSizeMatchesHeader(onDiskSizeWithHeader, headerBuf, offset, checksumSupport);
                ByteBuff curBlock = onDiskBlock.duplicate().position(0).limit(onDiskSizeWithHeader);
                if (verifyChecksum && !this.validateChecksum(offset, curBlock, this.hdrSize)) {
                    HFileBlock hFileBlock = null;
                    return hFileBlock;
                }
                int sizeWithoutChecksum = curBlock.getInt(Header.ON_DISK_DATA_SIZE_WITH_HEADER_INDEX);
                curBlock.limit(sizeWithoutChecksum);
                long duration = System.currentTimeMillis() - startTime;
                if (updateMetrics) {
                    HFile.updateReadLatency(duration, pread);
                }
                HFileBlock hFileBlock = HFileBlock.createFromBuff(curBlock, checksumSupport, offset, nextBlockOnDiskSize, this.fileContext, intoHeap ? ByteBuffAllocator.HEAP : this.allocator);
                if (!this.fileContext.isCompressedOrEncrypted()) {
                    hFileBlock.sanityCheckUncompressed();
                }
                LOG.trace("Read {} in {} ms", (Object)hFileBlock, (Object)duration);
                if (nextBlockOnDiskSize != -1) {
                    this.cacheNextBlockHeader(offset + (long)hFileBlock.getOnDiskSizeWithHeader(), onDiskBlock, onDiskSizeWithHeader, this.hdrSize);
                }
                initHFileBlockSuccess = true;
                HFileBlock hFileBlock2 = hFileBlock;
                return hFileBlock2;
            }
            finally {
                if (!initHFileBlockSuccess) {
                    onDiskBlock.release();
                }
            }
        }

        @Override
        public void setIncludesMemStoreTS(boolean includesMemstoreTS) {
            this.fileContext = new HFileContextBuilder(this.fileContext).withIncludesMvcc(includesMemstoreTS).build();
        }

        @Override
        public void setDataBlockEncoder(HFileDataBlockEncoder encoder) {
            this.encodedBlockDecodingCtx = encoder.newDataBlockDecodingContext(this.fileContext);
        }

        @Override
        public HFileBlockDecodingContext getBlockDecodingContext() {
            return this.encodedBlockDecodingCtx;
        }

        @Override
        public HFileBlockDecodingContext getDefaultBlockDecodingContext() {
            return this.defaultDecodingCtx;
        }

        private boolean validateChecksum(long offset, ByteBuff data, int hdrSize) {
            if (!this.fileContext.isUseHBaseChecksum()) {
                return false;
            }
            return ChecksumUtil.validateChecksum(data, this.pathName, offset, hdrSize);
        }

        @Override
        public void closeStreams() throws IOException {
            this.streamWrapper.close();
        }

        @Override
        public void unbufferStream() {
            if (this.streamLock.tryLock()) {
                try {
                    this.streamWrapper.unbuffer();
                }
                finally {
                    this.streamLock.unlock();
                }
            }
        }

        public String toString() {
            return "hfs=" + (Object)((Object)this.hfs) + ", path=" + this.pathName + ", fileContext=" + this.fileContext;
        }
    }

    private static class PrefetchedHeader {
        long offset = -1L;
        byte[] header = new byte[33];
        final ByteBuff buf = new SingleByteBuff(ByteBuffer.wrap(this.header, 0, this.header.length));

        private PrefetchedHeader() {
        }

        public String toString() {
            return "offset=" + this.offset + ", header=" + Bytes.toStringBinary(this.header);
        }
    }

    static interface FSReader {
        public HFileBlock readBlockData(long var1, long var3, boolean var5, boolean var6, boolean var7) throws IOException;

        public BlockIterator blockRange(long var1, long var3);

        public void closeStreams() throws IOException;

        public HFileBlockDecodingContext getBlockDecodingContext();

        public HFileBlockDecodingContext getDefaultBlockDecodingContext();

        public void setIncludesMemStoreTS(boolean var1);

        public void setDataBlockEncoder(HFileDataBlockEncoder var1);

        public void unbufferStream();
    }

    static interface BlockIterator {
        public HFileBlock nextBlock() throws IOException;

        public HFileBlock nextBlockWithBlockType(BlockType var1) throws IOException;

        public void freeBlocks();
    }

    static interface BlockWritable {
        public BlockType getBlockType();

        public void writeToBlock(DataOutput var1) throws IOException;
    }

    static class Writer
    implements ShipperListener {
        private State state = State.INIT;
        private final HFileDataBlockEncoder dataBlockEncoder;
        private HFileBlockEncodingContext dataBlockEncodingCtx;
        private HFileBlockDefaultEncodingContext defaultBlockEncodingCtx;
        private ByteArrayOutputStream baosInMemory;
        private BlockType blockType;
        private DataOutputStream userDataStream;
        private ByteArrayOutputStream onDiskBlockBytesWithHeader;
        private byte[] onDiskChecksum = HConstants.EMPTY_BYTE_ARRAY;
        private long startOffset;
        private long[] prevOffsetByType;
        private long prevOffset;
        private HFileContext fileContext;
        private final ByteBuffAllocator allocator;

        @Override
        public void beforeShipped() {
            if (this.getEncodingState() != null) {
                this.getEncodingState().beforeShipped();
            }
        }

        EncodingState getEncodingState() {
            return this.dataBlockEncodingCtx.getEncodingState();
        }

        public Writer(HFileDataBlockEncoder dataBlockEncoder, HFileContext fileContext) {
            this(dataBlockEncoder, fileContext, ByteBuffAllocator.HEAP);
        }

        public Writer(HFileDataBlockEncoder dataBlockEncoder, HFileContext fileContext, ByteBuffAllocator allocator) {
            if (fileContext.getBytesPerChecksum() < 33) {
                throw new RuntimeException("Unsupported value of bytesPerChecksum.  Minimum is 33 but the configured value is " + fileContext.getBytesPerChecksum());
            }
            this.allocator = allocator;
            this.dataBlockEncoder = dataBlockEncoder != null ? dataBlockEncoder : NoOpDataBlockEncoder.INSTANCE;
            this.dataBlockEncodingCtx = this.dataBlockEncoder.newDataBlockEncodingContext(HConstants.HFILEBLOCK_DUMMY_HEADER, fileContext);
            this.defaultBlockEncodingCtx = new HFileBlockDefaultEncodingContext(null, HConstants.HFILEBLOCK_DUMMY_HEADER, fileContext);
            this.baosInMemory = new ByteArrayOutputStream();
            this.prevOffsetByType = new long[BlockType.values().length];
            for (int i = 0; i < this.prevOffsetByType.length; ++i) {
                this.prevOffsetByType[i] = UNSET;
            }
            this.fileContext = fileContext;
        }

        DataOutputStream startWriting(BlockType newBlockType) throws IOException {
            if (this.state == State.BLOCK_READY && this.startOffset != -1L) {
                this.prevOffsetByType[this.blockType.getId()] = this.startOffset;
            }
            this.startOffset = -1L;
            this.blockType = newBlockType;
            this.baosInMemory.reset();
            this.baosInMemory.write(HConstants.HFILEBLOCK_DUMMY_HEADER);
            this.state = State.WRITING;
            this.userDataStream = new ByteBufferWriterDataOutputStream(this.baosInMemory);
            if (newBlockType == BlockType.DATA) {
                this.dataBlockEncoder.startBlockEncoding(this.dataBlockEncodingCtx, this.userDataStream);
            }
            return this.userDataStream;
        }

        void write(Cell cell) throws IOException {
            this.expectState(State.WRITING);
            this.dataBlockEncoder.encode(cell, this.dataBlockEncodingCtx, this.userDataStream);
        }

        void ensureBlockReady() throws IOException {
            Preconditions.checkState(this.state != State.INIT, "Unexpected state: " + (Object)((Object)this.state));
            if (this.state == State.BLOCK_READY) {
                return;
            }
            this.finishBlock();
        }

        private void finishBlock() throws IOException {
            if (this.blockType == BlockType.DATA) {
                this.dataBlockEncoder.endBlockEncoding(this.dataBlockEncodingCtx, this.userDataStream, this.baosInMemory.getBuffer(), this.blockType);
                this.blockType = this.dataBlockEncodingCtx.getBlockType();
            }
            this.userDataStream.flush();
            this.prevOffset = this.prevOffsetByType[this.blockType.getId()];
            this.state = State.BLOCK_READY;
            Bytes compressAndEncryptDat = this.blockType == BlockType.DATA || this.blockType == BlockType.ENCODED_DATA ? this.dataBlockEncodingCtx.compressAndEncrypt(this.baosInMemory.getBuffer(), 0, this.baosInMemory.size()) : this.defaultBlockEncodingCtx.compressAndEncrypt(this.baosInMemory.getBuffer(), 0, this.baosInMemory.size());
            if (compressAndEncryptDat == null) {
                compressAndEncryptDat = new Bytes(this.baosInMemory.getBuffer(), 0, this.baosInMemory.size());
            }
            if (this.onDiskBlockBytesWithHeader == null) {
                this.onDiskBlockBytesWithHeader = new ByteArrayOutputStream(compressAndEncryptDat.getLength());
            }
            this.onDiskBlockBytesWithHeader.reset();
            this.onDiskBlockBytesWithHeader.write(compressAndEncryptDat.get(), compressAndEncryptDat.getOffset(), compressAndEncryptDat.getLength());
            int numBytes = (int)ChecksumUtil.numBytes(this.onDiskBlockBytesWithHeader.size(), this.fileContext.getBytesPerChecksum());
            this.putHeader(this.onDiskBlockBytesWithHeader, this.onDiskBlockBytesWithHeader.size() + numBytes, this.baosInMemory.size(), this.onDiskBlockBytesWithHeader.size());
            if (this.onDiskChecksum.length != numBytes) {
                this.onDiskChecksum = new byte[numBytes];
            }
            ChecksumUtil.generateChecksums(this.onDiskBlockBytesWithHeader.getBuffer(), 0, this.onDiskBlockBytesWithHeader.size(), this.onDiskChecksum, 0, this.fileContext.getChecksumType(), this.fileContext.getBytesPerChecksum());
        }

        private void putHeader(byte[] dest, int offset, int onDiskSize, int uncompressedSize, int onDiskDataSize) {
            offset = this.blockType.put(dest, offset);
            offset = Bytes.putInt(dest, offset, onDiskSize - 33);
            offset = Bytes.putInt(dest, offset, uncompressedSize - 33);
            offset = Bytes.putLong(dest, offset, this.prevOffset);
            offset = Bytes.putByte(dest, offset, this.fileContext.getChecksumType().getCode());
            offset = Bytes.putInt(dest, offset, this.fileContext.getBytesPerChecksum());
            Bytes.putInt(dest, offset, onDiskDataSize);
        }

        private void putHeader(ByteBuff buff, int onDiskSize, int uncompressedSize, int onDiskDataSize) {
            buff.rewind();
            this.blockType.write(buff);
            buff.putInt(onDiskSize - 33);
            buff.putInt(uncompressedSize - 33);
            buff.putLong(this.prevOffset);
            buff.put(this.fileContext.getChecksumType().getCode());
            buff.putInt(this.fileContext.getBytesPerChecksum());
            buff.putInt(onDiskDataSize);
        }

        private void putHeader(ByteArrayOutputStream dest, int onDiskSize, int uncompressedSize, int onDiskDataSize) {
            this.putHeader(dest.getBuffer(), 0, onDiskSize, uncompressedSize, onDiskDataSize);
        }

        void writeHeaderAndData(FSDataOutputStream out) throws IOException {
            long offset = out.getPos();
            if (this.startOffset != (long)UNSET && offset != this.startOffset) {
                throw new IOException("A " + (Object)((Object)this.blockType) + " block written to a stream twice, first at offset " + this.startOffset + ", then at " + offset);
            }
            this.startOffset = offset;
            this.finishBlockAndWriteHeaderAndData((DataOutputStream)out);
        }

        protected void finishBlockAndWriteHeaderAndData(DataOutputStream out) throws IOException {
            this.ensureBlockReady();
            long startTime = System.currentTimeMillis();
            out.write(this.onDiskBlockBytesWithHeader.getBuffer(), 0, this.onDiskBlockBytesWithHeader.size());
            out.write(this.onDiskChecksum);
            HFile.updateWriteLatency(System.currentTimeMillis() - startTime);
        }

        byte[] getHeaderAndDataForTest() throws IOException {
            this.ensureBlockReady();
            byte[] output = new byte[this.onDiskBlockBytesWithHeader.size() + this.onDiskChecksum.length];
            System.arraycopy(this.onDiskBlockBytesWithHeader.getBuffer(), 0, output, 0, this.onDiskBlockBytesWithHeader.size());
            System.arraycopy(this.onDiskChecksum, 0, output, this.onDiskBlockBytesWithHeader.size(), this.onDiskChecksum.length);
            return output;
        }

        void release() {
            if (this.dataBlockEncodingCtx != null) {
                this.dataBlockEncodingCtx.close();
                this.dataBlockEncodingCtx = null;
            }
            if (this.defaultBlockEncodingCtx != null) {
                this.defaultBlockEncodingCtx.close();
                this.defaultBlockEncodingCtx = null;
            }
        }

        int getOnDiskSizeWithoutHeader() {
            this.expectState(State.BLOCK_READY);
            return this.onDiskBlockBytesWithHeader.size() + this.onDiskChecksum.length - 33;
        }

        int getOnDiskSizeWithHeader() {
            this.expectState(State.BLOCK_READY);
            return this.onDiskBlockBytesWithHeader.size() + this.onDiskChecksum.length;
        }

        int getUncompressedSizeWithoutHeader() {
            this.expectState(State.BLOCK_READY);
            return this.baosInMemory.size() - 33;
        }

        int getUncompressedSizeWithHeader() {
            this.expectState(State.BLOCK_READY);
            return this.baosInMemory.size();
        }

        boolean isWriting() {
            return this.state == State.WRITING;
        }

        public int encodedBlockSizeWritten() {
            return this.state != State.WRITING ? 0 : this.getEncodingState().getEncodedDataSizeWritten();
        }

        int blockSizeWritten() {
            return this.state != State.WRITING ? 0 : this.getEncodingState().getUnencodedDataSizeWritten();
        }

        ByteBuff cloneUncompressedBufferWithHeader() {
            this.expectState(State.BLOCK_READY);
            ByteBuff bytebuff = this.allocator.allocate(this.baosInMemory.size());
            this.baosInMemory.toByteBuff(bytebuff);
            int numBytes = (int)ChecksumUtil.numBytes(this.onDiskBlockBytesWithHeader.size(), this.fileContext.getBytesPerChecksum());
            this.putHeader(bytebuff, this.onDiskBlockBytesWithHeader.size() + numBytes, this.baosInMemory.size(), this.onDiskBlockBytesWithHeader.size());
            bytebuff.rewind();
            return bytebuff;
        }

        private ByteBuff cloneOnDiskBufferWithHeader() {
            this.expectState(State.BLOCK_READY);
            ByteBuff bytebuff = this.allocator.allocate(this.onDiskBlockBytesWithHeader.size());
            this.onDiskBlockBytesWithHeader.toByteBuff(bytebuff);
            bytebuff.rewind();
            return bytebuff;
        }

        private void expectState(State expectedState) {
            if (this.state != expectedState) {
                throw new IllegalStateException("Expected state: " + (Object)((Object)expectedState) + ", actual state: " + (Object)((Object)this.state));
            }
        }

        void writeBlock(BlockWritable bw, FSDataOutputStream out) throws IOException {
            bw.writeToBlock(this.startWriting(bw.getBlockType()));
            this.writeHeaderAndData(out);
        }

        HFileBlock getBlockForCaching(CacheConfig cacheConf) {
            HFileContext newContext = new HFileContextBuilder().withBlockSize(this.fileContext.getBlocksize()).withBytesPerCheckSum(0).withChecksumType(ChecksumType.NULL).withCompression(this.fileContext.getCompression()).withDataBlockEncoding(this.fileContext.getDataBlockEncoding()).withHBaseCheckSum(this.fileContext.isUseHBaseChecksum()).withCompressTags(this.fileContext.isCompressTags()).withIncludesMvcc(this.fileContext.isIncludesMvcc()).withIncludesTags(this.fileContext.isIncludesTags()).withColumnFamily(this.fileContext.getColumnFamily()).withTableName(this.fileContext.getTableName()).build();
            HFileBlockBuilder builder = new HFileBlockBuilder();
            ByteBuff buff = cacheConf.shouldCacheCompressed(this.blockType.getCategory()) ? this.cloneOnDiskBufferWithHeader() : this.cloneUncompressedBufferWithHeader();
            return builder.withBlockType(this.blockType).withOnDiskSizeWithoutHeader(this.getOnDiskSizeWithoutHeader()).withUncompressedSizeWithoutHeader(this.getUncompressedSizeWithoutHeader()).withPrevBlockOffset(this.prevOffset).withByteBuff(buff).withFillHeader(true).withOffset(this.startOffset).withNextBlockOnDiskSize(UNSET).withOnDiskDataSizeWithHeader(this.onDiskBlockBytesWithHeader.size() + this.onDiskChecksum.length).withHFileContext(newContext).withByteBuffAllocator(cacheConf.getByteBuffAllocator()).withShared(!buff.hasArray()).build();
        }

        private static enum State {
            INIT,
            WRITING,
            BLOCK_READY;

        }
    }

    public static final class BlockDeserializer
    implements CacheableDeserializer<Cacheable> {
        private BlockDeserializer() {
        }

        @Override
        public HFileBlock deserialize(ByteBuff buf, ByteBuffAllocator alloc) throws IOException {
            buf.limit(buf.limit() - 13).rewind();
            ByteBuff newByteBuff = buf.slice();
            buf.position(buf.limit());
            buf.limit(buf.limit() + 13);
            boolean usesChecksum = buf.get() == 1;
            long offset = buf.getLong();
            int nextBlockOnDiskSize = buf.getInt();
            return HFileBlock.createFromBuff(newByteBuff, usesChecksum, offset, nextBlockOnDiskSize, null, alloc);
        }

        @Override
        public int getDeserializerIdentifier() {
            return DESERIALIZER_IDENTIFIER;
        }
    }

    static class Header {
        static int BLOCK_MAGIC_INDEX = 0;
        static int ON_DISK_SIZE_WITHOUT_HEADER_INDEX = 8;
        static int UNCOMPRESSED_SIZE_WITHOUT_HEADER_INDEX = 12;
        static int PREV_BLOCK_OFFSET_INDEX = 16;
        static int CHECKSUM_TYPE_INDEX = 24;
        static int BYTES_PER_CHECKSUM_INDEX = 25;
        static int ON_DISK_DATA_SIZE_WITH_HEADER_INDEX = 29;

        Header() {
        }
    }
}

