/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kudu.client;

import java.io.Closeable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Random;
import org.apache.kudu.ColumnSchema;
import org.apache.kudu.Common;
import org.apache.kudu.Schema;
import org.apache.kudu.Type;
import org.apache.kudu.client.AlterTableOptions;
import org.apache.kudu.client.CreateTableOptions;
import org.apache.kudu.client.Delete;
import org.apache.kudu.client.Insert;
import org.apache.kudu.client.KuduClient;
import org.apache.kudu.client.KuduPartitioner;
import org.apache.kudu.client.KuduScanToken;
import org.apache.kudu.client.KuduScanner;
import org.apache.kudu.client.KuduScannerIterator;
import org.apache.kudu.client.KuduSession;
import org.apache.kudu.client.KuduTable;
import org.apache.kudu.client.Operation;
import org.apache.kudu.client.OperationResponse;
import org.apache.kudu.client.PartialRow;
import org.apache.kudu.client.RowResult;
import org.apache.kudu.client.TestScannerMultiTablet;
import org.apache.kudu.test.CapturingLogAppender;
import org.apache.kudu.test.ClientTestUtil;
import org.apache.kudu.test.KuduTestHarness;
import org.apache.kudu.test.RandomUtils;
import org.apache.kudu.util.DataGenerator;
import org.apache.kudu.util.Pair;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TestKuduScanner {
    private static final Logger LOG = LoggerFactory.getLogger(TestScannerMultiTablet.class);
    private static final String tableName = "TestKuduScanner";
    private static final int DIFF_FLUSH_SEC = 1;
    private KuduClient client;
    private Random random;
    private DataGenerator generator;
    @Rule
    public KuduTestHarness harness = new KuduTestHarness();

    @Before
    public void setUp() {
        this.client = this.harness.getClient();
        this.random = RandomUtils.getRandom();
        this.generator = new DataGenerator.DataGeneratorBuilder().random(this.random).build();
    }

    @Test(timeout=100000L)
    public void testIterable() throws Exception {
        KuduTable table = this.client.createTable(tableName, ClientTestUtil.getBasicSchema(), ClientTestUtil.getBasicCreateTableOptions());
        DataGenerator generator = new DataGenerator.DataGeneratorBuilder().random(RandomUtils.getRandom()).build();
        KuduSession session = this.client.newSession();
        ArrayList<Integer> insertKeys = new ArrayList<Integer>();
        int numRows = 10;
        for (int i = 0; i < numRows; ++i) {
            Insert insert = table.newInsert();
            PartialRow row = insert.getRow();
            generator.randomizeRow(row);
            insertKeys.add(row.getInt(0));
            session.apply((Operation)insert);
        }
        KuduScanner scanner = this.client.newScannerBuilder(table).build();
        HashSet<RowResult> results = new HashSet<RowResult>();
        HashSet<Integer> resultKeys = new HashSet<Integer>();
        for (RowResult rowResult : scanner) {
            results.add(rowResult);
            resultKeys.add(rowResult.getInt(0));
        }
        Assert.assertEquals((long)numRows, (long)results.size());
        Assert.assertTrue((boolean)resultKeys.containsAll(insertKeys));
        KuduScanner reuseScanner = this.client.newScannerBuilder(table).build();
        reuseScanner.setReuseRowResult(true);
        HashSet<RowResult> reuseResult = new HashSet<RowResult>();
        for (RowResult rowResult : reuseScanner) {
            reuseResult.add(rowResult);
        }
        Assert.assertEquals((long)1L, (long)reuseResult.size());
    }

    @Test(timeout=100000L)
    @KuduTestHarness.TabletServerConfig(flags={"--scanner_ttl_ms=5000", "--scanner_gc_check_interval_us=500000"})
    public void testKeepAlive() throws Exception {
        int rowCount = 500;
        long shortScannerTtlMs = 5000L;
        Schema tableSchema = new Schema(Collections.singletonList(new ColumnSchema.ColumnSchemaBuilder("key", Type.INT32).key(true).build()));
        CreateTableOptions tableOptions = new CreateTableOptions().setRangePartitionColumns(Collections.singletonList("key")).setNumReplicas(1);
        KuduTable table = this.client.createTable(tableName, tableSchema, tableOptions);
        KuduSession session = this.client.newSession();
        for (int i = 0; i < rowCount; ++i) {
            Insert insert = table.newInsert();
            PartialRow row = insert.getRow();
            row.addInt(0, i);
            session.apply((Operation)insert);
        }
        KuduScanner goodScanner = ((KuduScanner.KuduScannerBuilder)((KuduScanner.KuduScannerBuilder)this.client.newScannerBuilder(table).batchSizeBytes(100)).keepAlivePeriodMs(shortScannerTtlMs / 4L)).build();
        this.processKeepAliveScanner(goodScanner, shortScannerTtlMs);
        KuduScanner badScanner = ((KuduScanner.KuduScannerBuilder)((KuduScanner.KuduScannerBuilder)this.client.newScannerBuilder(table).batchSizeBytes(100)).keepAlivePeriodMs(shortScannerTtlMs * 2L)).build();
        try {
            this.processKeepAliveScanner(badScanner, shortScannerTtlMs);
            Assert.fail((String)"Should throw a scanner not found exception");
        }
        catch (RuntimeException ex) {
            Assert.assertTrue((boolean)ex.getMessage().matches(".*Scanner .* not found.*"));
        }
    }

    private void processKeepAliveScanner(KuduScanner scanner, long scannerTtlMs) throws Exception {
        int i = 0;
        KuduScannerIterator iterator = scanner.iterator();
        while (iterator.hasNext()) {
            iterator.next();
            if (i >= 5) continue;
            Thread.sleep(scannerTtlMs / 2L);
            ++i;
        }
    }

    @Test(timeout=100000L)
    public void testOpenScanWithDroppedPartition() throws Exception {
        Schema basicSchema = ClientTestUtil.getBasicSchema();
        String tableName = "testOpenScanWithDroppedPartition";
        PartialRow bottom = basicSchema.newPartialRow();
        bottom.addInt("key", 0);
        PartialRow middle = basicSchema.newPartialRow();
        middle.addInt("key", 1000);
        PartialRow top = basicSchema.newPartialRow();
        top.addInt("key", 2000);
        CreateTableOptions createOptions = new CreateTableOptions();
        createOptions.setRangePartitionColumns(Collections.singletonList("key"));
        createOptions.addRangePartition(bottom, middle);
        createOptions.addRangePartition(middle, top);
        KuduTable table = this.client.createTable("testOpenScanWithDroppedPartition", basicSchema, createOptions);
        int numRows = 1999;
        ClientTestUtil.loadDefaultTable((KuduClient)this.client, (String)"testOpenScanWithDroppedPartition", (int)numRows);
        KuduScanner scanner = ((KuduScanner.KuduScannerBuilder)this.client.newScannerBuilder(table).batchSizeBytes(100)).build();
        int rowsScanned = 0;
        int batchNum = 0;
        while (scanner.hasMoreRows()) {
            if (batchNum == 1) {
                CapturingLogAppender capture = new CapturingLogAppender();
                try (Closeable unused = capture.attach();){
                    this.client.alterTable("testOpenScanWithDroppedPartition", new AlterTableOptions().dropRangePartition(bottom, middle));
                    Thread.sleep(1000L);
                }
                KuduPartitioner partitioner = new KuduPartitioner.KuduPartitionerBuilder(table).build();
                Assert.assertEquals((String)"The partition was not dropped", (long)1L, (long)partitioner.numPartitions());
                Assert.assertTrue((boolean)capture.getAppendedText().contains("Deleting tablet data"));
                Assert.assertTrue((boolean)capture.getAppendedText().contains("successfully deleted"));
            }
            rowsScanned += scanner.nextRows().getNumRows();
            ++batchNum;
        }
        Assert.assertTrue((String)"All messages were consumed in the first batch", (batchNum > 1 ? 1 : 0) != 0);
        Assert.assertEquals((String)"Some message were not consumed", (long)numRows, (long)rowsScanned);
    }

    @Test(timeout=100000L)
    @KuduTestHarness.TabletServerConfig(flags={"--flush_threshold_secs=1"})
    public void testDiffScan() throws Exception {
        Schema schema = new Schema(Arrays.asList(new ColumnSchema.ColumnSchemaBuilder("key", Type.INT32).key(true).build(), new ColumnSchema.ColumnSchemaBuilder("is_deleted", Type.INT32).build()));
        KuduTable table = this.client.createTable(tableName, schema, ClientTestUtil.getBasicCreateTableOptions());
        int beforeBounds = 5;
        int numInserts = RandomUtils.nextIntInRange((Random)this.random, (int)1, (int)beforeBounds);
        int numUpdates = this.random.nextInt(beforeBounds);
        int numDeletes = this.random.nextInt(beforeBounds);
        List<Operation> beforeOps = this.generateMutationOperations(table, numInserts, numUpdates, numDeletes);
        Map<Integer, Operation.ChangeType> before = this.applyOperations(beforeOps);
        LOG.info("Before: {}", before);
        long startHT = this.client.getLastPropagatedTimestamp() + 1L;
        LOG.info("startHT: {}", (Object)startHT);
        int mutationBounds = 10;
        int expectedNumInserts = this.random.nextInt(mutationBounds);
        int expectedNumUpdates = this.random.nextInt(mutationBounds);
        int expectedNumDeletes = this.random.nextInt(mutationBounds);
        List<Operation> operations = this.generateMutationOperations(table, expectedNumInserts, expectedNumUpdates, expectedNumDeletes);
        Map<Integer, Operation.ChangeType> mutations = this.applyOperations(operations);
        LOG.info("Mutations: {}", mutations);
        long endHT = this.client.getLastPropagatedTimestamp() + 1L;
        LOG.info("endHT: {}", (Object)endHT);
        int afterBounds = 5;
        numInserts = this.random.nextInt(afterBounds);
        numUpdates = this.random.nextInt(afterBounds);
        numDeletes = this.random.nextInt(afterBounds);
        List<Operation> afterOps = this.generateMutationOperations(table, numInserts, numUpdates, numDeletes);
        Map<Integer, Operation.ChangeType> after = this.applyOperations(afterOps);
        LOG.info("After: {}", after);
        List tokens = ((KuduScanToken.KuduScanTokenBuilder)this.client.newScanTokenBuilder(table).diffScan(startHT, endHT)).build();
        ArrayList<RowResult> results = new ArrayList<RowResult>();
        for (KuduScanToken token : tokens) {
            KuduScanner scanner = KuduScanToken.deserializeIntoScanner((byte[])token.serialize(), (KuduClient)this.client);
            Schema projection = scanner.getProjectionSchema();
            int isDeletedIndex = projection.getIsDeletedIndex();
            Assert.assertEquals((long)(projection.getColumnCount() - 1), (long)isDeletedIndex);
            ColumnSchema isDeletedCol = projection.getColumnByIndex(isDeletedIndex);
            Assert.assertEquals((Object)Type.BOOL, (Object)isDeletedCol.getType());
            Assert.assertEquals((Object)Common.DataType.IS_DELETED, (Object)isDeletedCol.getWireType());
            Assert.assertEquals((Object)projection.getColumnByIndex(isDeletedIndex), (Object)projection.getColumn("is_deleted_"));
            for (RowResult row : scanner) {
                results.add(row);
            }
        }
        Assert.assertEquals((long)(mutations.size() - expectedNumDeletes), (long)results.size());
        int resultNumInserts = 0;
        int resultNumUpdates = 0;
        int resultExtra = 0;
        for (RowResult result : results) {
            Integer key = result.getInt(0);
            LOG.info("Processing key {}", (Object)key);
            Operation.ChangeType type = mutations.get(key);
            if (type == Operation.ChangeType.INSERT) {
                Assert.assertFalse((boolean)result.isDeleted());
                ++resultNumInserts;
                continue;
            }
            if (type == Operation.ChangeType.UPDATE) {
                Assert.assertFalse((boolean)result.isDeleted());
                ++resultNumUpdates;
                continue;
            }
            if (type == Operation.ChangeType.DELETE) {
                Assert.fail((String)"Shouldn't see any DELETEs");
                continue;
            }
            Assert.assertNull((Object)type);
            ++resultExtra;
        }
        Assert.assertEquals((long)expectedNumInserts, (long)resultNumInserts);
        Assert.assertEquals((long)expectedNumUpdates, (long)resultNumUpdates);
        Assert.assertEquals((long)0L, (long)resultExtra);
    }

    private Map<Integer, Operation.ChangeType> applyOperations(List<Operation> operations) throws Exception {
        HashMap<Integer, Operation.ChangeType> results = new HashMap<Integer, Operation.ChangeType>();
        if (operations.isEmpty()) {
            return results;
        }
        KuduSession session = this.client.newSession();
        if (this.random.nextBoolean()) {
            LOG.info("Waiting for a flush at the start of applyOperations");
            Thread.sleep(2L);
        }
        int flushInt = this.random.nextInt(operations.size());
        for (Operation op : operations) {
            OperationResponse resp;
            if (this.random.nextInt(operations.size()) == flushInt) {
                LOG.info("Waiting for a flush in the middle of applyOperations");
                Thread.sleep(2L);
            }
            if ((resp = session.apply(op)).hasRowError()) {
                LOG.error("Could not mutate row: " + resp.getRowError().getErrorStatus());
            }
            Assert.assertFalse((boolean)resp.hasRowError());
            results.put(op.getRow().getInt(0), op.getChangeType());
        }
        return results;
    }

    private List<Operation> generateMutationOperations(KuduTable table, int numInserts, int numUpdates, int numDeletes) throws Exception {
        ArrayList<Operation> results = new ArrayList<Operation>();
        ArrayList<MutationState> unfinished = new ArrayList<MutationState>();
        int minMutationsBound = 5;
        List<Pair> changeCounts = Arrays.asList(new Pair((Object)Operation.ChangeType.INSERT, (Object)numInserts), new Pair((Object)Operation.ChangeType.UPDATE, (Object)numUpdates), new Pair((Object)Operation.ChangeType.DELETE, (Object)numDeletes));
        for (Pair changeCount : changeCounts) {
            Operation.ChangeType type = (Operation.ChangeType)changeCount.getFirst();
            int count = (Integer)changeCount.getSecond();
            for (int i = 0; i < count; ++i) {
                Insert insert = table.newInsert();
                PartialRow row = insert.getRow();
                this.generator.randomizeRow(row);
                int key = row.getInt(0);
                results.add((Operation)insert);
                unfinished.add(new MutationState(key, type, this.random.nextInt(minMutationsBound)));
            }
        }
        while (!unfinished.isEmpty()) {
            int index = this.random.nextInt(unfinished.size());
            MutationState state = (MutationState)unfinished.get(index);
            if (state.numMutations >= state.minMutations && state.currentType == state.endType) {
                unfinished.remove(index);
                continue;
            }
            Object op = state.currentType == Operation.ChangeType.INSERT || state.currentType == Operation.ChangeType.UPDATE ? (this.random.nextBoolean() ? table.newUpdate() : table.newDelete()) : table.newInsert();
            PartialRow row = table.getSchema().newPartialRow();
            row.addInt(0, state.key);
            this.generator.randomizeRow(row, false);
            op.setRow(row);
            results.add((Operation)op);
            state.currentType = op.getChangeType();
            ++state.numMutations;
        }
        return results;
    }

    @Test(timeout=100000L)
    public void testDiffScanIsDeleted() throws Exception {
        Schema schema = new Schema(Arrays.asList(new ColumnSchema.ColumnSchemaBuilder("key", Type.INT32).key(true).build()));
        KuduTable table = this.client.createTable(tableName, schema, ClientTestUtil.getBasicCreateTableOptions());
        KuduSession session = this.client.newSession();
        Insert insert = table.newInsert();
        insert.getRow().addInt(0, 0);
        session.apply((Operation)insert);
        long startHT = this.client.getLastPropagatedTimestamp() + 1L;
        Delete delete = table.newDelete();
        delete.getRow().addInt(0, 0);
        session.apply((Operation)delete);
        long endHT = this.client.getLastPropagatedTimestamp() + 1L;
        KuduScanner scanner = ((KuduScanner.KuduScannerBuilder)this.client.newScannerBuilder(table).diffScan(startHT, endHT)).build();
        ArrayList<RowResult> results = new ArrayList<RowResult>();
        for (RowResult row : scanner) {
            results.add(row);
        }
        Assert.assertEquals((long)1L, (long)results.size());
        RowResult row = (RowResult)results.get(0);
        Assert.assertEquals((long)0L, (long)row.getInt(0));
        Assert.assertTrue((boolean)row.hasIsDeleted());
        Assert.assertTrue((boolean)row.isDeleted());
    }

    private static class MutationState {
        final int key;
        final Operation.ChangeType endType;
        final int minMutations;
        Operation.ChangeType currentType = Operation.ChangeType.INSERT;
        int numMutations = 0;

        MutationState(int key, Operation.ChangeType endType, int minMutations) {
            this.key = key;
            this.endType = endType;
            this.minMutations = minMutations;
        }
    }
}

