/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kudu.client;

import com.google.common.collect.ImmutableList;
import com.stumbleupon.async.Deferred;
import java.io.Closeable;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;
import java.sql.Date;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.Callable;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import org.apache.kudu.ColumnSchema;
import org.apache.kudu.Schema;
import org.apache.kudu.Type;
import org.apache.kudu.client.AlterTableOptions;
import org.apache.kudu.client.AsyncKuduClient;
import org.apache.kudu.client.AsyncKuduScanner;
import org.apache.kudu.client.Bytes;
import org.apache.kudu.client.CreateTableOptions;
import org.apache.kudu.client.Insert;
import org.apache.kudu.client.KuduClient;
import org.apache.kudu.client.KuduException;
import org.apache.kudu.client.KuduPredicate;
import org.apache.kudu.client.KuduScanToken;
import org.apache.kudu.client.KuduScanner;
import org.apache.kudu.client.KuduSession;
import org.apache.kudu.client.KuduTable;
import org.apache.kudu.client.NonRecoverableException;
import org.apache.kudu.client.Operation;
import org.apache.kudu.client.OperationResponse;
import org.apache.kudu.client.PartialRow;
import org.apache.kudu.client.RecoverableException;
import org.apache.kudu.client.ReplicaSelection;
import org.apache.kudu.client.RowResult;
import org.apache.kudu.client.RowResultIterator;
import org.apache.kudu.client.RpcProxy;
import org.apache.kudu.client.SessionConfiguration;
import org.apache.kudu.client.Status;
import org.apache.kudu.test.CapturingLogAppender;
import org.apache.kudu.test.ClientTestUtil;
import org.apache.kudu.test.KuduTestHarness;
import org.apache.kudu.test.RandomUtils;
import org.apache.kudu.util.DateUtil;
import org.apache.kudu.util.TimestampUtil;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TestKuduClient {
    private static final Logger LOG = LoggerFactory.getLogger(TestKuduClient.class);
    private static final String TABLE_NAME = "TestKuduClient";
    private static final int SHORT_SCANNER_TTL_MS = 5000;
    private static final int SHORT_SCANNER_GC_US = 500000;
    private static final Schema basicSchema = ClientTestUtil.getBasicSchema();
    private KuduClient client;
    private AsyncKuduClient asyncClient;
    @Rule
    public KuduTestHarness harness = new KuduTestHarness();

    @Before
    public void setUp() {
        this.client = this.harness.getClient();
        this.asyncClient = this.harness.getAsyncClient();
    }

    @Test(timeout=100000L)
    public void testLastPropagatedTimestamps() throws Exception {
        KuduTable table = this.client.createTable(TABLE_NAME, basicSchema, ClientTestUtil.getBasicCreateTableOptions());
        this.client.newScannerBuilder(table).build().nextRows().getNumRows();
        Assert.assertTrue((boolean)this.client.hasLastPropagatedTimestamp());
        Assert.assertTrue((boolean)this.client.hasLastPropagatedTimestamp());
        Assert.assertTrue((boolean)this.asyncClient.hasLastPropagatedTimestamp());
        long initialTs = this.client.getLastPropagatedTimestamp();
        Assert.assertEquals((long)initialTs, (long)this.client.getLastPropagatedTimestamp());
        Assert.assertEquals((long)initialTs, (long)this.asyncClient.getLastPropagatedTimestamp());
        this.client.updateLastPropagatedTimestamp(initialTs - 1L);
        Assert.assertEquals((long)initialTs, (long)this.client.getLastPropagatedTimestamp());
        Assert.assertEquals((long)initialTs, (long)this.asyncClient.getLastPropagatedTimestamp());
        this.client.updateLastPropagatedTimestamp(initialTs + 1L);
        Assert.assertEquals((long)(initialTs + 1L), (long)this.client.getLastPropagatedTimestamp());
        Assert.assertEquals((long)(initialTs + 1L), (long)this.asyncClient.getLastPropagatedTimestamp());
    }

    @Test(timeout=100000L)
    public void testCreateDeleteTable() throws Exception {
        this.client.createTable(TABLE_NAME, basicSchema, ClientTestUtil.getBasicCreateTableOptions());
        Assert.assertFalse((boolean)this.client.getTablesList().getTablesList().isEmpty());
        Assert.assertTrue((boolean)this.client.getTablesList().getTablesList().contains(TABLE_NAME));
        this.client.deleteTable(TABLE_NAME);
        Assert.assertFalse((boolean)this.client.getTablesList().getTablesList().contains(TABLE_NAME));
        ArrayList<ColumnSchema> columns = new ArrayList<ColumnSchema>(basicSchema.getColumns());
        columns.add(new ColumnSchema.ColumnSchemaBuilder("one more", Type.STRING).build());
        Schema newSchema = new Schema(columns);
        this.client.createTable(TABLE_NAME, newSchema, ClientTestUtil.getBasicCreateTableOptions());
        KuduTable table = this.client.openTable(TABLE_NAME);
        Assert.assertEquals((long)newSchema.getColumnCount(), (long)table.getSchema().getColumnCount());
        Assert.assertTrue((boolean)table.getPartitionSchema().isSimpleRangePartitioning());
        Assert.assertEquals((long)4096L, (long)newSchema.getColumn("column3_s").getDesiredBlockSize());
        Assert.assertEquals((Object)ColumnSchema.Encoding.DICT_ENCODING, (Object)newSchema.getColumn("column3_s").getEncoding());
        Assert.assertEquals((Object)ColumnSchema.CompressionAlgorithm.LZ4, (Object)newSchema.getColumn("column3_s").getCompressionAlgorithm());
    }

    @Test(timeout=100000L)
    public void testCreateTableTooManyColumns() throws Exception {
        ArrayList<ColumnSchema> cols = new ArrayList<ColumnSchema>();
        cols.add(new ColumnSchema.ColumnSchemaBuilder("key", Type.STRING).key(true).build());
        for (int i = 0; i < 1000; ++i) {
            cols.add(new ColumnSchema.ColumnSchemaBuilder("c" + i, Type.STRING).build());
        }
        Schema schema = new Schema(cols);
        try {
            this.client.createTable(TABLE_NAME, schema, ClientTestUtil.getBasicCreateTableOptions());
            Assert.fail();
        }
        catch (NonRecoverableException nre) {
            Assert.assertThat((Object)nre.toString(), (Matcher)CoreMatchers.containsString((String)"number of columns 1001 is greater than the permitted maximum"));
        }
    }

    @Test(timeout=100000L)
    public void testCreateDeleteTableWitExtraConfigs() throws Exception {
        Map extraConfigs = new HashMap<String, String>();
        extraConfigs.put("kudu.table.history_max_age_sec", "7200");
        this.client.createTable(TABLE_NAME, basicSchema, ClientTestUtil.getBasicCreateTableOptions().setExtraConfigs(extraConfigs));
        KuduTable table = this.client.openTable(TABLE_NAME);
        extraConfigs = table.getExtraConfig();
        Assert.assertTrue((boolean)extraConfigs.containsKey("kudu.table.history_max_age_sec"));
        Assert.assertEquals((Object)"7200", extraConfigs.get("kudu.table.history_max_age_sec"));
    }

    @Test(timeout=100000L)
    @KuduTestHarness.TabletServerConfig(flags={"--scanner_ttl_ms=5000", "--scanner_gc_check_interval_us=500000"})
    public void testScannerExpiration() throws Exception {
        int numRows = 1000;
        this.client.createTable(TABLE_NAME, basicSchema, new CreateTableOptions().addHashPartitions((List)ImmutableList.of((Object)"key"), 2));
        KuduSession session = this.client.newSession();
        KuduTable table = this.client.openTable(TABLE_NAME);
        for (int i = 0; i < numRows; ++i) {
            Insert insert = ClientTestUtil.createBasicSchemaInsert((KuduTable)table, (int)i);
            session.apply((Operation)insert);
        }
        KuduScanner scanner = ((KuduScanner.KuduScannerBuilder)((KuduScanner.KuduScannerBuilder)new KuduScanner.KuduScannerBuilder(this.asyncClient, table).replicaSelection(ReplicaSelection.CLOSEST_REPLICA)).batchSizeBytes(100)).build();
        int rows = scanner.nextRows().getNumRows();
        Assert.assertTrue((String)"Scanner did not read any rows", (rows > 0 ? 1 : 0) != 0);
        Thread.sleep(10000L);
        try {
            scanner.nextRows();
            Assert.fail((String)"Exception was not thrown when accessing an expired scanner");
        }
        catch (NonRecoverableException ex) {
            Assert.assertTrue((String)("Expected Scanner not found error, got:\n" + ex.toString()), (boolean)ex.getMessage().matches(".*Scanner .* not found.*"));
        }
        scanner.close();
    }

    @Test(timeout=100000L)
    @KuduTestHarness.TabletServerConfig(flags={"--scanner_ttl_ms=5000", "--scanner_gc_check_interval_us=500000"})
    public void testKeepAlive() throws Exception {
        int numRows = 1000;
        this.client.createTable(TABLE_NAME, basicSchema, new CreateTableOptions().addHashPartitions((List)ImmutableList.of((Object)"key"), 2));
        KuduSession session = this.client.newSession();
        KuduTable table = this.client.openTable(TABLE_NAME);
        for (int i = 0; i < numRows; ++i) {
            Insert insert = ClientTestUtil.createBasicSchemaInsert((KuduTable)table, (int)i);
            session.apply((Operation)insert);
        }
        KuduScanner scanner = ((KuduScanner.KuduScannerBuilder)((KuduScanner.KuduScannerBuilder)new KuduScanner.KuduScannerBuilder(this.asyncClient, table).replicaSelection(ReplicaSelection.CLOSEST_REPLICA)).batchSizeBytes(100)).build();
        scanner.keepAlive();
        int accum = scanner.nextRows().getNumRows();
        while (scanner.hasMoreRows()) {
            int rows = scanner.nextRows().getNumRows();
            accum += rows;
            if (scanner.currentTablet() == null) {
                LOG.info(String.format("Between tablets after scanning %d rows", accum));
                break;
            }
            if (accum != numRows) continue;
            Assert.fail((String)"All rows were in a single tablet.");
        }
        scanner.keepAlive();
        accum += scanner.nextRows().getNumRows();
        Random random = RandomUtils.getRandom();
        for (int i = 0; i < 10; ++i) {
            Thread.sleep(1250L);
            if (i % 3 == 0) {
                RpcProxy.failNextRpcs((int)random.nextInt(4), (Exception)new RecoverableException(Status.ServiceUnavailable((String)"testKeepAlive")));
            }
            scanner.keepAlive();
        }
        while (scanner.hasMoreRows()) {
            accum += scanner.nextRows().getNumRows();
        }
        Assert.assertEquals((String)"All rows were not scanned", (long)numRows, (long)accum);
        try {
            scanner.keepAlive();
            Assert.fail((String)"Exception was not thrown when calling keepAlive on a closed scanner");
        }
        catch (IllegalStateException ex) {
            Assert.assertThat((Object)ex.getMessage(), (Matcher)CoreMatchers.containsString((String)"Scanner has already been closed"));
        }
    }

    @Test(timeout=100000L)
    public void testTableWithDefaults() throws Exception {
        ArrayList<ColumnSchema> cols = new ArrayList<ColumnSchema>();
        cols.add(new ColumnSchema.ColumnSchemaBuilder("key", Type.STRING).key(true).build());
        cols.add(new ColumnSchema.ColumnSchemaBuilder("c1", Type.STRING).nullable(true).build());
        cols.add(new ColumnSchema.ColumnSchemaBuilder("c2", Type.STRING).nullable(true).defaultValue((Object)"def").build());
        cols.add(new ColumnSchema.ColumnSchemaBuilder("c3", Type.STRING).nullable(false).build());
        cols.add(new ColumnSchema.ColumnSchemaBuilder("c4", Type.STRING).nullable(false).defaultValue((Object)"def").build());
        Schema schema = new Schema(cols);
        this.client.createTable(TABLE_NAME, schema, ClientTestUtil.getBasicCreateTableOptions());
        KuduSession session = this.client.newSession();
        KuduTable table = this.client.openTable(TABLE_NAME);
        ImmutableList rows = ImmutableList.of((Object)"r1,a,b,c,d", (Object)"r2,NULL,NULL,c,d", (Object)"r3,-,-,c,-", (Object)"fail_1,a,b,c,NULL", (Object)"fail_2,a,b,NULL,d");
        for (String row : rows) {
            try {
                String[] fields = row.split(",", -1);
                Insert insert = table.newInsert();
                for (int i = 0; i < fields.length; ++i) {
                    if (fields[i].equals("-")) continue;
                    if (fields[i].equals("NULL")) {
                        insert.getRow().setNull(i);
                        continue;
                    }
                    insert.getRow().addString(i, fields[i]);
                }
                session.apply((Operation)insert);
            }
            catch (IllegalArgumentException e) {
                Assert.assertTrue((String)e.getMessage(), (boolean)e.getMessage().matches("c[34] cannot be set to null"));
            }
        }
        session.flush();
        ImmutableList expectedStrings = ImmutableList.of((Object)"STRING key=r1, STRING c1=a, STRING c2=b, STRING c3=c, STRING c4=d", (Object)"STRING key=r2, STRING c1=NULL, STRING c2=NULL, STRING c3=c, STRING c4=d", (Object)"STRING key=r3, STRING c1=NULL, STRING c2=def, STRING c3=c, STRING c4=def");
        List rowStrings = ClientTestUtil.scanTableToStrings((KuduTable)table, (KuduPredicate[])new KuduPredicate[0]);
        Collections.sort(rowStrings);
        Assert.assertArrayEquals((Object[])rowStrings.toArray(new String[0]), (Object[])expectedStrings.toArray(new String[0]));
    }

    @Test(timeout=100000L)
    public void testVarchars() throws Exception {
        Schema schema = ClientTestUtil.createManyVarcharsSchema();
        this.client.createTable(TABLE_NAME, schema, ClientTestUtil.getBasicCreateTableOptions());
        KuduSession session = this.client.newSession();
        KuduTable table = this.client.openTable(TABLE_NAME);
        for (int i = 0; i < 100; ++i) {
            Insert insert = table.newInsert();
            PartialRow row = insert.getRow();
            row.addVarchar("key", String.format("key_%02d", i));
            row.addVarchar("c2", "c2_" + i);
            if (i % 2 == 1) {
                row.addVarchar("c3", "c3_" + i);
            }
            row.addVarchar("c4", "c4_" + i);
            row.addVarchar("c1", "c1_" + i);
            session.apply((Operation)insert);
            if (i % 50 != 0) continue;
            session.flush();
        }
        session.flush();
        List rowStrings = ClientTestUtil.scanTableToStrings((KuduTable)table, (KuduPredicate[])new KuduPredicate[0]);
        Assert.assertEquals((long)100L, (long)rowStrings.size());
        Assert.assertEquals((Object)"VARCHAR key(10)=key_03, VARCHAR c1(10)=c1_3, VARCHAR c2(10)=c2_3, VARCHAR c3(10)=c3_3, VARCHAR c4(10)=c4_3", rowStrings.get(3));
        Assert.assertEquals((Object)"VARCHAR key(10)=key_04, VARCHAR c1(10)=c1_4, VARCHAR c2(10)=c2_4, VARCHAR c3(10)=NULL, VARCHAR c4(10)=c4_4", rowStrings.get(4));
        KuduScanner scanner = this.client.newScannerBuilder(table).build();
        Assert.assertTrue((String)"Scanner should have returned row", (boolean)scanner.hasMoreRows());
        RowResultIterator rows = scanner.nextRows();
        RowResult next = rows.next();
        try {
            next.getInt("c2");
            Assert.fail((String)"IllegalArgumentException was not thrown when accessing a VARCHAR column with getInt");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    @Test(timeout=100000L)
    public void testStrings() throws Exception {
        Schema schema = ClientTestUtil.createManyStringsSchema();
        this.client.createTable(TABLE_NAME, schema, ClientTestUtil.getBasicCreateTableOptions());
        KuduSession session = this.client.newSession();
        KuduTable table = this.client.openTable(TABLE_NAME);
        for (int i = 0; i < 100; ++i) {
            Insert insert = table.newInsert();
            PartialRow row = insert.getRow();
            row.addString("key", String.format("key_%02d", i));
            row.addString("c2", "c2_" + i);
            if (i % 2 == 1) {
                row.addString("c3", "c3_" + i);
            }
            row.addString("c4", "c4_" + i);
            row.addString("c1", "c1_" + i);
            session.apply((Operation)insert);
            if (i % 50 != 0) continue;
            session.flush();
        }
        session.flush();
        List rowStrings = ClientTestUtil.scanTableToStrings((KuduTable)table, (KuduPredicate[])new KuduPredicate[0]);
        Assert.assertEquals((long)100L, (long)rowStrings.size());
        Assert.assertEquals((Object)"STRING key=key_03, STRING c1=c1_3, STRING c2=c2_3, STRING c3=c3_3, STRING c4=c4_3", rowStrings.get(3));
        Assert.assertEquals((Object)"STRING key=key_04, STRING c1=c1_4, STRING c2=c2_4, STRING c3=NULL, STRING c4=c4_4", rowStrings.get(4));
        KuduScanner scanner = this.client.newScannerBuilder(table).build();
        Assert.assertTrue((String)"Scanner should have returned row", (boolean)scanner.hasMoreRows());
        RowResultIterator rows = scanner.nextRows();
        RowResult next = rows.next();
        try {
            next.getInt("c2");
            Assert.fail((String)"IllegalArgumentException was not thrown when accessing a string column with getInt");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    @Test(timeout=100000L)
    public void testUTF8() throws Exception {
        Schema schema = ClientTestUtil.createManyStringsSchema();
        this.client.createTable(TABLE_NAME, schema, ClientTestUtil.getBasicCreateTableOptions());
        KuduTable table = this.client.openTable(TABLE_NAME);
        Insert insert = table.newInsert();
        PartialRow row = insert.getRow();
        row.addString("key", "\u0e01\u0e02\u0e03\u0e04\u0e05\u0e06\u0e07");
        row.addString("c1", "\u2701\u2702\u2703\u2704\u2706");
        row.addString("c2", "hello");
        row.addString("c4", "\ud83d\udc31");
        KuduSession session = this.client.newSession();
        session.apply((Operation)insert);
        session.flush();
        List rowStrings = ClientTestUtil.scanTableToStrings((KuduTable)table, (KuduPredicate[])new KuduPredicate[0]);
        Assert.assertEquals((long)1L, (long)rowStrings.size());
        Assert.assertEquals((Object)"STRING key=\u0e01\u0e02\u0e03\u0e04\u0e05\u0e06\u0e07, STRING c1=\u2701\u2702\u2703\u2704\u2706, STRING c2=hello, STRING c3=NULL, STRING c4=\ud83d\udc31", rowStrings.get(0));
    }

    @Test(timeout=100000L)
    public void testBinaryColumns() throws Exception {
        Schema schema = ClientTestUtil.createSchemaWithBinaryColumns();
        this.client.createTable(TABLE_NAME, schema, ClientTestUtil.getBasicCreateTableOptions());
        byte[] testArray = new byte[]{1, 2, 3, 4, 5, 6, 7, 8, 9};
        KuduSession session = this.client.newSession();
        KuduTable table = this.client.openTable(TABLE_NAME);
        for (int i = 0; i < 100; ++i) {
            Insert insert = table.newInsert();
            PartialRow row = insert.getRow();
            row.addBinary("key", String.format("key_%02d", i).getBytes(StandardCharsets.UTF_8));
            row.addString("c1", "\u2701\u2702\u2703\u2704\u2706");
            row.addDouble("c2", (double)i);
            if (i % 2 == 1) {
                row.addBinary("c3", testArray);
            }
            session.apply((Operation)insert);
            if (i % 50 != 0) continue;
            session.flush();
        }
        session.flush();
        List rowStrings = ClientTestUtil.scanTableToStrings((KuduTable)table, (KuduPredicate[])new KuduPredicate[0]);
        Assert.assertEquals((long)100L, (long)rowStrings.size());
        for (int i = 0; i < rowStrings.size(); ++i) {
            StringBuilder expectedRow = new StringBuilder();
            expectedRow.append(String.format("BINARY key=\"key_%02d\", STRING c1=\u2701\u2702\u2703\u2704\u2706, DOUBLE c2=%.1f, BINARY c3=", i, (double)i));
            if (i % 2 == 1) {
                expectedRow.append(Bytes.pretty((byte[])testArray));
            } else {
                expectedRow.append("NULL");
            }
            Assert.assertEquals((Object)expectedRow.toString(), rowStrings.get(i));
        }
    }

    @Test(timeout=100000L)
    public void testTimestampColumns() throws Exception {
        Schema schema = ClientTestUtil.createSchemaWithTimestampColumns();
        this.client.createTable(TABLE_NAME, schema, ClientTestUtil.getBasicCreateTableOptions());
        ArrayList<Long> timestamps = new ArrayList<Long>();
        KuduSession session = this.client.newSession();
        KuduTable table = this.client.openTable(TABLE_NAME);
        long lastTimestamp = 0L;
        for (int i = 0; i < 100; ++i) {
            Insert insert = table.newInsert();
            PartialRow row = insert.getRow();
            long timestamp = System.currentTimeMillis() * 1000L;
            while (timestamp == lastTimestamp) {
                timestamp = System.currentTimeMillis() * 1000L;
            }
            timestamps.add(timestamp);
            row.addLong("key", timestamp);
            if (i % 2 == 1) {
                row.addLong("c1", timestamp);
            }
            session.apply((Operation)insert);
            if (i % 50 == 0) {
                session.flush();
            }
            lastTimestamp = timestamp;
        }
        session.flush();
        List rowStrings = ClientTestUtil.scanTableToStrings((KuduTable)table, (KuduPredicate[])new KuduPredicate[0]);
        Assert.assertEquals((long)100L, (long)rowStrings.size());
        for (int i = 0; i < rowStrings.size(); ++i) {
            StringBuilder expectedRow = new StringBuilder();
            expectedRow.append(String.format("UNIXTIME_MICROS key=%s, UNIXTIME_MICROS c1=", TimestampUtil.timestampToString((long)((Long)timestamps.get(i)))));
            if (i % 2 == 1) {
                expectedRow.append(TimestampUtil.timestampToString((long)((Long)timestamps.get(i))));
            } else {
                expectedRow.append("NULL");
            }
            Assert.assertEquals((Object)expectedRow.toString(), rowStrings.get(i));
        }
    }

    @Test(timeout=100000L)
    public void testDateColumns() throws Exception {
        Schema schema = ClientTestUtil.createSchemaWithDateColumns();
        this.client.createTable(TABLE_NAME, schema, ClientTestUtil.getBasicCreateTableOptions());
        ArrayList<Integer> dates = new ArrayList<Integer>();
        KuduSession session = this.client.newSession();
        KuduTable table = this.client.openTable(TABLE_NAME);
        for (int i = 0; i < 100; ++i) {
            Insert insert = table.newInsert();
            PartialRow row = insert.getRow();
            dates.add(i);
            Date date = DateUtil.epochDaysToSqlDate((int)i);
            row.addDate("key", date);
            if (i % 2 == 1) {
                row.addDate("c1", date);
            }
            session.apply((Operation)insert);
            if (i % 50 != 0) continue;
            session.flush();
        }
        session.flush();
        List rowStrings = ClientTestUtil.scanTableToStrings((KuduTable)table, (KuduPredicate[])new KuduPredicate[0]);
        Assert.assertEquals((long)100L, (long)rowStrings.size());
        for (int i = 0; i < rowStrings.size(); ++i) {
            String sdate = DateUtil.epochDaysToDateString((int)((Integer)dates.get(i)));
            StringBuilder expectedRow = new StringBuilder();
            expectedRow.append(String.format("DATE key=%s, DATE c1=", sdate));
            if (i % 2 == 1) {
                expectedRow.append(sdate);
            } else {
                expectedRow.append("NULL");
            }
            Assert.assertEquals((Object)expectedRow.toString(), rowStrings.get(i));
        }
    }

    @Test(timeout=100000L)
    public void testDecimalColumns() throws Exception {
        Schema schema = ClientTestUtil.createSchemaWithDecimalColumns();
        this.client.createTable(TABLE_NAME, schema, ClientTestUtil.getBasicCreateTableOptions());
        KuduSession session = this.client.newSession();
        KuduTable table = this.client.openTable(TABLE_NAME);
        Assert.assertEquals((long)38L, (long)table.getSchema().getColumn("c1").getTypeAttributes().getPrecision());
        for (int i = 0; i < 9; ++i) {
            Insert insert = table.newInsert();
            PartialRow row = insert.getRow();
            row.addDecimal("key", BigDecimal.valueOf(i));
            if (i % 2 == 1) {
                row.addDecimal("c1", BigDecimal.valueOf(i));
            }
            session.apply((Operation)insert);
        }
        session.flush();
        List rowStrings = ClientTestUtil.scanTableToStrings((KuduTable)table, (KuduPredicate[])new KuduPredicate[0]);
        Assert.assertEquals((long)9L, (long)rowStrings.size());
        for (int i = 0; i < rowStrings.size(); ++i) {
            StringBuilder expectedRow = new StringBuilder();
            expectedRow.append(String.format("DECIMAL key(18, 0)=%s, DECIMAL c1(38, 0)=", String.valueOf(i)));
            if (i % 2 == 1) {
                expectedRow.append(i);
            } else {
                expectedRow.append("NULL");
            }
            Assert.assertEquals((Object)expectedRow.toString(), rowStrings.get(i));
        }
    }

    @Test
    public void testScanWithLimit() throws Exception {
        KuduScanner scanner;
        int[] limits;
        int[] nonPositives;
        AsyncKuduClient asyncClient = this.harness.getAsyncClient();
        this.client.createTable(TABLE_NAME, basicSchema, ClientTestUtil.getBasicTableOptionsWithNonCoveredRange());
        KuduTable table = this.client.openTable(TABLE_NAME);
        KuduSession session = this.client.newSession();
        int numRows = 100;
        for (int key = 0; key < numRows; ++key) {
            session.apply((Operation)ClientTestUtil.createBasicSchemaInsert((KuduTable)table, (int)key));
        }
        for (int limit : nonPositives = new int[]{-1, 0}) {
            try {
                ((KuduScanner.KuduScannerBuilder)this.client.newScannerBuilder(table).limit((long)limit)).build();
                Assert.fail();
            }
            catch (IllegalArgumentException e) {
                Assert.assertTrue((boolean)e.getMessage().contains("Need a strictly positive number"));
            }
        }
        for (int limit : limits = new int[]{numRows - 1, numRows, numRows + 1}) {
            scanner = ((KuduScanner.KuduScannerBuilder)this.client.newScannerBuilder(table).limit((long)limit)).build();
            int count = 0;
            while (scanner.hasMoreRows()) {
                count += scanner.nextRows().getNumRows();
            }
            Assert.assertEquals((String)String.format("Limit %d returned %d/%d rows", limit, count, numRows), (long)Math.min(numRows, limit), (long)count);
        }
        for (int limit : limits) {
            scanner = ((AsyncKuduScanner.AsyncKuduScannerBuilder)new AsyncKuduScanner.AsyncKuduScannerBuilder(asyncClient, table).limit((long)limit)).build();
            Assert.assertEquals((long)Math.min(limit, numRows), (long)ClientTestUtil.countRowsInScan((AsyncKuduScanner)scanner));
        }
    }

    @Test
    public void testScanWithPredicates() throws Exception {
        Schema schema = ClientTestUtil.createManyStringsSchema();
        this.client.createTable(TABLE_NAME, schema, ClientTestUtil.getBasicCreateTableOptions());
        KuduSession session = this.client.newSession();
        session.setFlushMode(SessionConfiguration.FlushMode.AUTO_FLUSH_BACKGROUND);
        KuduTable table = this.client.openTable(TABLE_NAME);
        for (int i = 0; i < 100; ++i) {
            Insert insert = table.newInsert();
            PartialRow row = insert.getRow();
            row.addString("key", String.format("key_%02d", i));
            row.addString("c1", "c1_" + i);
            row.addString("c2", "c2_" + i);
            if (i % 2 == 0) {
                row.addString("c3", "c3_" + i);
            }
            session.apply((Operation)insert);
        }
        session.flush();
        Assert.assertEquals((long)100L, (long)ClientTestUtil.scanTableToStrings((KuduTable)table, (KuduPredicate[])new KuduPredicate[0]).size());
        Assert.assertEquals((long)50L, (long)ClientTestUtil.scanTableToStrings((KuduTable)table, (KuduPredicate[])new KuduPredicate[]{KuduPredicate.newComparisonPredicate((ColumnSchema)schema.getColumn("key"), (KuduPredicate.ComparisonOp)KuduPredicate.ComparisonOp.GREATER_EQUAL, (String)"key_50")}).size());
        Assert.assertEquals((long)25L, (long)ClientTestUtil.scanTableToStrings((KuduTable)table, (KuduPredicate[])new KuduPredicate[]{KuduPredicate.newComparisonPredicate((ColumnSchema)schema.getColumn("key"), (KuduPredicate.ComparisonOp)KuduPredicate.ComparisonOp.GREATER, (String)"key_74")}).size());
        Assert.assertEquals((long)25L, (long)ClientTestUtil.scanTableToStrings((KuduTable)table, (KuduPredicate[])new KuduPredicate[]{KuduPredicate.newComparisonPredicate((ColumnSchema)schema.getColumn("key"), (KuduPredicate.ComparisonOp)KuduPredicate.ComparisonOp.GREATER, (String)"key_24"), KuduPredicate.newComparisonPredicate((ColumnSchema)schema.getColumn("c1"), (KuduPredicate.ComparisonOp)KuduPredicate.ComparisonOp.LESS_EQUAL, (String)"c1_49")}).size());
        Assert.assertEquals((long)50L, (long)ClientTestUtil.scanTableToStrings((KuduTable)table, (KuduPredicate[])new KuduPredicate[]{KuduPredicate.newComparisonPredicate((ColumnSchema)schema.getColumn("key"), (KuduPredicate.ComparisonOp)KuduPredicate.ComparisonOp.GREATER, (String)"key_24"), KuduPredicate.newComparisonPredicate((ColumnSchema)schema.getColumn("key"), (KuduPredicate.ComparisonOp)KuduPredicate.ComparisonOp.GREATER_EQUAL, (String)"key_50")}).size());
        Assert.assertEquals((long)0L, (long)ClientTestUtil.scanTableToStrings((KuduTable)table, (KuduPredicate[])new KuduPredicate[]{KuduPredicate.newComparisonPredicate((ColumnSchema)schema.getColumn("c1"), (KuduPredicate.ComparisonOp)KuduPredicate.ComparisonOp.GREATER, (String)"c1_30"), KuduPredicate.newComparisonPredicate((ColumnSchema)schema.getColumn("c2"), (KuduPredicate.ComparisonOp)KuduPredicate.ComparisonOp.LESS, (String)"c2_20")}).size());
        Assert.assertEquals((long)0L, (long)ClientTestUtil.scanTableToStrings((KuduTable)table, (KuduPredicate[])new KuduPredicate[]{KuduPredicate.newComparisonPredicate((ColumnSchema)schema.getColumn("c2"), (KuduPredicate.ComparisonOp)KuduPredicate.ComparisonOp.GREATER, (String)"c2_30"), KuduPredicate.newComparisonPredicate((ColumnSchema)schema.getColumn("c2"), (KuduPredicate.ComparisonOp)KuduPredicate.ComparisonOp.LESS, (String)"c2_20")}).size());
        Assert.assertEquals((long)100L, (long)ClientTestUtil.scanTableToStrings((KuduTable)table, (KuduPredicate[])new KuduPredicate[]{KuduPredicate.newIsNotNullPredicate((ColumnSchema)schema.getColumn("c1")), KuduPredicate.newIsNotNullPredicate((ColumnSchema)schema.getColumn("key"))}).size());
        Assert.assertEquals((long)50L, (long)ClientTestUtil.scanTableToStrings((KuduTable)table, (KuduPredicate[])new KuduPredicate[]{KuduPredicate.newIsNotNullPredicate((ColumnSchema)schema.getColumn("c3"))}).size());
        Assert.assertEquals((long)0L, (long)ClientTestUtil.scanTableToStrings((KuduTable)table, (KuduPredicate[])new KuduPredicate[]{KuduPredicate.newIsNullPredicate((ColumnSchema)schema.getColumn("c2")), KuduPredicate.newIsNullPredicate((ColumnSchema)schema.getColumn("key"))}).size());
        Assert.assertEquals((long)50L, (long)ClientTestUtil.scanTableToStrings((KuduTable)table, (KuduPredicate[])new KuduPredicate[]{KuduPredicate.newIsNullPredicate((ColumnSchema)schema.getColumn("c3"))}).size());
        Assert.assertEquals((long)3L, (long)ClientTestUtil.scanTableToStrings((KuduTable)table, (KuduPredicate[])new KuduPredicate[]{KuduPredicate.newInListPredicate((ColumnSchema)schema.getColumn("key"), (List)ImmutableList.of((Object)"key_30", (Object)"key_01", (Object)"invalid", (Object)"key_99"))}).size());
        Assert.assertEquals((long)3L, (long)ClientTestUtil.scanTableToStrings((KuduTable)table, (KuduPredicate[])new KuduPredicate[]{KuduPredicate.newInListPredicate((ColumnSchema)schema.getColumn("c2"), (List)ImmutableList.of((Object)"c2_30", (Object)"c2_1", (Object)"invalid", (Object)"c2_99"))}).size());
        Assert.assertEquals((long)2L, (long)ClientTestUtil.scanTableToStrings((KuduTable)table, (KuduPredicate[])new KuduPredicate[]{KuduPredicate.newInListPredicate((ColumnSchema)schema.getColumn("c2"), (List)ImmutableList.of((Object)"c2_30", (Object)"c2_1", (Object)"invalid", (Object)"c2_99")), KuduPredicate.newIsNotNullPredicate((ColumnSchema)schema.getColumn("c2")), KuduPredicate.newInListPredicate((ColumnSchema)schema.getColumn("key"), (List)ImmutableList.of((Object)"key_30", (Object)"key_45", (Object)"invalid", (Object)"key_99"))}).size());
    }

    @Test
    public void testGetAuthnToken() throws Exception {
        byte[] token = (byte[])this.asyncClient.exportAuthenticationCredentials().join();
        Assert.assertNotNull((Object)token);
    }

    private int countRowsForTestScanNonCoveredTable(KuduTable table, Integer lowerBound, Integer upperBound) throws Exception {
        PartialRow bound;
        KuduScanner.KuduScannerBuilder scanBuilder = this.client.newScannerBuilder(table);
        if (lowerBound != null) {
            bound = basicSchema.newPartialRow();
            bound.addInt(0, lowerBound.intValue());
            scanBuilder.lowerBound(bound);
        }
        if (upperBound != null) {
            bound = basicSchema.newPartialRow();
            bound.addInt(0, upperBound.intValue());
            scanBuilder.exclusiveUpperBound(bound);
        }
        KuduScanner scanner = scanBuilder.build();
        int count = 0;
        while (scanner.hasMoreRows()) {
            count += scanner.nextRows().getNumRows();
        }
        return count;
    }

    @Test(timeout=100000L)
    public void testScanNonCoveredTable() throws Exception {
        int key;
        this.client.createTable(TABLE_NAME, basicSchema, ClientTestUtil.getBasicTableOptionsWithNonCoveredRange());
        KuduSession session = this.client.newSession();
        session.setFlushMode(SessionConfiguration.FlushMode.AUTO_FLUSH_BACKGROUND);
        KuduTable table = this.client.openTable(TABLE_NAME);
        for (key = 0; key < 100; ++key) {
            session.apply((Operation)ClientTestUtil.createBasicSchemaInsert((KuduTable)table, (int)key));
        }
        for (key = 200; key < 300; ++key) {
            session.apply((Operation)ClientTestUtil.createBasicSchemaInsert((KuduTable)table, (int)key));
        }
        session.flush();
        Assert.assertEquals((long)0L, (long)session.countPendingErrors());
        Assert.assertEquals((long)200L, (long)this.countRowsForTestScanNonCoveredTable(table, null, null));
        Assert.assertEquals((long)100L, (long)this.countRowsForTestScanNonCoveredTable(table, null, 200));
        Assert.assertEquals((long)0L, (long)this.countRowsForTestScanNonCoveredTable(table, null, -1));
        Assert.assertEquals((long)0L, (long)this.countRowsForTestScanNonCoveredTable(table, 120, 180));
        Assert.assertEquals((long)0L, (long)this.countRowsForTestScanNonCoveredTable(table, 300, null));
    }

    @Test(timeout=100000L)
    public void testAutoClose() throws Exception {
        try (KuduClient localClient = new KuduClient.KuduClientBuilder(this.harness.getMasterAddressesAsString()).build();){
            localClient.createTable(TABLE_NAME, basicSchema, ClientTestUtil.getBasicCreateTableOptions());
            KuduTable table = localClient.openTable(TABLE_NAME);
            KuduSession session = localClient.newSession();
            session.setFlushMode(SessionConfiguration.FlushMode.MANUAL_FLUSH);
            Insert insert = ClientTestUtil.createBasicSchemaInsert((KuduTable)table, (int)0);
            session.apply((Operation)insert);
        }
        KuduTable table = this.client.openTable(TABLE_NAME);
        AsyncKuduScanner scanner = new AsyncKuduScanner.AsyncKuduScannerBuilder(this.asyncClient, table).build();
        Assert.assertEquals((long)1L, (long)ClientTestUtil.countRowsInScan((AsyncKuduScanner)scanner));
    }

    @Test(timeout=100000L)
    public void testCloseShortlyAfterOpen() throws Exception {
        CapturingLogAppender cla = new CapturingLogAppender();
        try (Closeable c = cla.attach();){
            try (KuduClient localClient = new KuduClient.KuduClientBuilder(this.harness.getMasterAddressesAsString()).build();){
                localClient.exportAuthenticationCredentials();
            }
            Thread.sleep(500L);
        }
        Assert.assertFalse((String)cla.getAppendedText(), (boolean)cla.getAppendedText().contains("Exception"));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(timeout=100000L)
    public void testNoLogSpewOnConnectionRefused() throws Exception {
        CapturingLogAppender cla = new CapturingLogAppender();
        try (Closeable c = cla.attach();){
            this.harness.killAllMasterServers();
            try (KuduClient localClient = new KuduClient.KuduClientBuilder(this.harness.getMasterAddressesAsString()).build();){
                localClient.exportAuthenticationCredentials();
                Assert.fail((String)"Should have failed to connect.");
            }
            catch (NonRecoverableException e) {
                Assert.assertTrue((String)("Bad exception string: " + e.getMessage()), (boolean)e.getMessage().matches(".*Master config .+ has no leader. Exceptions received:.*Connection refused.*Connection refused.*Connection refused.*"));
            }
        }
        finally {
            this.harness.startAllMasterServers();
        }
        String logText = cla.getAppendedText();
        Assert.assertFalse((String)"Should not claim to have lost a connection in the log", (boolean)logText.contains("lost connection to peer"));
        Assert.assertFalse((String)"Should not have netty spew in log", (boolean)logText.contains("socket.nio.AbstractNioSelector"));
    }

    @Test(timeout=100000L)
    public void testCustomNioExecutor() throws Exception {
        long startTime = System.nanoTime();
        try (final KuduClient localClient = new KuduClient.KuduClientBuilder(this.harness.getMasterAddressesAsString()).nioExecutors((Executor)Executors.newFixedThreadPool(1), (Executor)Executors.newFixedThreadPool(2)).bossCount(1).workerCount(2).build();){
            int t;
            long buildTime = (System.nanoTime() - startTime) / 1000000000L;
            Assert.assertTrue((String)"Building KuduClient is slow, maybe netty get stuck", (buildTime < 3L ? 1 : 0) != 0);
            localClient.createTable(TABLE_NAME, basicSchema, ClientTestUtil.getBasicCreateTableOptions());
            Thread[] threads = new Thread[4];
            for (t = 0; t < 4; ++t) {
                final int id = t;
                threads[t] = new Thread(new Runnable(){

                    @Override
                    public void run() {
                        try {
                            KuduTable table = localClient.openTable(TestKuduClient.TABLE_NAME);
                            KuduSession session = localClient.newSession();
                            session.setFlushMode(SessionConfiguration.FlushMode.AUTO_FLUSH_SYNC);
                            for (int i = 0; i < 100; ++i) {
                                Insert insert = ClientTestUtil.createBasicSchemaInsert((KuduTable)table, (int)(id * 100 + i));
                                session.apply((Operation)insert);
                            }
                            session.close();
                        }
                        catch (Exception e) {
                            Assert.fail((String)("insert thread should not throw exception: " + e));
                        }
                    }
                });
                threads[t].start();
            }
            for (t = 0; t < 4; ++t) {
                threads[t].join();
            }
        }
    }

    @Test(expected=IllegalArgumentException.class)
    public void testNoDefaultPartitioning() throws Exception {
        this.client.createTable(TABLE_NAME, basicSchema, new CreateTableOptions());
    }

    @Test(timeout=100000L)
    public void testOpenTableClearsNonCoveringRangePartitions() throws KuduException {
        CreateTableOptions options = ClientTestUtil.getBasicCreateTableOptions();
        PartialRow lower = basicSchema.newPartialRow();
        PartialRow upper = basicSchema.newPartialRow();
        lower.addInt("key", 0);
        upper.addInt("key", 1);
        options.addRangePartition(lower, upper);
        this.client.createTable(TABLE_NAME, basicSchema, options);
        KuduTable table = this.client.openTable(TABLE_NAME);
        KuduScanToken.KuduScanTokenBuilder tokenBuilder = this.client.newScanTokenBuilder(table);
        List tokens = tokenBuilder.build();
        Assert.assertEquals((long)1L, (long)tokens.size());
        try (KuduClient alterClient = new KuduClient.KuduClientBuilder(this.harness.getMasterAddressesAsString()).defaultAdminOperationTimeoutMs(50000L).build();){
            lower = basicSchema.newPartialRow();
            upper = basicSchema.newPartialRow();
            lower.addInt("key", 1);
            AlterTableOptions alter = new AlterTableOptions();
            alter.addRangePartition(lower, upper);
            alterClient.alterTable(TABLE_NAME, alter);
        }
        tokenBuilder = this.client.newScanTokenBuilder(table);
        tokens = tokenBuilder.build();
        Assert.assertEquals((long)1L, (long)tokens.size());
        table = this.client.openTable(TABLE_NAME);
        tokenBuilder = this.client.newScanTokenBuilder(table);
        tokens = tokenBuilder.build();
        Assert.assertEquals((long)2L, (long)tokens.size());
    }

    @Test(timeout=100000L)
    public void testCreateTableWithConcurrentInsert() throws Exception {
        KuduTable table = this.client.createTable(TABLE_NAME, ClientTestUtil.createManyStringsSchema(), ClientTestUtil.getBasicCreateTableOptions().setWait(false));
        Insert insert = table.newInsert();
        insert.getRow().addString("key", "key_0");
        insert.getRow().addString("c1", "c1_0");
        insert.getRow().addString("c2", "c2_0");
        KuduSession session = this.client.newSession();
        OperationResponse resp = session.apply((Operation)insert);
        Assert.assertFalse((boolean)resp.hasRowError());
        Assert.assertTrue((boolean)this.client.isCreateTableDone(TABLE_NAME));
    }

    @Test(timeout=100000L)
    public void testCreateTableWithConcurrentAlter() throws Exception {
        Deferred d = this.asyncClient.createTable(TABLE_NAME, ClientTestUtil.createManyStringsSchema(), ClientTestUtil.getBasicCreateTableOptions());
        while (true) {
            try {
                this.client.alterTable(TABLE_NAME, new AlterTableOptions().renameTable("foo"));
            }
            catch (KuduException e) {
                if (e.getStatus().isNotFound()) continue;
                throw e;
            }
            break;
        }
        d.join();
    }

    @Test(timeout=100000L)
    public void testReadYourWritesSyncLeaderReplica() throws Exception {
        this.readYourWrites(SessionConfiguration.FlushMode.AUTO_FLUSH_SYNC, ReplicaSelection.LEADER_ONLY);
    }

    @Test(timeout=100000L)
    public void testReadYourWritesSyncClosestReplica() throws Exception {
        this.readYourWrites(SessionConfiguration.FlushMode.AUTO_FLUSH_SYNC, ReplicaSelection.CLOSEST_REPLICA);
    }

    @Test(timeout=100000L)
    public void testReadYourWritesBatchLeaderReplica() throws Exception {
        this.readYourWrites(SessionConfiguration.FlushMode.MANUAL_FLUSH, ReplicaSelection.LEADER_ONLY);
    }

    @Test(timeout=100000L)
    public void testReadYourWritesBatchClosestReplica() throws Exception {
        this.readYourWrites(SessionConfiguration.FlushMode.MANUAL_FLUSH, ReplicaSelection.CLOSEST_REPLICA);
    }

    private void readYourWrites(final SessionConfiguration.FlushMode flushMode, final ReplicaSelection replicaSelection) throws Exception {
        Schema schema = ClientTestUtil.createManyStringsSchema();
        this.client.createTable(TABLE_NAME, schema, ClientTestUtil.getBasicCreateTableOptions());
        int tasksNum = 4;
        ArrayList<2> callables = new ArrayList<2>();
        for (int t = 0; t < 4; ++t) {
            Callable<Void> callable = new Callable<Void>(){

                @Override
                public Void call() throws Exception {
                    AsyncKuduClient asyncKuduClient = new AsyncKuduClient.AsyncKuduClientBuilder(TestKuduClient.this.harness.getMasterAddressesAsString()).defaultAdminOperationTimeoutMs(50000L).build();
                    try (KuduClient kuduClient = asyncKuduClient.syncClient();){
                        KuduSession session = kuduClient.newSession();
                        session.setFlushMode(flushMode);
                        KuduTable table = kuduClient.openTable(TestKuduClient.TABLE_NAME);
                        for (int i = 0; i < 3; ++i) {
                            for (int j = 100 * i; j < 100 * (i + 1); ++j) {
                                Insert insert = table.newInsert();
                                PartialRow row = insert.getRow();
                                row.addString("key", String.format("key_%02d", j));
                                row.addString("c1", "c1_" + j);
                                row.addString("c2", "c2_" + j);
                                row.addString("c3", "c3_" + j);
                                session.apply((Operation)insert);
                            }
                            session.flush();
                            for (int k = 0; k < 3; ++k) {
                                AsyncKuduScanner scanner = ((AsyncKuduScanner.AsyncKuduScannerBuilder)((AsyncKuduScanner.AsyncKuduScannerBuilder)asyncKuduClient.newScannerBuilder(table).readMode(AsyncKuduScanner.ReadMode.READ_YOUR_WRITES)).replicaSelection(replicaSelection)).build();
                                KuduScanner syncScanner = new KuduScanner(scanner);
                                long preTs = asyncKuduClient.getLastPropagatedTimestamp();
                                Assert.assertNotEquals((long)-1L, (long)preTs);
                                long rowCount = ClientTestUtil.countRowsInScan((KuduScanner)syncScanner);
                                long expectedCount = 100L * (long)(i + 1);
                                Assert.assertTrue((expectedCount <= rowCount ? 1 : 0) != 0);
                                Assert.assertNotEquals((long)-1L, (long)scanner.getSnapshotTimestamp());
                                Assert.assertTrue((preTs < scanner.getSnapshotTimestamp() ? 1 : 0) != 0);
                                syncScanner.close();
                            }
                        }
                    }
                    return null;
                }
            };
            callables.add(callable);
        }
        ExecutorService executor = Executors.newFixedThreadPool(4);
        List futures = executor.invokeAll(callables);
        for (Future future : futures) {
            future.get();
        }
    }

    private void runTestCallDuringLeaderElection(String clientMethodName) throws Exception {
        Method methodToInvoke = KuduClient.class.getMethod(clientMethodName, new Class[0]);
        for (int i = 0; i < 5; ++i) {
            try (KuduClient cl = new KuduClient.KuduClientBuilder(this.harness.getMasterAddressesAsString()).build();){
                this.harness.restartLeaderMaster();
                methodToInvoke.invoke((Object)cl, new Object[0]);
                continue;
            }
        }
        this.harness.killAllMasterServers();
        try (KuduClient cl = new KuduClient.KuduClientBuilder(this.harness.getMasterAddressesAsString()).defaultAdminOperationTimeoutMs(5000L).build();){
            try {
                methodToInvoke.invoke((Object)cl, new Object[0]);
                Assert.fail();
            }
            catch (InvocationTargetException ex) {
                Assert.assertTrue((boolean)(ex.getTargetException() instanceof KuduException));
                KuduException realEx = (KuduException)ex.getTargetException();
                Assert.assertTrue((boolean)realEx.getStatus().isTimedOut());
            }
        }
    }

    @Test(timeout=100000L)
    public void testExportAuthenticationCredentialsDuringLeaderElection() throws Exception {
        this.runTestCallDuringLeaderElection("exportAuthenticationCredentials");
    }

    @Test(timeout=100000L)
    public void testGetHiveMetastoreConfigDuringLeaderElection() throws Exception {
        this.runTestCallDuringLeaderElection("getHiveMetastoreConfig");
    }

    @Test(timeout=100000L)
    public void testClientLocationNoLocation() throws Exception {
        this.client.listTabletServers();
        Assert.assertEquals((Object)"", (Object)this.client.getLocationString());
    }

    @Test(timeout=100000L)
    @KuduTestHarness.LocationConfig(locations={"/L0:6"})
    public void testClientLocation() throws Exception {
        this.client.listTabletServers();
        Assert.assertEquals((Object)"/L0", (Object)this.client.getLocationString());
    }

    @Test(timeout=100000L)
    public void testSessionOnceClosed() throws Exception {
        this.client.createTable(TABLE_NAME, basicSchema, ClientTestUtil.getBasicCreateTableOptions());
        KuduTable table = this.client.openTable(TABLE_NAME);
        KuduSession session = this.client.newSession();
        session.setFlushMode(SessionConfiguration.FlushMode.MANUAL_FLUSH);
        Insert insert = ClientTestUtil.createBasicSchemaInsert((KuduTable)table, (int)0);
        session.apply((Operation)insert);
        session.close();
        Assert.assertTrue((boolean)session.isClosed());
        insert = ClientTestUtil.createBasicSchemaInsert((KuduTable)table, (int)1);
        CapturingLogAppender cla = new CapturingLogAppender();
        try (Closeable c = cla.attach();){
            session.apply((Operation)insert);
        }
        String loggedText = cla.getAppendedText();
        Assert.assertTrue((String)("Missing warning:\n" + loggedText), (boolean)loggedText.contains("this is unsafe"));
    }

    @Test(timeout=100000L)
    public void testSchemaDriftPattern() throws Exception {
        KuduTable table = this.client.createTable(TABLE_NAME, ClientTestUtil.createManyStringsSchema(), ClientTestUtil.getBasicCreateTableOptions().setWait(false));
        KuduSession session = this.client.newSession();
        Insert insert = table.newInsert();
        PartialRow row = insert.getRow();
        row.addString("key", "key_0");
        row.addString("c1", "c1_0");
        row.addString("c2", "c2_0");
        row.addString("c3", "c3_0");
        row.addString("c4", "c4_0");
        OperationResponse resp = session.apply((Operation)insert);
        Assert.assertFalse((boolean)resp.hasRowError());
        boolean retried = false;
        while (true) {
            try {
                Insert insertExtra = table.newInsert();
                PartialRow rowExtra = insertExtra.getRow();
                rowExtra.addString("key", "key_1");
                rowExtra.addString("c1", "c1_1");
                rowExtra.addString("c2", "c2_1");
                rowExtra.addString("c3", "c2_1");
                rowExtra.addString("c4", "c2_1");
                rowExtra.addString("c5", "c5_1");
                OperationResponse respExtra = session.apply((Operation)insertExtra);
                Assert.assertFalse((boolean)respExtra.hasRowError());
            }
            catch (IllegalArgumentException e) {
                if (retried) {
                    throw e;
                }
                if (e.getMessage().contains("Unknown column")) {
                    this.client.alterTable(TABLE_NAME, new AlterTableOptions().addNullableColumn("c5", Type.STRING));
                    table = this.client.openTable(TABLE_NAME);
                    retried = true;
                    continue;
                }
                throw e;
            }
            break;
        }
        Assert.assertTrue((boolean)retried);
        Insert insertOld = table.newInsert();
        PartialRow rowOld = insertOld.getRow();
        rowOld.addString("key", "key_3");
        rowOld.addString("c1", "c1_3");
        rowOld.addString("c2", "c2_3");
        rowOld.addString("c3", "c3_3");
        rowOld.addString("c4", "c4_3");
        OperationResponse respOld = session.apply((Operation)insertOld);
        Assert.assertFalse((boolean)respOld.hasRowError());
    }
}

